
#include <mail/so/spamstop/tools/so-common/StorageBase.h>
#include <mail/so/spamstop/tools/general_shingler/data/keys.h>
#include <mail/so/spamstop/tools/general_shingler/data/helper.h>
#include <mail/so/libs/talkative_config/config.h>
#include "../storages/storage.h"
#include "base.h"
#include "db.h"
#include "proxy.h"
#include "curl_base.h"
#include "current_period.h"
#include "shingle.h"
#include "short_url.h"
#include "tags_info.h"
#include "dns.h"

namespace NGeneralShingler {

    void TUpdate::Apply(TUpdateData & update, const NJson::TJsonValue & messageFields) const {
        for(const auto & field : fields) {
            const auto & fieldName = field.first;
            switch (type) {
                case TUpdateType::Inc: {
                    field.second->Incr(messageFields, update.incrs, fieldName);
                    break;
                }
                case TUpdateType::Time: {
                    field.second->JsonToAnyvalue(timeSet.GetCurrentPeriod(), update.sets[fieldName]);
                    break;
                }
                case TUpdateType::FirstTime: {
                    field.second->JsonToAnyvalue(timeSet.GetCurrentPeriod(), update.setsOnInsert[fieldName]);
                    break;
                }
                case TUpdateType::Flag: {
                    field.second->Or(messageFields, update.ors, fieldName);
                    break;
                }
                case TUpdateType::ZeroInit:
                    field.second->JsonToAnyvalue(0, update.setsOnInsert[fieldName]);
                    break;
                case TUpdateType::Set:
                    field.second->JsonToAnyvalue(messageFields, update.sets, fieldName);
                    break;
                case TUpdateType::FieldsMax: {
                    if(!messageFields.Has(fieldName))
                        break;

                    const auto fields = messageFields[fieldName];

                    if(!fields["f1"].IsString() || !fields["f2"].IsString())
                        ythrow TWithBackTrace<yexception>() << "fields for FieldsMax must be map with f1, f2 fields, got: " << fields;

                    update.fieldsMax[fieldName] = {fields["f1"].GetString(), fields["f2"].GetString()};
                    break;
                };
            }
        }
    }

    TUpdate::TUpdate(const TString& typeName, const NConfig::TConfig &config, const TFieldSet & set, const THashMap<TString, TTimeSet>& timeSetsByName)
            : type(FromString<TUpdateType>(typeName)) {

        switch (type) {
            case TUpdateType::Inc:
            case TUpdateType::Flag:
            case TUpdateType::ZeroInit:
            case TUpdateType::Set:
            case TUpdateType::FieldsMax:
                fields = set.LinkFields(config);
                break;
            case TUpdateType::Time:
            case TUpdateType::FirstTime:{
                fields = set.LinkFields(NTalkativeConfig::Get(config, "fields"));
                const auto & setName = NTalkativeConfig::Get<TString>(config, "time_set");
                timeSet = NHelper::GetSafe(timeSetsByName, setName);
                break;
            }
        }
    }

    TAtomicSharedPtr<TSchemeBase> SchemeFactory(const NConfig::TConfig &config,
        const THashMap<TString, TTimeSet>& timeSetsByName,
        const THashMap<TString, TAtomicSharedPtr<TCacheContext>>& cacheByName,
        const TFieldSets& sets, const TStorage & storage
    ) {
        const auto type = FromString<TSchemeType>(NTalkativeConfig::Get<TString>(config, "type"));

        TAtomicSharedPtr <TCacheContext> cacheContext;
        if(config.Has("cache")) {
            const auto & cacheName = config["cache"].Get<TString>();

            cacheContext = NHelper::GetSafe(cacheByName, cacheName, cacheContext);
        }
        switch (type) {

            case TSchemeType::DB: {

                const TFieldSet &fields(sets.GetSetSafe(config));
                TVector<TAtomicSharedPtr<TKeyScheme>> update_keys, find_keys;
                TVector<TUpdate> updates;

                if(config.Has("keys")) {
                    for(const auto & v : NTalkativeConfig::Get<NConfig::TDict>(config, "keys")) {
                        update_keys.emplace_back(KeyFactory(v.first, v.second, fields, timeSetsByName));
                    }
                    find_keys = update_keys;
                }
                if(config.Has("update_keys")) {
                    for (const auto &v : NTalkativeConfig::Get<NConfig::TDict>(config, "update_keys")) {
                        update_keys.emplace_back(KeyFactory(v.first, v.second, fields, timeSetsByName));
                    }
                }
                if(config.Has("find_keys")) {
                    for (const auto &v : NTalkativeConfig::Get<NConfig::TDict>(config, "find_keys")) {
                        find_keys.emplace_back(KeyFactory(v.first, v.second, fields, timeSetsByName));
                    }
                }

                if(config.Has("updates")) {
                    for (const auto &v : NTalkativeConfig::Get<NConfig::TDict>(config, "updates")) {
                        updates.emplace_back(v.first, v.second, fields, timeSetsByName);
                    }
                }

                const auto &dbName = NTalkativeConfig::Get<TString>(config, "db");
                const auto &collectionName = NTalkativeConfig::Get<TString>(config, "collection");

                TTempDBContext dbContext({ storage.GetDB(dbName), collectionName });

                const auto upsert = config.Has("upsert") ? NTalkativeConfig::Get<bool>(config, "upsert") : false;

                TAggregationType aggregationType = TAggregationType::Concatenate;

                if(config.Has("aggregation_type")) {
                    const auto & rawType = NTalkativeConfig::Get<TString>(config, "aggregation_type");
                    aggregationType = FromString<TAggregationType>(rawType);
                }

                return MakeAtomicShared<TDbScheme>(std::move(update_keys), std::move(find_keys), updates, fields, cacheContext, dbContext, upsert, aggregationType);
            }
            case TSchemeType::PROXY: {

                const TFieldSet &fields(sets.GetSetSafe(config));
                const auto& host = NTalkativeConfig::Get<TString>(config, "host");
                const auto& name = NTalkativeConfig::Get<TString>(config, "source");
                const auto& source = storage.GetCurls(name);

                auto traits = MakeAtomicShared<NCurl::TPoolTraits>();

                if(auto it = MapFindPtr(config.Get<NConfig::TDict>(), "traits"))
                    traits = MakeAtomicShared<NCurl::TPoolTraits>(it->Get<NConfig::TDict>());

                return MakeAtomicShared<TProxyScheme>(fields, cacheContext, source, std::move(traits), host);
            }
            case TSchemeType::CURRENT_PERIOD: {
                THashMap<TString, TTimeSet> timesets;
                for(const auto & setNameConf : NTalkativeConfig::Get<NConfig::TArray>(config, "sets")) {

                    const auto & setName = NTalkativeConfig::Get<TString>(setNameConf);

                    const auto & timeSet = NHelper::GetSafe(timeSetsByName, setName);

                    timesets.emplace(setName, timeSet);

                }
                return MakeAtomicShared<TCurrentPeriodScheme>(timesets);
            }
            case TSchemeType::SHINGLE_BY_TEXT:  {
                return MakeAtomicShared<TShingleByTextScheme>();
            }
            case TSchemeType::SHORT_URL_RESOLVER: {
                TMaybe<TDuration> timeout;
                if (config.Has("timeout")) {
                    timeout = NTalkativeConfig::As<TDuration>(config, "timeout");
                }

                const auto& source = storage.GetCurls(NTalkativeConfig::Get<TString>(config, "source"));
                return MakeAtomicShared<TShortUrlResolverScheme>(sets.GetSetSafe(config), cacheContext, source, timeout);
            }
            case TSchemeType::TAGS_INFO: {
                const auto& tags = storage.GetTags(NTalkativeConfig::Get<TString>(config, "source"));
                return MakeAtomicShared<TTagsInfoScheme>(tags);
            }
            case TSchemeType::DNS: {
                const auto& dns = storage.GetDNS(NTalkativeConfig::Get<TString>(config, "source"));
                return MakeAtomicShared<TDNSScheme>(sets.GetSetSafe(config), cacheContext, dns);
            }
            case TSchemeType::CACHE: {
                const TFieldSet &fields(sets.GetSetSafe(config));
                return MakeAtomicShared<TCachedSchemeBase>(fields, cacheContext);
            }
        }
    }
}


