#include <numeric>
#include "cached.h"
#include <mail/so/spamstop/tools/general_shingler/data/cache.h>

namespace NGeneralShingler {
    class TCacheFuture : public IFuture {
    public:
        void DoRun(TCont * cont) override {
            if(internalFuture && master) {
                internalFuture->DoRun(cont);
            }
        }

        NJson::TJsonValue::TArray DoExtract() override {
            NJson::TJsonValue::TArray internalResults;
            if(internalFuture)
                internalResults = internalFuture->DoExtract();
            if(master)
                master->PutInCache(internalResults);
            for (auto &&res : internalResults)
                results.emplace_back(std::move(res));
            return results;
        }

        TCacheFuture() = default;

        explicit TCacheFuture(NJson::TJsonValue::TArray && results)
                : results(std::move(results)) {}

        explicit TCacheFuture(NJson::TJsonValue::TArray && results, TCachedSchemeBase * master, THolder<IFuture> && internalFuture)
                : results(std::move(results)), master(master), internalFuture(std::move(internalFuture)) {}

    private:
        NJson::TJsonValue::TArray results;
        TCachedSchemeBase * master{};
        THolder<IFuture> internalFuture;
    };

    THolder<IFuture> TCachedSchemeBase::Find(TInstant deadline, const TShardedFields& sharded) {
        NJson::TJsonValue::TArray jsResults;

        if(!hasCache) {
            return MakeHolder<TCacheFuture>(std::move(jsResults), this, InternalFind(deadline, sharded));
        }

        if(sharded.fields) {
            TShardedFields shardedMesagesForFound(sharded.shard);
            for (const auto &messageFields : sharded.fields) {
                const auto &map = messageFields.GetMapSafe();
                const bool isArrays = std::all_of(map.cbegin(), map.cend(),
                                                  [](const std::pair<TString, NJson::TJsonValue> &v) {
                                                      return v.second.IsArray();
                                                  });

                *logger << TLOG_DEBUG << messageFields << " is arrays = " << isArrays << Endl;

                if (!isArrays) {
                    const auto key = cacheContext->CreateKey(messageFields, fields);

                    NJson::TJsonValue::TArray cached;
                    if (FindInCache(key, cached)) {
                        for (auto &&item : cached)
                            jsResults.emplace_back(std::move(item));
                        *logger << TLOG_DEBUG << messageFields << " with key = " << key << " found in cache"
                                << Endl;
                    } else {
                        *logger << TLOG_DEBUG << messageFields << " with key = " << key << " not found in cache"
                                << Endl;
                        shardedMesagesForFound.fields.emplace_back(messageFields);
                    }
                } else {

                    const size_t maxSize = std::accumulate(map.cbegin(), map.cend(), size_t(0), [](size_t s,
                                                                                                   const std::pair<TString, NJson::TJsonValue> &v) {
                        return std::max(s, v.second.GetArray().size());
                    });

                    NJson::TJsonValue notFoundInCaches(NJson::JSON_MAP);
                    bool allFoundInCache = true;

                    for (const auto &v : map) {
                        notFoundInCaches[v.first].SetType(NJson::JSON_ARRAY);
                    }

                    for (size_t i = 0; i < maxSize; i++) {
                        NJson::TJsonValue partFields(NJson::JSON_MAP);;
                        for (const auto &v : map) {
                            const auto &ar = v.second.GetArray();
                            partFields[v.first] = ar[i];
                        }
                        const auto key = cacheContext->CreateKey(partFields, fields);

                        NJson::TJsonValue::TArray cached;
                        if (FindInCache(key, cached)) {
                            for (auto &&item : cached)
                                jsResults.emplace_back(std::move(item));
                            *logger << TLOG_DEBUG << partFields << " with key = " << key << " found in cache"
                                    << Endl;
                        } else {
                            *logger << TLOG_DEBUG << partFields << " with key = " << key << " not found in caches"
                                    << Endl;
                            for (const auto &v : partFields.GetMap()) {
                                notFoundInCaches[v.first].GetArraySafe().emplace_back(v.second);
                            }
                            allFoundInCache = false;
                        }
                    }

                    if (!allFoundInCache) {
                        shardedMesagesForFound.fields.emplace_back(std::move(notFoundInCaches));
                    }
                }
            }

            if(shardedMesagesForFound.fields.empty())
                return MakeHolder<TCacheFuture>(std::move(jsResults));
            else
                return MakeHolder<TCacheFuture>(std::move(jsResults), this, InternalFind(deadline, shardedMesagesForFound));
        } else {
            const size_t key = 0;

            NJson::TJsonValue::TArray cached;
            if (FindInCache(key, cached)) {
                for (auto &&item : cached)
                    jsResults.emplace_back(std::move(item));
                return MakeHolder<TCacheFuture>(std::move(jsResults));
            } else {
                return MakeHolder<TCacheFuture>(std::move(jsResults), this, InternalFind(deadline, {}));
            }
        }
    }

    void TCachedSchemeBase::Update(const TDeque<TShardedFields>& sharded) {
        for(const auto& shard : sharded) {
            PutInCache(shard.fields);
        }
    }

    THolder<IFuture> TCachedSchemeBase::InternalFind(TInstant /*deadline*/, const TShardedFields& /*sharded*/) {
        return MakeHolder<TCacheFuture>();
    }

    TCachedSchemeBase::TCachedSchemeBase(TFieldSet fields, TAtomicSharedPtr <TCacheContext> cacheContext) noexcept
            :   fields(std::move(fields)),
                cacheContext(std::move(cacheContext)),
                hasCache(bool(this->cacheContext))
    {}

    TCachedSchemeBase::~TCachedSchemeBase() = default;

    bool TCachedSchemeBase::FindInCache(size_t key, NJson::TJsonValue::TArray & jsResults) const {
        return cacheContext->cache->Get(key, jsResults);
    }

    void TCachedSchemeBase::PutInCache(NJson::TJsonValue::TArray values) {
        if(!cacheContext)
            return;

        THashMap<size_t, TDeque<NJson::TJsonValue>> groupedByKey;
        for(auto & val : values) {
            const auto key = cacheContext->CreateKey(val, fields);
            groupedByKey[key].emplace_back(std::move(val));
        }
        for(auto & group : groupedByKey) {
            cacheContext->cache->Add(group.first, std::move(group.second));
        }
    }
}
