#include <library/cpp/logger/log.h>
#include <library/cpp/config/config.h>

#include <mail/so/spamstop/tools/mongoclient_v3/StorageMongo.h>
#include <mail/so/spamstop/tools/postgreclient/PostgreBase.h>
#include <mail/so/spamstop/tools/lucene_storage/storage.h>
#include <util/generic/serialized_enum.h>
#include <mail/so/libs/talkative_config/config.h>

#include "db.h"

namespace NGeneralShingler {
    TDB::TDB(const NConfig::TConfig &config, const TAtomicSharedPtr<IThreadPool>& pool, const TAtomicSharedPtr<TLog>& logger) {
        if(!TryFromString<TDBStorageType>(NTalkativeConfig::Get<TString>(config, "type"), storageType)) {
            TStringStream msg;
            for(const auto & p : GetEnumNames<TDBStorageType>())
                msg << p.second << ',';
            ythrow TWithBackTrace<yexception>() << "valid storage types are: " << msg.Str() << "; got : " << config.Get<TString>();
        }

        InitDB(config["connection"], std::move(pool), logger);
    }

    void TDB::InitDB(const NConfig::TConfig &config, const TAtomicSharedPtr<IThreadPool>& pool, const TAtomicSharedPtr<TLog>& logger) {
        switch (storageType) {
            case TDBStorageType::Mongo:
                db = MakeAtomicShared<mongo_v3::TStorageMongo>();
                break;
            case TDBStorageType::Postgre:
                db = MakeHolder<sql::TPostgreBase>(
                        TPoolParams{32, TDuration::MilliSeconds(50)},
                        TPoolParams{128, TDuration::MilliSeconds(100)},
                        pool, logger
                );
                break;
            case TDBStorageType::Lucene:
                db = MakeHolder<NLucene::TStorage>();
                break;
        }
        db->Connect(config);
    }

    const TAtomicSharedPtr<TStorageBase>& TDB::GetStorage() const {
        return db;
    }
}
