#include <mail/so/spamstop/tools/so-common/StorageBase.h>
#include <mail/so/libs/talkative_config/config.h>
#include <mail/so/libs/curl/storage.h>

#include "db.h"
#include "dns.h"
#include "tags.h"
#include "storage.h"

namespace NGeneralShingler {
    template<typename T>
    static const T& GetStorage(const THashMap<TString, T>& storageByName, const TString & name, const char* type) {
        auto it = storageByName.find(name);
        if (storageByName.end() == it)
            ythrow TWithBackTrace<yexception>() << "cannot find " << type << ' ' << name;
        return it->second;
    }

    const TAtomicSharedPtr<NTagSets::TStorage<ui64, TString>>& TStorage::GetTags(const TString & name) const {
        return GetStorage(tagsByName, name, "tags");
    }

    const TAtomicSharedPtr<TStorageBase>& TStorage::GetDB(const TString & name) const {
        return GetStorage(dbsByName, name, "db");
    }

    const TAtomicSharedPtr<NCurl::TStorage>& TStorage::GetCurls(const TString & name) const {
        return GetStorage(curlsByName, name, "curl");;
    }

    const TAtomicSharedPtr<TDNS>& TStorage::GetDNS(const TString & name) const {
        return GetStorage(dnsByName, name, "dns");
    }

    template<typename T, typename F>
    static void CreateStorages(const NConfig::TConfig &config, const TString& type, T& storagesByName, const F& functor) {
        if (config.Has(type)) {
            const auto & typeConfig = config[type];
            if (!typeConfig.IsA<NConfig::TDict>())
                ythrow TWithBackTrace<yexception>() << type << " section must be dict: " << typeConfig;

            for (const auto & p : typeConfig.Get<NConfig::TDict>())
                storagesByName.emplace(p.first, functor(p.second));
        }
    }

    TStorage::TStorage(const NConfig::TConfig &config, const TAtomicSharedPtr<IThreadPool> &pool, const TAtomicSharedPtr<TLog>& logger, TSimpleScheduler& scheduler) {
        CreateStorages(config, "dbs", dbsByName, [&pool, &logger](const NConfig::TConfig &config) {
            return TDB(config, pool, logger).GetStorage();
        });
        CreateStorages(config, "curls", curlsByName, [](const NConfig::TConfig &config) {
            return MakeAtomicShared<NCurl::TStorage>(TPoolParams(config["easy"]));
        });
        CreateStorages(config, "dns", dnsByName, [](const NConfig::TConfig &config) {
            return MakeAtomicShared<TDNS>(config);
        });
        CreateStorages(config, "tags", tagsByName, [&scheduler](const NConfig::TConfig &config) {
            TTags tags(config);
            tags.SetSchedules(scheduler);
            return tags.GetStorage();
        });
    }

    TStorage::~TStorage() = default;
}
