#include <mapreduce/yt/interface/client.h>
#include <util/system/env.h>
#include <mail/so/libs/curl/curl.h>
#include <mail/so/libs/talkative_config/config.h>
#include <util/string/split.h>

#include "tags.h"

namespace NGeneralShingler {
    namespace NTagSource {
        class TRegexpSource : public NTagSets::NHelpers::THashRegexpReplaceSource<TString> {
            using TBase = NTagSets::NHelpers::THashRegexpReplaceSource<TString>;
        public:
            TRegexpSource(const TDestination& destination, const NConfig::TConfig &config)
                : TBase(destination, CreateRegexpInfo(config))
            { }

        private:
            static TBase::TReplaceInfo CreateRegexpInfo(const NConfig::TConfig &config) {
                TBase::TReplaceInfo info;
                if (config.Has("regexp_replace")) {
                    for (const auto& item : NTalkativeConfig::Get<NConfig::TDict>(config, "regexp_replace")) {
                        info.emplace_back(item.first, NTalkativeConfig::Get<TString>(item.second));
                    }
                }
                return info;
            }
        };

        class TRequestSource : public TRegexpSource {
        public:
            TRequestSource(const TDestination& destination, const NConfig::TConfig &config)
                : TRegexpSource(destination, config)
                , tag(NTalkativeConfig::Get<TString>(config, "tag"))
                , curl(NCurl::TOptions().SetParseHeaders(false))
            {
                NCurl::TRequestContext request;
                request.SetRequestTimeout(TDuration::Minutes(1));
                request.SetHost(NTalkativeConfig::Get<TString>(config, "host"));
                curl.Setup(std::move(request));
            }

            void Update() final {
                NCurl::TSimpleArtifacts artifacts;
                if(auto error = curl.Perform(artifacts)){
                    ythrow TWithBackTrace<yexception>() << "error: " << *error;
                }

                if (artifacts.code != 200)
                    ythrow TWithBackTrace<yexception>() << "unexpected code: " << artifacts.code;


                THashSet<TString> set = StringSplitter(artifacts.body.Str()).SplitBySet("\r\n ");

                SetData(tag, std::move(set));
            }

        private:
            TString tag;
            NCurl::TCurl curl;
        };

        struct YTInitializer {
            YTInitializer() { NYT::JoblessInitialize(); }
        };

        static TString GetToken(const NConfig::TConfig &config) {
            if (config.Has("token_env")) {
                const TString token = NTalkativeConfig::Get<TString>(config, "token_env");
                return GetEnv(token);
            }
            return NTalkativeConfig::Get<TString>(config, "token");
        }

        class TYTSource : public TRegexpSource {
        public:
            TYTSource(const TDestination& destination, const NConfig::TConfig &config)
                : TRegexpSource(destination, config)
                , tagPath(NTalkativeConfig::Get<TString>(config, "tag"))
                , valuePath(NTalkativeConfig::Get<TString>(config, "value"))
                , tablePath(NTalkativeConfig::Get<TString>(config, "table"))
            {
                static const YTInitializer init;
                client = NYT::CreateClient(NTalkativeConfig::Get<TString>(config, "url"), NYT::TCreateClientOptions().Token(GetToken(config)));
            }

            void Update() final {
                THashMap<TString, TRegexpSource::TSet> data;

                auto reader = client->CreateTableReader<NYT::TNode>(tablePath);
                for (; reader->IsValid(); reader->Next()) {
                    auto& row = reader->GetRow();
                    auto& tag = row[tagPath];
                    const TString& value = row[valuePath].AsString();

                    if (tag.IsString()) {
                        data[tag.AsString()].insert(value);
                    } else {
                        for (const auto& item : tag.AsList()) {
                            data[item.AsString()].insert(value);
                        }
                    }
                }

                for (const auto& item : data) {
                    SetData(item.first, item.second);
                }
            }

        private:
            NYT::IClientPtr client;
            TString tagPath;
            TString valuePath;
            TString tablePath;
        };
    }

    TTags::TTags(const NConfig::TConfig &config) {
        if (!config.Has("sources"))
            ythrow TWithBackTrace<yexception>() << "tags config must contain sources: " << config;

        const auto& sources = config["sources"];
        if (!sources.IsA<NConfig::TArray>())
            ythrow TWithBackTrace<yexception>() << "headers must be dict: " << sources;

        for (const auto& source : sources.Get<NConfig::TArray>()) {
            const auto time = source.Has("update_time")? NTalkativeConfig::As<TDuration>(source, "update_time"): TDuration::Zero();
            switch (FromString<TTagSourceType>(NTalkativeConfig::Get<TString>(source, "type"))) {
                case TTagSourceType::Request:
                    manager.CreateScheduleSource<NTagSource::TRequestSource>(time, source);
                    break;
                case TTagSourceType::YT:
                    manager.CreateScheduleSource<NTagSource::TYTSource>(time, source);
                    break;
                default:
                    Y_VERIFY(false);
            }
        }
    }

    const TTags::TStorage& TTags::GetStorage() const {
        return manager.Get();
    }

    void TTags::SetSchedules(TSimpleScheduler& scheduler) {
        manager.SetSchedules(scheduler);
    }
}
