#include <util/generic/bt_exception.h>
#include <util/generic/ymath.h>
#include "point.h"

namespace NSoKNN {

    NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TCoordinate& coordinate) {
        writer.BeginList();
        for (auto c : coordinate.Vec)
            writer.WriteInt(c);
        writer.EndList();

        return writer;
    }

    TCoordinate TCoordinate::FromJson(const NJson::TJsonValue& src) {
        if(!src.IsArray())
            ythrow TWithBackTrace<yexception>() << "TCoordinate expects array, got: " << src;
        const auto& array = src.GetArray();

        if(!array)
            ythrow yexception() << "empty coordinate";

        TCoordinate coordinate(TVector<int>(Reserve(array.size())));

        for (const auto& v : array) {
            coordinate.Vec.emplace_back(v.GetDoubleRobust());
        }

        return coordinate;
    }

    NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TJsonPoint& point) {
        writer.BeginObject();

        writer.WriteKey("coordinate");
        writer << point.Coordinate;
        writer.WriteKey("value");
        writer.WriteJsonValue(&point.Value);

        writer.EndObject();

        return writer;
    }

    TJsonPoint TJsonPoint::FromJson(const NJson::TJsonValue& src) {
        TCoordinate coordinate = TCoordinate::FromJson(src["coordinate"]);

        NJson::TJsonValue value;
        if (auto v = src.GetValueByPath("value"))
            value = *v;

        return TJsonPoint(std::move(coordinate), std::move(value));
    }

    NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TValueWithDistance& value) {
        writer.BeginObject();

        writer.WriteKey("distance").WriteDouble(value.Distance);
        writer.WriteKey("value").WriteJsonValue(&value.Value);

        if(value.Coordinate) {
            writer.WriteKey("coordinate");
            writer << *value.Coordinate;
        }

        writer.EndObject();

        return writer;
    }

    TValueWithDistance TValueWithDistance::FromJson(const NJson::TJsonValue& src) {
        double distance{};
        if (auto v = src.GetValueByPath("distance")) {
            distance = v->GetDoubleSafe();
        }

        NJson::TJsonValue value;
        if (auto v = src.GetValueByPath("value")) {
            value = *v;
        }

        TMaybe<TCoordinate> coordinate;
        if (auto v = src.GetValueByPath("coordinate")) {
            coordinate = TCoordinate::FromJson(*v);
        }

        return TValueWithDistance(std::move(value), distance, std::move(coordinate));
    }

    IOutputStream& operator<<(IOutputStream& stream, const TGetNeighborsRequest& request) {
        {
            NJsonWriter::TBuf writer(NJsonWriter::HEM_DONT_ESCAPE_HTML, &stream);
            writer << request;
        }
        return stream;
    }

    NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TGetNeighborsRequest& request) {

        writer.BeginObject();
        writer.WriteKey("coordinate");
        writer << request.Coordinate;
        {
            writer.WriteKey("embeds").BeginObject();
            {
                writer.WriteKey("body").BeginList();
                for (const auto &val : request.BodyEmbed) {
                    writer.WriteFloat(val);
                }
                writer.EndList();
            }
            writer.EndObject();
        }
        writer.EndObject();

        return writer;
    }

    IOutputStream& operator<<(IOutputStream& stream, const TGetNeighborsResponse& response) {
        {
            NJsonWriter::TBuf writer(NJsonWriter::HEM_DONT_ESCAPE_HTML, &stream);
            writer << response;
        }
        return stream;
    }

    NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TGetNeighborsResponse& response) {
        writer.BeginObject();
        for (const auto& [ns, neighbors] : response.Neighbors) {
            writer.WriteKey(ns);
            writer.BeginList();
            for (const auto& n : neighbors)
                writer << n;
            writer.EndList();
        }
        writer.EndObject();

        return writer;
    }

    TGetNeighborsResponse TGetNeighborsResponse::FromJson(const NJson::TJsonValue& src) {
        if(!src.IsMap())
            ythrow TWithBackTrace<yexception>() << "TGetNeighborsResponse expects map, got: " << src.GetType() << ':' << src;

        THashMap<TString, TVector<TValueWithDistance>> groupedNeighbors;

        for(const auto& [group, js]: src.GetMap()) {
            const auto& srcNeighbours = js.GetArraySafe();

            auto& neighbors = groupedNeighbors.emplace(group, Reserve(srcNeighbours.size())).first->second;

            for (const auto& v : srcNeighbours) {
                neighbors.emplace_back(TValueWithDistance::FromJson(v));
            }
        }

        return {std::move(groupedNeighbors)};
    }

    TJsonPoint& TAddPointRequest::GetPointMutable() {
        return Point;
    }

    const TDeque<TString> TAddPointRequest::GetNamespaces() const {
        return Namespaces;
    }

    IOutputStream& operator<<(IOutputStream& stream, const TAddPointRequest& request) {
        {
            NJsonWriter::TBuf writer(NJsonWriter::HEM_DONT_ESCAPE_HTML, &stream);
            writer << request;
        }
        return stream;
    }

    NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TAddPointRequest& request) {
        writer.BeginObject();

        writer.WriteKey("point");
        writer << request.Point;

        writer.WriteKey("ns");
        writer.BeginList();
        for(const auto& g: request.Namespaces)
            writer.WriteString(g);
        writer.EndList();

        writer.EndObject();

        return writer;
    }

    TAddPointRequest TAddPointRequest::FromJson(const NJson::TJsonValue& src) {
        TDeque<TString> groups;
        for(const auto& g : src["ns"].GetArraySafe()) {
            groups.emplace_back(g.GetStringSafe());
        }
        return TAddPointRequest(TJsonPoint::FromJson(src["point"]), std::move(groups));
    }

    TDeletePointRequest TDeletePointRequest::FromJson(const NJson::TJsonValue& src) {
        TDeque<TString> groups;
        for(const auto& g : src["ns"].GetArraySafe()) {
            groups.emplace_back(g.GetStringSafe());
        }

        double distance{};
        if(auto it = src.GetValueByPath("distance")) {
            distance = it->GetDoubleRobust();
        }

        return TDeletePointRequest(TJsonPoint::FromJson(src["point"]), std::move(groups), distance);
    }
} // namespace NSoKNN
