#pragma once

#include <library/cpp/json/json_writer.h>
#include <library/cpp/dot_product/dot_product.h>
#include <util/generic/maybe.h>
#include "coordiante.h"

namespace NSoKNN {
    struct TJsonPoint {
    public:
        TJsonPoint() = default;
        explicit TJsonPoint(TCoordinate coordinate) noexcept
            : Coordinate(std::move(coordinate)) {
        }
        explicit TJsonPoint(TCoordinate coordinate, NJson::TJsonValue value) noexcept
            : Coordinate(std::move(coordinate))
            , Value(std::move(value)) {
        }

        Y_SAVELOAD_DEFINE(Coordinate, Value)

        friend NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TJsonPoint& point);

        static TJsonPoint FromJson(const NJson::TJsonValue& src);

        TCoordinate Coordinate;
        NJson::TJsonValue Value;
    };

    struct TValueWithDistance {
        TValueWithDistance(NJson::TJsonValue value, double distance, TMaybe<TCoordinate> coordinate = Nothing()) noexcept
            : Value(std::move(value))
            , Distance(distance)
            , Coordinate(std::move(coordinate)){
        }

        friend NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TValueWithDistance& value);

        static TValueWithDistance FromJson(const NJson::TJsonValue& src);

        NJson::TJsonValue Value;
        double Distance{};
        TMaybe<TCoordinate> Coordinate;
    };

    class TGetNeighborsRequest {
    public:
        TGetNeighborsRequest() = default;
        explicit TGetNeighborsRequest(TCoordinate coordinate, TVector<float> bodyEmbed) noexcept
            : Coordinate(std::move(coordinate)), BodyEmbed(std::move(bodyEmbed)) {
        }

        friend IOutputStream& operator<<(IOutputStream& stream, const TGetNeighborsRequest& response);

        friend NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TGetNeighborsRequest& point);

    private:
        TCoordinate Coordinate;
        TVector<float> BodyEmbed;
    };

    class TGetNeighborsResponse {
    public:
        friend IOutputStream& operator<<(IOutputStream& stream, const TGetNeighborsResponse& request);

        friend NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TGetNeighborsResponse& request);

        static TGetNeighborsResponse FromJson(const NJson::TJsonValue& src);

        THashMap<TString, TVector<TValueWithDistance>> Neighbors;
    };

    class TAddPointRequest {
    public:
        explicit TAddPointRequest(TJsonPoint point, TDeque<TString> namespaces) noexcept
            : Point(std::move(point))
            , Namespaces(std::move(namespaces)){
        }

        TJsonPoint& GetPointMutable();
        const TDeque<TString> GetNamespaces() const;

        friend IOutputStream& operator<<(IOutputStream& stream, const TAddPointRequest& request);

        friend NJsonWriter::TBuf& operator<<(NJsonWriter::TBuf& writer, const TAddPointRequest& request);

        static TAddPointRequest FromJson(const NJson::TJsonValue& src);

    private:
        TJsonPoint Point;
        TDeque<TString> Namespaces;
    };

    class TDeletePointRequest {
    public:
        explicit TDeletePointRequest(TJsonPoint point, TDeque<TString> namespaces, double distance) noexcept
            : Point(std::move(point))
            , Namespaces(std::move(namespaces))
            , Distance(distance){
        }

        TJsonPoint& GetPointMutable() { return Point; }
        const TDeque<TString> GetNamespaces() const { return Namespaces; }
        double GetDistance() const { return Distance; }

        static TDeletePointRequest FromJson(const NJson::TJsonValue& src);

    private:
        TJsonPoint Point;
        TDeque<TString> Namespaces;
        double Distance{};
    };
} // namespace NSoKNN
