#include <mail/so/libs/lru_cache/double_barrel.h>

#include <kernel/lemmer/core/language.h>
#include <kernel/lemmer/core/lemmer.h>
#include <kernel/lemmer/core/options.h>
#include <kernel/lemmer/tools/order.h>

#include <util/charset/unidata.h>
#include <util/generic/algorithm.h>
#include <util/generic/string.h>
#include <util/stream/buffered.h>

TDoubleBarrelLRUCache<TUtf16String, TUtf16String> LemmerCache(65536);

static void NormalizeToken(const TWtringBuf& token, IOutputStream& out) {
    bool percent = false;
    bool allDigit = true;
    bool other = false;
    for (wchar16 c: token) {
        if (c == u'%') {
            percent = true;
            break;
        } else if (!IsDigit(c)) {
            allDigit = false;
            if (!IsAlpha(c)) {
                other = true;
                break;
            }
        }
    }
    if (percent || other || token.length() > MAXWORD_LEN) {
        out << token;
        return;
    }
    if (allDigit) {
        out << "%SmallNumber%";
        return;
    }
    TUtf16String fromCache;
    if (LemmerCache.Get(token, fromCache)) {
        out << fromCache;
    } else {
        TWLemmaArray lemmas;
        NLemmer::AnalyzeWord(
            TWideToken{token.data(), token.length()},
            lemmas,
            LI_ALL_LANGUAGES,
            nullptr,
            NLemmer::TAnalyzeWordOpt::DefaultLemmerTestOpt());
        if (lemmas.empty()) {
            LemmerCache.Put(token, token);
            out << token;
        } else {
            auto best = MinElement(lemmas.begin(), lemmas.end(), NLemmer::TLemmaOrder());
            TWtringBuf result{best->GetText(), best->GetTextLength()};
            LemmerCache.Put(token, result);
            out << result;
        }
    }
}

static void ProcessLine(const TUtf16String& line, IOutputStream& out) {
    size_t prev = 0;
    size_t pos = 0;
    size_t size = line.Size();
    bool first = true;
    while (pos < size) {
        if (IsSpace(line.at(pos))) {
            if (pos > prev) {
                if (first) {
                    first = false;
                } else {
                    out << ' ';
                }
                NormalizeToken(TWtringBuf{line.Data() + prev, pos - prev}, out);
            }
            ++pos;
            prev = pos;
        } else {
            ++pos;
        }
    }
    if (prev < size) {
        if (!first) {
            out << ' ';
        }
        NormalizeToken(TWtringBuf{line.Data() + prev, size - prev}, out);
    }
    out << Endl;
}

int main(int, char*[])
{
    TBufferedInput in(&Cin, 65536);
    TBufferedOutput out(&Cout, 65536);
    TUtf16String line;
    while (in.ReadLine(line)) {
        ProcessLine(line, out);
    }
    out.Flush();
    Cout.Flush();
    return 0;
}

