#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/config/config.h>

#include <mail/so/spamstop/tools/lsa/app/app.h>
#include <mail/so/spamstop/tools/so-clients/functional_clients/LSAShClient.h>

#include <util/stream/file.h>
#include <util/string/subst.h>

const NTesting::TPortHolder PORT = NTesting::GetFreePort();

static const TString RESOURCES_ROOT = ArcadiaSourceRoot() + "/mail/so/spamstop/tools/lsa/app/ut/resources/";

NConfig::TConfig ReadConfig() {
    TString rawConfig = []() {
        TFileInput input(
                ArcadiaSourceRoot() + "/mail/so/spamstop/tools/lsa/lsa_config/files/templates/lsa-prod.conf.template");
        return input.ReadAll();
    }();

    SubstGlobal(rawConfig, "$(HTTP_PORT)", TStringBuilder{} << PORT);
    SubstGlobal(rawConfig, "$(HOME_DIR)", GetWorkPath());

    NJson::TJsonValue jsonConfig = NJson::ReadJsonFastTree(rawConfig);

    {
        NJson::TJsonValue &mlConfig = jsonConfig["ml"];
        mlConfig["cb_model_for_checkform"] = RESOURCES_ROOT + "model_frodo.ctm";
        mlConfig.EraseValue("cb_model_for_tagged_themes");
        mlConfig["dssm_configs"] = NJson::TJsonValue(NJson::JSON_ARRAY);
        mlConfig["dssm_configs"].AppendValue(GetWorkPath() + "/dssm/dssm/model.json");

        mlConfig.EraseValue("hnsw");
        mlConfig["mlp"] = RESOURCES_ROOT + "mlp.bin";
        mlConfig.EraseValue("ng_tab_config");
        mlConfig["tab_models"] = NJson::TJsonValue(NJson::JSON_MAP);
        mlConfig["themes"] = RESOURCES_ROOT + "themes.bin";
        mlConfig.EraseValue("vw");
        mlConfig["vw_configs"] = NJson::TJsonValue(NJson::JSON_ARRAY);
        mlConfig["w2v"] = GetWorkPath() + "/w2v/w2v.bin";
    }

    rawConfig = NJson::WriteJson(jsonConfig);

    TStringStream stream(rawConfig);
    return NConfig::TConfig::FromJson(stream);
}

static NFuncClient::TLSA MakeLsaClient() {
    TClientConfig lsaConfig;
    lsaConfig.Host = "localhost";
    lsaConfig.Port = PORT;
    return NFuncClient::TLSA(TRequestClient::TArgs(std::move(lsaConfig)));
}

TEST(LSA, Shingler) {
    const NConfig::TConfig config = ReadConfig();
    auto server = RunLsaApp(config);

    const NFuncClient::TLSA lsaClient = MakeLsaClient();

    NLSA::TRequestData requestData;
    requestData.AddModelId("1150288283");
    requestData.Add("word1 word2 word3", NLSA::TField::Body);

    NLSA::TResponseData response;
    EXPECT_NO_THROW(response = lsaClient.Perform(requestData, false));

    EXPECT_NEAR(response.GetPredictionsByModel().at("1150288283").at("DSSM_SEGMENT_TEXT"), 0.03284087777, 1e-3);

    server->Stop();
    server->Wait();
}