#pragma once

#include <util/ysaveload.h>
#include <util/generic/vector.h>
#include <contrib/libs/eigen/Eigen/Core>

namespace NLSA{
    using TMatrix = Eigen::MatrixXf;
    using TViewMatrix = Eigen::Map<Eigen::MatrixXf>;
    struct TViewMatrixData {

        TViewMatrixData(TViewMatrixData&&) noexcept = default;
        TViewMatrixData(const TViewMatrixData& viewMatrixData)
            : data(viewMatrixData.data), view(MakeHolder<TViewMatrix>(data.data(), viewMatrixData.view->rows(), viewMatrixData.view->cols())) {}
        TViewMatrixData() = default;
        explicit TViewMatrixData(TVector<float> d) : data(std::move(d)), view(MakeHolder<TViewMatrix>(data.data(), 1, data.size())) {}
        TViewMatrixData(size_t cols, size_t rows) : data(rows * cols), view(MakeHolder<TViewMatrix>(data.data(), rows, cols)) {}

        TVector<float> data;
        THolder<TViewMatrix> view;
    };
}

template<>
class TSerializer<NLSA::TMatrix> {
public:
static inline void Save(IOutputStream* rh, const NLSA::TMatrix & v) {
    ::SaveSize(rh, static_cast<size_t>(v.cols()));
    ::SaveSize(rh, static_cast<size_t>(v.rows()));

    for(int y = 0; y < v.rows(); y++) for(int x = 0; x < v.cols(); x++)
        ::Save(rh, v(y, x));
}
static inline void Load(IInputStream* in, NLSA::TMatrix& t) {
    const size_t cols = ::LoadSize(in);
    const size_t rows = ::LoadSize(in);
    t.resize(rows, cols);

    for(int y = 0; y < t.rows(); y++) for(int x = 0; x < t.cols(); x++)
        ::Load(in, t(y, x));
}
};

template<>
class TSerializer<NLSA::TViewMatrixData> {
public:
    static inline void Save(IOutputStream* rh, const NLSA::TViewMatrixData & v) {
        Y_VERIFY(v.view);
        Y_VERIFY(v.data.size() == size_t(v.view->cols() * v.view->rows()));
        ::SaveSize(rh, static_cast<size_t>(v.view->cols()));
        ::SaveSize(rh, static_cast<size_t>(v.view->rows()));

        ::SaveArray(rh, v.data.data(), v.data.size());
    }
    static inline void Load(IInputStream* in, NLSA::TViewMatrixData& t) {
        const size_t cols = ::LoadSize(in);
        const size_t rows = ::LoadSize(in);
        t.data.resize(rows * cols);
        ::LoadArray(in, t.data.data(), t.data.size());


        t.view = MakeHolder<NLSA::TViewMatrix>(t.data.data(), rows, cols);
    }
};

inline IOutputStream & operator<<(IOutputStream & s, const NLSA::TMatrix & m)
{
    for(size_t y = 0; y < size_t(m.rows()); y++)
    {
        for(size_t x = 0; x < size_t(m.cols()); x++)
        {
            s << m(y, x) << ' ';
            if(x != size_t(m.cols() - 1))
                s << ' ';
        }
        if(y != size_t(m.rows() - 1))
            s << '\n';
    }
    return s;
};
