#pragma once

#include <array>

#include <library/cpp/json/json_value.h>
#include <util/generic/maybe.h>
#include <util/generic/hash_set.h>
#include <util/string/cast.h>
#include <mail/so/libs/lsa_projectors/fields.h>
#include "tabs.h"

namespace NJson{
    class TJsonWriter;
}

namespace NLSA {

    class TControlSum{
    public:
        TControlSum() = default;
        TControlSum(ui64 sum) noexcept : Sum(sum) {}

        bool operator==(const TControlSum& cs) const;
        bool operator!=(const TControlSum& cs) const;
        bool Empty() const;
        ui64 GetSum() const;
        void Update(const TString& value);
    private:
        ui64 Sum{};
    };

    class TRequestData{
    public:
        bool operator==(const TRequestData& data) const;

        size_t Size(std::initializer_list<TField> fields) const;

        void AddModelId(TString id) { modelsIds.emplace(std::move(id)); }

        const THashSet<TString> & GetModelsIds() const { return modelsIds; }

        void Add(TString value, TField field);

        const TVector<TString>& Get(TField field) const;

        TVector<TString> GetPrefixedWords(TField field) const;
        TVector<TString> GetPrefixedWords(std::initializer_list<TField> fields) const;

        friend IOutputStream& operator<<(IOutputStream& stream, const TRequestData& data);
        void FromJson(const TStringBuf & src);

    private:
        void VerifyControlSums() const;

    private:
        std::array<TVector<TString>, GetEnumItemsCount<TField>()> fields;
        THashSet<TString> modelsIds;
        std::array<TControlSum, GetEnumItemsCount<TField>()> controlSums;
    };

    using TPredictionsMap = THashMap<TString, double>;

    struct TResolutionContext {
        NLSA::TPredictionsMap predictionsMap;
        TVector<TString> rules;
    };

    struct TResponseData{
    public:
        struct TTheme{
            float GetDistance() const { return Distance; }
            const TString & GetDescription() const { return Description; }

            TTheme() = default;
            TTheme(TString Description, float Distance) : Description(std::move(Description)), Distance(Distance) {}

            TString Description;
            float Distance = 0;
        };

        using TThemeChain = TVector<TTheme>;

        void AddPrediction(const TString& model, TPredictionsMap predictions) { modelsPredictions[model] = std::move(predictions); }
        void SetRules(const TString& model, TVector<TString> rules) { modelsRules[model] = std::move(rules); }
        void AddRules(TVector<TString> rule) { indepRules.insert(indepRules.end(), std::make_move_iterator(rule.begin()), std::make_move_iterator(rule.end())); }
        void SetMatchsPercent(float m) { MatchsPercent = m; }
        void SetThemes(TThemeChain themes) { Themes = std::move(themes); }
        void SetTestingThemes(TString name, TThemeChain && themes) { Testing.emplace(std::move(name), std::move(themes)); }
        void SetDistancesToCompls(TVector<float> distances) { minDistancesToCompls = std::move(distances); }

        float GetMatchsPercent() const { return MatchsPercent; }
        const auto & GetPredictionsByModel() const { return modelsPredictions; }
        const auto & GetRulesByModel() const { return modelsRules; }
        const auto & GetIndepRules() const { return indepRules; }
        const TThemeChain GetThemes() const { return Themes; }
        const THashMap<TString, TThemeChain>& GetTesting() const { return Testing; }
        const TVector<float>& GetDistancesToCompls() const { return minDistancesToCompls; }
        const NJson::TJsonValue& GetOriginalJson() const { return json; }

        TString ToJsonString() const;
        static TResponseData FromJson(const TStringBuf & src);

    private:
        THashMap<TString, TThemeChain> Testing;
        TThemeChain Themes;
        float MatchsPercent = 0;
        THashMap<TString, TPredictionsMap> modelsPredictions;
        THashMap<TString, TVector<TString>> modelsRules;
        TVector<TString> indepRules;
        TVector<float> minDistancesToCompls;

        NJson::TJsonValue json;
    };
}

