#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/buffer.h>
#include <util/stream/buffer.h>
#include <mail/so/spamstop/tools/lsa/data/transfer.h>

using namespace NLSA;

Y_UNIT_TEST_SUITE(Transfer) {

    void CheckRequestData(const TRequestData &requestData, const TStringBuf expected) {
        TStringStream s;
        s << requestData;
        UNIT_ASSERT_VALUES_EQUAL_C(s.Str(), expected, "expected: <" << expected << ">, but got: <" << s.Str() << '>');
    }

    Y_UNIT_TEST(Main) {
        TRequestData requestData;
        CheckRequestData(requestData, R"({"cs":{},"models_ids":[]})");

        requestData.AddModelId("model_id");
        CheckRequestData(requestData, R"({"cs":{},"models_ids":["model_id"]})");

        requestData.AddModelId("another_id");
        CheckRequestData(requestData, R"({"cs":{},"models_ids":["another_id","model_id"]})");

        requestData.AddModelId("another_id");
        CheckRequestData(requestData, R"({"cs":{},"models_ids":["another_id","model_id"]})");

        requestData.Add("body", TField::Body);
        {
            const auto expected = TVector<TString>{"body"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::Body), expected);
        }
        {
            const auto expected = TVector<TString>{"body_body"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::Body), expected);
        }

        requestData.Add("body2", TField::Body);
        requestData.Add("some_subject", TField::Subject);
        requestData.Add("some_fromname", TField::Fromname);
        requestData.Add("some_fromaddr", TField::Fromaddr);
        requestData.Add("Стажировки бывают двух видов: летние и «всесезонные».", TField::FullBody);
        requestData.Add("Мы открыли набор на летнюю стажировку. Впереди ещё месяц зимы", TField::PureBody);
        requestData.Add("Большинство стажёров — как раз студенты дневных отделений технических вузов", TField::PureSubject);
        requestData.Add("Летние, как понятно из названия, проходят летом", TField::PureDeobfuscatedBody);
        requestData.Add("Это не означает, что выходить надо строго первого июня и ни днём позже", TField::PureDeobfuscatedSubject);
        requestData.Add("luckybug@yandex-team.ru", TField::FullFromAddr);
        requestData.Add("luckybug", TField::FullFromAddrLogin);

        {
            const auto expected = TVector<TString>{"body", "body2"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::Body), expected);
        }

        {
            const auto expected = TVector<TString>{"body_body", "body_body2"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::Body), expected);
        }
        {
            const auto expected = TVector<TString>{"some_subject"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::Subject), expected);
        }

        {
            const auto expected = TVector<TString>{"subject_some_subject"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::Subject), expected);
        }
        {
            const auto expected = TVector<TString>{"some_fromname"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::Fromname), expected);
        }

        {
            const auto expected = TVector<TString>{"fromname_some_fromname"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::Fromname), expected);
        }
        {
            const auto expected = TVector<TString>{"some_fromaddr"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::Fromaddr), expected);
        }

        {
            const auto expected = TVector<TString>{"fromaddr_some_fromaddr"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::Fromaddr), expected);
        }
        {
            const auto expected = TVector<TString>{"Мы открыли набор на летнюю стажировку. Впереди ещё месяц зимы"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::PureBody), expected);
        }

        {
            const auto expected = TVector<TString>{
                    "pure_body_Мы открыли набор на летнюю стажировку. Впереди ещё месяц зимы"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::PureBody), expected);
        }
        {
            const auto expected = TVector<TString>{
                    "Большинство стажёров — как раз студенты дневных отделений технических вузов"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::PureSubject), expected);
        }
        {
            const auto expected = TVector<TString>{
                    "pure_subject_Большинство стажёров — как раз студенты дневных отделений технических вузов"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::PureSubject), expected);
        }
        {
            const auto expected = TVector<TString>{"Стажировки бывают двух видов: летние и «всесезонные»."};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::FullBody), expected);
        }
        {
            const auto expected = TVector<TString>{"full_body_Стажировки бывают двух видов: летние и «всесезонные»."};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::FullBody), expected);
        }
        {
            const auto expected = TVector<TString>{"Летние, как понятно из названия, проходят летом"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::PureDeobfuscatedBody), expected);
        }
        {
            const auto expected = TVector<TString>{"pure_deobfuscated_body_Летние, как понятно из названия, проходят летом"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::PureDeobfuscatedBody), expected);
        }
        {
            const auto expected = TVector<TString>{"Это не означает, что выходить надо строго первого июня и ни днём позже"};
            UNIT_ASSERT_EQUAL(requestData.Get(TField::PureDeobfuscatedSubject), expected);

        }
        {
            const auto expected = TVector<TString>{"pure_deobfuscated_subject_Это не означает, что выходить надо строго первого июня и ни днём позже"};
            UNIT_ASSERT_EQUAL(requestData.GetPrefixedWords(TField::PureDeobfuscatedSubject), expected);
        }
        {
            const auto expected = TVector<TString>{"full_fromaddr_luckybug@yandex-team.ru"};
            UNIT_ASSERT_VALUES_EQUAL(requestData.GetPrefixedWords(TField::FullFromAddr), expected);
        }
        {
            const auto expected = TVector<TString>{"full_fromlogin_luckybug"};
            UNIT_ASSERT_VALUES_EQUAL(requestData.GetPrefixedWords(TField::FullFromAddrLogin), expected);
        }

        const TString expected = R"({"body":["body","body2"],"subject":["some_subject"],"fromname":["some_fromname"],"fromaddr":["some_fromaddr"],"pure_body":["Мы открыли набор на летнюю стажировку. Впереди ещё месяц зимы"],"pure_subject":["Большинство стажёров — как раз студенты дневных отделений технических вузов"],"full_body":["Стажировки бывают двух видов: летние и «всесезонные»."],"pure_deobfuscated_body":["Летние, как понятно из названия, проходят летом"],"pure_deobfuscated_subject":["Это не означает, что выходить надо строго первого июня и ни днём позже"],"full_fromaddr":["luckybug@yandex-team.ru"],"full_fromlogin":["luckybug"],"cs":{"body":2539039802788591037,"subject":2830576319973502388,"fromname":4971347849741141261,"fromaddr":11954684848256533461,"pure_body":5826031546037622442,"pure_subject":11612273152437226553,"full_body":17065982970693298402,"pure_deobfuscated_body":1851479184421752240,"pure_deobfuscated_subject":7243948861487075007,"full_fromaddr":1395179009289033689,"full_fromlogin":15930870058668961531},"models_ids":["another_id","model_id"]})";
        CheckRequestData(requestData, expected);

        TRequestData parsed;
        parsed.FromJson(expected);
        UNIT_ASSERT_VALUES_EQUAL(parsed, requestData);
//                 /* "subject" */,
//                 /* "fromname" */,
//                 /* "" */,
//                 /* "pure_body" */,
//                 /* "" */,
//                 /* "" */,
//                FullSubject /* "full_body" */,
//
//                 /* "" */,
//                 /* "" */,
    }
}
