#include <library/cpp/json/json_reader.h>
#include <util/string/split.h>
#include <util/generic/set.h>
#include <mail/so/spamstop/tools/so-common/algorithm.h>
#include <mail/so/libs/talkative_config/config.h>
#include <library/cpp/string_utils/quote/quote.h>
#include "storage.h"

namespace NLucene{

    void NLucene::TStorage::SetReadTimeout(const TDuration &timeout) {
        requestTimeout = timeout;
    }

    TDuration NLucene::TStorage::GetReadTimeout() const {
        return requestTimeout;
    }

    void NLucene::TStorage::SetWriteTimeout(const TDuration &) {

    }

    TDuration NLucene::TStorage::GetWriteTimeout() const {
        return TDuration();
    }

    size_t TStorage::Count(const TString &/*collectionName*/, const TFindAction &/*action*/) {
        return 0;
    }

    size_t TStorage::Remove(const TString &/*collectionName*/, const TFindAction &/*action*/) {
        return 0;
    }

    static inline void WriteFunc1Arg(TStringBuf function, const NAnyValue::TScalar & value, TStringBuf key, NAnyValue::TWriter & writer) {
        with_av_json_kmap(writer, key) {
            with_av_json_karray(writer, function) {
                writer.Write(value);
            }
        }
    }

    static void WriteInc(const NAnyValue::TScalar & value, TStringBuf key, NAnyValue::TWriter & writer) {
        WriteFunc1Arg("inc", value, key, writer);
    }

    static void WriteSetOnInsert(const NAnyValue::TScalar & value, TStringBuf key, NAnyValue::TWriter & writer) {
        WriteFunc1Arg("default", value, key, writer);
    }

    static void WriteIfIncrs(NAnyValue::TWriter & writer, const TQueryData & query, const TUpdateData & update) {
        if(!query.gt.empty() || !query.lt.empty()) {
            for (const auto &incr : update.incrs) {
                with_av_json_kmap(writer, incr.first) {
                    with_av_json_karray(writer, "inc") {
                        writer.Write(incr.second);
                        with_av_json_map(writer) {
                            with_av_json_karray(writer, "and") {
                                for (const auto &gt : query.gt) {
                                    with_av_json_map(writer) {
                                        with_av_json_karray(writer, "gt") {
                                            with_av_json_map(writer) {
                                                with_av_json_karray(writer, "get") {
                                                    writer.Write(gt.first);
                                                }
                                            }
                                            writer.Write(gt.second);
                                        }
                                    }
                                }
                                for (const auto &lt : query.lt) {
                                    with_av_json_map(writer) {
                                        with_av_json_karray(writer, "lt") {
                                            with_av_json_map(writer) {
                                                with_av_json_karray(writer, "get") {
                                                    writer.Write(lt.first);
                                                }
                                            }
                                            writer.Write(lt.second);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } else {
            for (const auto &incr : update.incrs)
                WriteInc(incr.second, incr.first, writer);
        }
    }

    void WriteUpdateActionToArrayWriter(NAnyValue::TWriter & writer, const TQueryData & query, const TUpdateData & update) {
        if(!query.equals.empty()) {
            with_av_json_map(writer)
                {
                    for (const auto &eq : query.equals)
                        writer.Write(eq.first, eq.second);

                    for (const auto &set : update.sets)
                        writer.Write(set.first, set.second);

                    for (const auto &set : update.setsOnInsert)
                        WriteSetOnInsert(set.second, set.first, writer);

                    WriteIfIncrs(writer, query, update);
                }
        }

        if(!query.in.empty()) {
            for(auto i : xrange(query.in.cbegin()->second.size())) {
                with_av_json_map(writer)
                    {
                        for (const auto &p : query.in) {
                            writer.Write(p.first, p.second[i]);

                            for (const auto &set : update.sets)
                                writer.Write(set.first, set.second);

                            for (const auto &set : update.setsOnInsert)
                                WriteSetOnInsert(set.second, set.first, writer);

                            WriteIfIncrs(writer, query, update);
                        }
                    }
            }
        }

        if(!query.ors.empty()) {
            for(const auto & orQuery : query.ors) {
                WriteUpdateActionToArrayWriter(writer, orQuery, update);
            }
        }
    }

    static void CreateKeysSet(const TQueryData & query, TVector<TString>& sharedKeys, TVector<TString>& ltOnlyKeys, TVector<TString>& gtOnlyKeys) {
        for (const auto &gt : query.gt)
            gtOnlyKeys.emplace_back(gt.first);
        for (const auto &lt : query.lt)
            ltOnlyKeys.emplace_back(lt.first);

        std::sort(gtOnlyKeys.begin(), gtOnlyKeys.end());
        std::sort(ltOnlyKeys.begin(), ltOnlyKeys.end());
        std::set_intersection(gtOnlyKeys.cbegin(), gtOnlyKeys.cend(), ltOnlyKeys.cbegin(), ltOnlyKeys.cend(), std::back_inserter(sharedKeys));

        set_difference_in_place_container(gtOnlyKeys, sharedKeys);
        set_difference_in_place_container(ltOnlyKeys, sharedKeys);
    }

    struct TScalarWriter{
        friend IOutputStream & operator << (IOutputStream & s, const TScalarWriter & writer) {
            const auto & scalar = writer.scalar;
            if(scalar.GetType() == NAnyValue::TScalarType::I64 && scalar.AsI64() < 0)
                s.Write('\\');
            return s << scalar;
        }

        explicit TScalarWriter(const NAnyValue::TScalar & scalar) : scalar(scalar) {}

        const NAnyValue::TScalar & scalar;
    };

    static TString CreateQuery(const TQueryData & query) {
        TStringStream s;

        const char *delimiter = "";

        TVector<TString> sharedKeys(Reserve(std::min(query.gt.size(), query.lt.size())));
        TVector<TString> ltOnlyKeys(Reserve(query.lt.size()));
        TVector<TString> gtOnlyKeys(Reserve(query.gt.size()));
        CreateKeysSet(query, sharedKeys, ltOnlyKeys, gtOnlyKeys);

        for (const auto &key : sharedKeys) {
            s << delimiter << key << ":[" << TScalarWriter(query.gt.find(key)->second) << " TO " << TScalarWriter(query.lt.find(key)->second) << ']';
            delimiter = " AND ";
        }

        for (const auto &key : gtOnlyKeys) {
            s << delimiter << key << ":[" << TScalarWriter(query.gt.find(key)->second) << " TO 9999999]";
            delimiter = " AND ";
        }

        for (const auto &key : ltOnlyKeys) {
            s << delimiter << key << ":[-9999999 TO " << TScalarWriter(query.lt.find(key)->second) << ']';
            delimiter = " AND ";
        }

        for(const auto & eq : query.equals) {
            s << delimiter << eq.first << ':' << TScalarWriter(eq.second);
            delimiter = " AND ";
        }

        if(!query.in.empty()) {
            s << delimiter << '(';
            const char * del = "";
            for(auto i : xrange(query.in.cbegin()->second.size())) {
                s << del << '(';
                del = " OR ";

                const char * del2 = "";
                for(const auto & p : query.in) {
                    s << del2 << p.first << ':' << TScalarWriter(p.second[i]);
                    del2 = " AND ";
                }
                s << ')';
            }
            s << ')';
            delimiter = " AND ";
        }

        {
            const char * orDel = "";
            for(const auto & orQuery : query.ors) {
                s << orDel << '(' << CreateQuery(orQuery) << ')';
                orDel = " OR ";
            }
        }

        return std::move(s.Str());
    }

    size_t TStorage::Update(const TString &collectionName, const TUpdateAction &action) {
        TString localUrl;
        {
            TStringOutput s(localUrl);
            s << masterUrl
                << "/sequential/update?service=" << collectionName;
        }

        TStringStream post;
        {
            NAnyValue::TWriter writer(&post, false);

            with_av_json_map(writer) {
                writer.Write("AddIfNotExists", action.upsert);
                writer.Write("prefix", (action.query.shard.Defined() ? *action.query.shard : prefix));

                with_av_json_karray(writer, "docs") {
                    WriteUpdateActionToArrayWriter(writer, action.query, action.update);
                }
            }
        }
        NCurl::TRequestContext requestContext;
        requestContext
                .SetPostData(post.Str())
                .SetContentType("application/json")
                .SetHost(std::move(localUrl));
        masterPool->Process(std::move(requestContext), poolTraits);

        return 0;
    }

    struct TGroupKey{
        bool operator==(const TGroupKey & g) const { return h == g.h; }
        TGroupKey(TGroupKey&&) = default;
        TGroupKey(const TGroupKey &) = default;

        TGroupKey(bool upsert, const TMaybe<size_t> &shard)
                : upsert(upsert), shard(shard), h(CombineHashes(THash<bool>()(upsert), THash<TMaybe<size_t>>()(shard)))
        {}

        const bool upsert;
        const TMaybe<size_t> shard;
        const size_t h;

        explicit operator size_t() const { return h; }
    };

    void TStorage::UpdateSeries(
            const TString& collectionName,
            const TActionSeries & actions,
            bool /*ordered*/) {

        if(actions.empty())
            return;

        TString localUrl;
        {
            TStringOutput s(localUrl);
            s << masterUrl << "/update?service=" << collectionName
//                    << "&shard=" << 1
//                    << "&wait=true"
                       ;
        }

        THashMap<TGroupKey, TVector<const TUpdateAction*>> groupedActions;

        for(const auto & action : actions) {
            groupedActions[TGroupKey(action.upsert, action.query.shard)].emplace_back(&action);
        }

        TVector<NCurl::TRequestContext> contexts(Reserve(groupedActions.size()));

        TStringStream post;

        for(const auto & group : groupedActions) {
            const bool upsert = group.first.upsert;

            const auto shard = (group.first.shard.Defined() ? *group.first.shard : prefix);

            post << "--testbound\r\n"
                    "URI: /update\r\n"
                    "ZooShardId: " << shard << "\r\n\r\n";


            {
                NAnyValue::TWriter writer(&post, false);

                with_av_json_map(writer) {
                    writer.Write("AddIfNotExists", upsert);
                    writer.Write("prefix", shard);

                    with_av_json_karray(writer, "docs") {
                        for (const auto *action : group.second) {
                            WriteUpdateActionToArrayWriter(writer, action->query, action->update);
                        }
                    }
                }
            }

            post << "\r\n";
        }
        post << "--testbound--";
        Cdbg << post.Str() << Endl;

        contexts.emplace_back()
                .AddHeader("Content-Type: multipart/mixed; boundary=testbound")
                .SetPostData(std::move(post.Str()))
                .SetContentType("application/json")
                .SetConnectTimeout(TDuration::Max())
                .SetRequestTimeout(TDuration::Max())
                .SetHost(std::move(localUrl));

        masterPool->Process(std::move(contexts), poolTraits);
    };

    void TStorage::UpdateBulk(const TString &/*collectionName*/, const TUpdateAction &/*action*/) {
        ythrow TWithBackTrace<yexception>() << "unsupported";
    }

    void TStorage::Find(const TString &collectionName, const TFindAction &action, TFindResults &result) {
        FindWithLimit(collectionName, action, result, Nothing());
    }

    void TStorage::FindOne(const TString & collectionName, const TFindAction & action, NAnyValue::TScalarMap & result) {
        TFindResults results;
        FindWithLimit(collectionName, action, results, 1);
        if(!results.empty())
            result = std::move(results.front());
    }

    class TFuture : public TStorage::IFuture {
    public:
        TFindResults Get(TCont* cont) override {
            const auto url = requestContext.url;
            auto arts = pool.Process(std::move(requestContext), poolTraits, deadline, cont);
            if (arts.code != 200)
                ythrow yexception() << url << " server return error: " << arts.code << "; firstline: " << arts.firstLine << "; body: " << arts.body.Str();

            NAnyValue::TAnyValue target;
            {
                NAnyValue::TAnyValueParser parser(target);

                if(!NJson::ReadJson(&arts.body, &parser))
                    ythrow yexception() << "cannot parse json: " << arts.body.Str();
            }

            if(!target.IsMap())
                return {};

            const auto hitsCount = target.AsMap()["hitsCount"].GetRobust<size_t>();

            TFindResults results;
            results.reserve(hitsCount);

            for(const auto & hit : target.AsMap()["hitsArray"].AsArray()) {
                results.emplace_back(hit.ToScalarMap());
            }

            return results;
        }

        explicit TFuture(
                TInstant deadline,
                NCurl::TStorage & pool,
                const NCurl::TPoolTraits & poolTraits,
                NCurl::TRequestContext && requestContext) :
                deadline(deadline),
                pool(pool),
                poolTraits(poolTraits),
                requestContext(std::move(requestContext))
        { }

    private:
        TInstant deadline;
        NCurl::TStorage & pool;
        const NCurl::TPoolTraits & poolTraits;
        NCurl::TRequestContext requestContext;
    };

    THolder<TStorage::IFuture> TStorage::FindNonblock(
            TInstant deadline,
            const TString& collectionName,
            const TFindAction& action) {
        TStringStream localUrl;
        {
            localUrl << replicaUrl
                     << "/sequential/search?service=" << collectionName
                     << "&prefix=" << (action.query.shard.Defined() ? *action.query.shard : prefix)
                     << "&get=*";

            localUrl << "&text=" << CGIEscapeRet(CreateQuery(action.query));
        }

        NCurl::TRequestContext requestContext;
        requestContext
                .SetRequestTimeout(requestTimeout)
                .SetConnectTimeout(connectTimeout)
                .SetHost(std::move(localUrl.Str()));
        return MakeHolder<TFuture>(deadline, *replicaPool, poolTraits, std::move(requestContext));
    }

    void TStorage::Connect(const TString & uri) {
        Split(uri, ",", masterUrl, replicaUrl);
    }

    static void ConfigurePool(const NConfig::TConfig & config, const TString& name, TString& url, THolder<NCurl::TStorage>& pool) {
        const auto& poolConf = NTalkativeConfig::Get(config, name);
        url = NTalkativeConfig::Get<TString>(poolConf, "url");

        if (poolConf.Has("pool")) {
            pool = MakeHolder<NCurl::TStorage>(TPoolParams(poolConf["pool"]));
        }
    }

    void TStorage::Connect(const NConfig::TConfig & config) {
        ConfigurePool(config, "master", masterUrl, masterPool);
        ConfigurePool(config, "replica", replicaUrl, replicaPool);

        if(config.Has("request_timeout"))
            requestTimeout = NTalkativeConfig::As<TDuration>(config, "request_timeout");

        if(config.Has("connect_timeout"))
            connectTimeout = NTalkativeConfig::As<TDuration>(config, "connect_timeout");

        if(config.Has("prefix")) {
            const auto & prefixConf = config["prefix"];
            if(!prefixConf.IsNumeric())
                ythrow TWithBackTrace<yexception>() << "prefix for lucene must be numeric, got: " << prefixConf;
            prefix = prefixConf.As<size_t>();
        }
    }

    void TStorage::FindWithLimit(
            const TString &collectionName,
            const TFindAction &action,
            TFindResults &results,
            const TMaybe<size_t> &count) {

        TStringStream localUrl;
        {
            localUrl << replicaUrl
              << "/sequential/search?service=" << collectionName
              //              << "&failover-delay=20"
              << "&prefix=" << (action.query.shard.Defined() ? *action.query.shard : prefix)
              << "&get=*"
//          << "&wait=true"
                    ;

            if (Y_UNLIKELY(count.Defined()))
                localUrl << "&length=" << *count;

            localUrl << "&text=" << CGIEscapeRet(CreateQuery(action.query));
        }
        NCurl::TRequestContext requestContext;
        requestContext
                .SetRequestTimeout(requestTimeout)
                .SetConnectTimeout(connectTimeout)
                .SetHost(std::move(localUrl.Str()));

        auto arts = replicaPool->Process(std::move(requestContext), poolTraits);

        if (arts.code != 200)
            return;

        NAnyValue::TAnyValue target;
        {
            NAnyValue::TAnyValueParser parser(target);

            if(!NJson::ReadJson(&arts.body, &parser))
                ythrow yexception() << "cannot parse json: " << arts.body.Str();
        }

        if(!target.IsMap())
            return;

        const auto hitsCount = target.AsMap()["hitsCount"].GetRobust<size_t>();

        results.reserve(hitsCount);

        for(const auto & hit : target.AsMap()["hitsArray"].AsArray()) {
            results.emplace_back(hit.ToScalarMap());
        }

//        search?service=so_test1&prefix=0
    };
}   //namespace NLucene

