/*
 * Collection.cpp
 *
 *  Created on: 15 февр. 2016 г.
 *      Author: luckybug
 */

#include "Collection.h"
#include "Error.h"

namespace mongo_v3 {
    void Collection::setWriteTimeout(const TDuration& timeout) {
        WriteConcern wc = getWriteConcern();
        wc.setTimeout(timeout);
        setWriteConcern(wc);
    }

    TDuration Collection::getWriteTimeout() const {
        return getWriteConcern().getTimeout();
    }

    void Collection::createIndex(const Document& keys) {
        bson_error_t error;
        if (!mongoc_collection_create_index(collection, keys.bson, mongoc_index_opt_get_default(), &error))
            throw Error(error);
    }

    void Collection::remove(const Document& query) {
        bson_error_t error;
        if (!mongoc_collection_remove(
                collection,
                MONGOC_REMOVE_NONE,
                query.bson,
                NULL,
                &error))
            throw Error(error);
    }

    size_t Collection::size() const {
        bson_error_t error;
        i64 s = mongoc_collection_count(collection, MONGOC_QUERY_NONE, NULL, 0, 0, NULL, &error);

        if (s < 0)
            throw Error(error) << "error in response size of collection";

        return s;
    }

    size_t Collection::count(const Document& query) const {
        bson_error_t error;
        i64 s = mongoc_collection_count(collection, MONGOC_QUERY_NONE, query.bson, 0, 0, NULL, &error);

        if (s < 0)
            throw Error(error) << "error in response size of collection";

        return s;
    }

    void Collection::update(const Document& query, const Document& doc, bool upsert) {
        bson_error_t error;
        bool res = mongoc_collection_update(
            collection,
            upsert ? MONGOC_UPDATE_UPSERT : MONGOC_UPDATE_MULTI_UPDATE,
            query.bson,
            doc.bson,
            NULL,
            &error);
        if (!res)
            throw Error(error) << "update error";
    }

    Document Collection::findAndModify(const Document& query, const Document& update, bool upsert, bool _new) {
        bson_t reply;
        bson_error_t error;
        bool res = mongoc_collection_find_and_modify(
            collection,
            query.bson,
            NULL,
            update.bson,
            NULL,
            false,
            upsert,
            _new,
            &reply,
            &error);

        if (!res)
            throw Error(error) << "findAndModify error";

        Document replyDoc(&reply);
        bson_destroy(&reply);

        std::pair<bool, Document::Iterator> it = replyDoc.find("value");

        if (!it.first)
            throw Error(error) << "findAndModify error";

        return it.second.getWithCasting<Document>();
    }

    Cursor Collection::find(const Document& query, size_t limit) {
        Document opts("limit", static_cast<i64>(limit));
        Cursor cursor(mongoc_collection_find_with_opts(collection, query.bson, opts.bson, NULL));
        cursor.setTimeout(readTimeout);

        return cursor;
    }

    Collection::Collection(mongoc_client_t* client, const char* db, const char* collectionName) {
        collection = mongoc_client_get_collection(client, db, collectionName);
    }

    Collection::Collection(const Collection& c) {
        collection = mongoc_collection_copy(c.collection);
        setReadTimeout(c.readTimeout);
    }

    Collection& Collection::operator=(const Collection& c) {
        uninit();
        collection = mongoc_collection_copy(c.collection);
        setReadTimeout(c.readTimeout);

        return *this;
    }

    Collection::~Collection() {
        uninit();
    }

    void Collection::uninit() {
        if (collection)
            mongoc_collection_destroy(collection);
        collection = 0;
        readTimeout = TDuration();
    }
    void Collection::setWriteConcern(const WriteConcern& concern) {
        if (!concern.isValid())
            throw Error() << "concern isn't valid";
        mongoc_collection_set_write_concern(collection, concern.getRawConcern());
    }

    WriteConcern Collection::getWriteConcern() const {
        return mongoc_collection_get_write_concern(collection);
    };
} /* namespace mongo_v3 */
