/*
 * Document.cpp
 *
 *  Created on: 15 февр. 2016 г.
 *      Author: luckybug
 */

#include "Document.h"
#include "Error.h"

#include <util/generic/yexception.h>

namespace mongo_v3 {
    template <>
    bool TestCompatible<i32>(BsonType bType) {
        return bType == BT_INT32 || bType == BT_INT64 || BT_DOUBLE;
    }
    template <>
    bool TestCompatible<i64>(BsonType bType) {
        return bType == BT_INT32 || bType == BT_INT64 || BT_DOUBLE;
    }
    template <>
    bool TestCompatible<double>(BsonType bType) {
        return bType == BT_INT32 || bType == BT_INT64 || BT_DOUBLE;
    }
    template <>
    bool TestCompatible<const char*>(BsonType bType) {
        return bType == BT_STRING;
    }
    template <>
    bool TestCompatible<Oid>(BsonType bType) {
        return bType == BT_OID;
    }
    template <>
    bool TestCompatible<DocumentArray>(BsonType bType) {
        return bType == BT_ARRAY;
    }
    template <>
    bool TestCompatible<Document>(BsonType bType) {
        return bType == BT_DOCUMENT;
    }
    template <>
    bool TestCompatible<bool>(BsonType bType) {
        return bType == BT_BOOL || bType == BT_INT32 || bType == BT_INT64 || BT_DOUBLE;
    }
    template <>
    bool TestCompatible<BinaryData>(BsonType bType) {
        return bType == BT_BINARY;
    }

    IOutputStream& operator<<(IOutputStream& out, BsonType type) {
        switch (type) {
            case BT_DOUBLE:
                return out << "double";
            case BT_INT64:
                return out << "int64";
            case BT_INT32:
                return out << "int32";
            case BT_STRING:
                return out << "string";
            case BT_OID:
                return out << "oid";
            case BT_ARRAY:
                return out << "array";
            case BT_DOCUMENT:
                return out << "document";
            case BT_BOOL:
                return out << "bool";
            case BT_BINARY:
                return out << "binary";
            default:
                return out << "unknown " << static_cast<int>(type);
        }
    }

    BsonType ConvertBsonType(bson_type_t t) {
        switch (t) {
            case BSON_TYPE_DOUBLE:
                return BT_DOUBLE;
            case BSON_TYPE_INT64:
                return BT_INT64;
            case BSON_TYPE_INT32:
                return BT_INT32;
            case BSON_TYPE_UTF8:
                return BT_STRING;
            case BSON_TYPE_OID:
                return BT_OID;
            case BSON_TYPE_DOCUMENT:
                return BT_DOCUMENT;
            case BSON_TYPE_ARRAY:
                return BT_ARRAY;
            case BSON_TYPE_BOOL:
                return BT_BOOL;
            case BSON_TYPE_BINARY:
                return BT_BINARY;
            default:
                return BT_UNKNOWN;
        }
    };

    TString Oid::toStr() const {
        char tmp[25];
        bson_oid_to_string(&oid, tmp);
        return tmp;
    }
    bool Oid::operator==(const Oid& anotherOid) const {
        return bson_oid_equal(&oid, &anotherOid.oid);
    }
    bool Oid::operator!=(const Oid& anotherOid) const {
        return !bson_oid_equal(&oid, &anotherOid.oid);
    }
    Oid::Oid(const char* str) {
        bson_oid_init_from_string(&oid, str);
    }

    Oid::Oid(const bson_oid_t* _oid) {
        bson_oid_copy(_oid, &oid);
    }

    void BsonOwner::reset(const bson_t* b) {
        uninit();
        bson = (b ? bson_copy(b) : bson_new());
    }

    BsonOwner::BsonOwner()
        : bson(NULL)
    {
        init();
    }

    BsonOwner::BsonOwner(const bson_t* b)
        : bson(NULL)
    {
        reset(b);
    }

    BsonOwner::BsonOwner(const BsonMove& b)
        : bson(b.bson)
    {
    }

    BsonOwner::BsonOwner(const BsonOwner& anotherDoc)
        : bson(NULL)
    {
        reset(anotherDoc.bson);
    }
    BsonOwner& BsonOwner::operator=(const BsonOwner& anotherDoc) {
        reset(anotherDoc.bson);
        return *this;
    }

    BsonOwner::~BsonOwner() {
        uninit();
    }

    void BsonOwner::init() {
        uninit();

        bson = bson_new();
    }

    void BsonOwner::uninit() {
        if (bson) {
            bson_destroy(bson);
            bson = 0;
        }
    }

    Document::Document()
        : BsonOwner()
    {
    }
    Document::Document(const bson_t* b)
        : BsonOwner(b)
    {
    }
    Document::Document(const BsonMove& b)
        : BsonOwner(b)
    {
    }
    Document::Document(const Document& anotherDoc)
        : BsonOwner(anotherDoc)
    {
    }

    IOutputStream& operator<<(IOutputStream& stream, const Document& doc) {
        return stream << doc.toStr();
    }

    Document& Document::append(const char* key, i32 doc) {
        BSON_APPEND_INT32(bson, key, doc);
        return *this;
    }

    Document& Document::append(const char* key, i64 doc) {
        BSON_APPEND_INT64(bson, key, doc);
        return *this;
    }

    Document& Document::append(const char* key, const char* doc) {
        BSON_APPEND_UTF8(bson, key, doc);
        return *this;
    }

    Document& Document::append(const char* key, double doc) {
        BSON_APPEND_DOUBLE(bson, key, doc);
        return *this;
    }

    Document& Document::append(const char* key, const Document& doc) {
        BSON_APPEND_DOCUMENT(bson, key, doc.bson);
        return *this;
    }

    Document& Document::append(const char* key, const DocumentArray& doc) {
        BSON_APPEND_ARRAY(bson, key, doc.bson);
  return *this;
    }

    Document& Document::append(const char* key, const Oid& doc) {
        BSON_APPEND_OID(bson, key, &doc.oid);
        return *this;
    }

    Document& Document::append(const char* key, bool doc) {
        BSON_APPEND_BOOL(bson, key, doc);
        return *this;
    }

    Document& Document::append(const char* key, const BinaryData& doc) {
        BSON_APPEND_BINARY(bson, key, BSON_SUBTYPE_BINARY, static_cast<const ui8*>(doc.data()), doc.size());
        return *this;
    }

    Document Document::lessThen(int treshold) {
        return mongo_v3::Document("$lt", treshold);
    }

    Document Document::greaterThen(int treshold) {
        return mongo_v3::Document("$gt", treshold);
    }

    Document Document::createFromJSON(const TString& jsonStr) {
        bson_error_t error;
        bson_t* doc = bson_new_from_json(reinterpret_cast<const ui8*>(jsonStr.c_str()), jsonStr.size(), &error);

        if (!doc) {
            ythrow Error(error);
        }

        return Document(BsonMove(doc));
    }

    TString Document::toStr() const {
        size_t length = 0;
        char* raw = bson_as_json(bson, &length);
        TString result(raw, raw + length);
        bson_free(raw);
        return result;
    }

    bool Document::isEmpty() const {
        return bson_empty0(bson);
    }

    const unsigned char* Document::objdata() const {
        return bson_get_data(bson);
    }

    size_t Document::objsize() const {
        return bson->len;
    }

    Document::Iterator Document::iter() const {
        return Document::Iterator(*this);
    }

    std::pair<bool, Document::Iterator> Document::find(const char* key) const {
        Document::Iterator it = iter();
        bool res = bson_iter_find(&it.iter, key);

        return std::make_pair(res, it);
    }
    //// Document Iterator

    const char* Document::Iterator::key() const {
        return bson_iter_key(&iter);
    }
    BsonType Document::Iterator::type() const {
        return ConvertBsonType(bson_iter_type(&iter));
    }

    template <>
    i32 Document::Iterator::get<i32>() const {
        return bson_iter_int32(&iter);
    }
    template <>
    i64 Document::Iterator::get<i64>() const {
        return bson_iter_int64(&iter);
    }
    template <>
    double Document::Iterator::get<double>() const {
        return bson_iter_double(&iter);
    }
    template <>
    Oid Document::Iterator::get<Oid>() const {
        return bson_iter_oid(&iter);
    }

    template <>
    Document Document::Iterator::get<Document>() const {
        ui32 len = 0;
        const unsigned char* data = 0;
        bson_iter_document(&iter, &len, &data);

        if (!data)
            return Document();

        return Document(BsonMove(bson_new_from_data(data, len)));
    }

    template <>
    DocumentArray Document::Iterator::get<DocumentArray>() const {
        const ui8* data = NULL;
        ui32 len = 0;
        bson_iter_array(&iter, &len, &data);

        if (!data)
            return DocumentArray();

        return DocumentArray(BsonMove(bson_new_from_data(data, len)));
    }

    template <>
    const char* Document::Iterator::get<const char*>() const {
        ui32 l = 0;
        return bson_iter_utf8(&iter, &l);
    }

    template <>
    bool Document::Iterator::get<bool>() const {
        return bson_iter_bool(&iter);
    }

    template <>
    BinaryData Document::Iterator::get<BinaryData>() const {
        bson_subtype_t subtype;
        ui32 size = 0;
        const ui8* data = NULL;
        bson_iter_binary(&iter, &subtype, &size, &data);

        return BinaryData(data, size);
    }

    template <>
    double Document::Iterator::getWithCasting<double>() const {
        const BsonType t = type();
        switch (t) {
            case BT_DOUBLE:
                return get<double>();
            case BT_INT64:
                return double(get<i64>());
            case BT_INT32:
                return double(get<i32>());
            default:
                ythrow yexception() << "cannot cast " << t << " to double";
        }
    }
    template <>
    i64 Document::Iterator::getWithCasting<i64>() const {
        const BsonType t = type();
        switch (t) {
            case BT_DOUBLE:
                return i64(get<double>());
            case BT_INT64:
                return get<i64>();
            case BT_INT32:
                return i64(get<i32>());
            default:
                ythrow yexception() << "cannot cast " << t << " to i64";
        }
    }
    template <>
    bool Document::Iterator::getWithCasting<bool>() const {
        const BsonType t = type();
        switch (t) {
            case BT_BOOL:
                return get<bool>();
            case BT_DOUBLE:
                return bool(get<double>());
            case BT_INT64:
                return bool(get<i64>());
            case BT_INT32:
                return bool(get<i32>());
            default:
                ythrow yexception() << "cannot cast " << t << " to bool";
        }
    }

    template <>
    i32 Document::Iterator::getWithCasting<i32>() const {
        const BsonType t = type();
        switch (t) {
            case BT_DOUBLE:
                return i32(get<double>());
            case BT_INT64:
                return i32(get<i64>());
            case BT_INT32:
                return get<i32>();
            default:
                ythrow yexception() << "cannot cast " << t << " to i32";
        }
    }

    template <>
    const char* Document::Iterator::getWithCasting<const char*>() const {
        const BsonType t = type();
        if (!TestCompatible<const char*>(t))
            ythrow yexception()
                << "Bson type " << t
                << " doesn't compaitiable with type const char*";
        return get<const char*>();
    }

    template <>
    Oid Document::Iterator::getWithCasting<Oid>() const {
        const BsonType t = type();
        if (!TestCompatible<Oid>(t))
            ythrow yexception()
                << "Bson type " << t
                << " doesn't compaitiable with type Oid";
        return get<Oid>();
    }

    template <>
    Document Document::Iterator::getWithCasting<Document>() const {
        const BsonType t = type();
        if (!TestCompatible<Document>(t))
            ythrow yexception()
                << "Bson type " << t
                << " doesn't compaitiable with type Document";
        return get<Document>();
    }

    template <>
    DocumentArray Document::Iterator::getWithCasting<DocumentArray>() const {
        const BsonType t = type();
        if (!TestCompatible<DocumentArray>(t))
            ythrow yexception()
                << "Bson type " << t
                << " doesn't compaitiable with type DocumentArray";
        return get<DocumentArray>();
    }

    template <>
    BinaryData Document::Iterator::getWithCasting<BinaryData>() const {
        const BsonType t = type();
        if (!TestCompatible<BinaryData>(t))
            ythrow yexception()
                << "Bson type " << t
                << " doesn't compaitiable with type BinaryData";
        return get<BinaryData>();
    }

    Document::Iterator::Iterator(const Document& bson) {
        bson_iter_init(&iter, bson.bson);
    }
    bool Document::Iterator::next() {
        return bson_iter_next(&iter);
    }

    bool Document::Iterator::getBool() const {
        return getWithCasting<bool>();
    }
    i32 Document::Iterator::getI32() const {
        return getWithCasting<i32>();
    }
    i64 Document::Iterator::getI64() const {
        return getWithCasting<i64>();
    }
    double Document::Iterator::getDouble() const {
        return getWithCasting<double>();
    }
    const char* Document::Iterator::getString() const {
        return getWithCasting<const char*>();
    }
    Oid Document::Iterator::getOid() const {
        return getWithCasting<Oid>();
    }
    Document Document::Iterator::getDocument() const {
        return getWithCasting<Document>();
    }
    DocumentArray Document::Iterator::getArray() const {
        return getWithCasting<DocumentArray>();
    }
    BinaryData Document::Iterator::getBinary() const {
        return getWithCasting<BinaryData>();
    }

} /* namespace mongo_v3 */
