#pragma once

#include <string>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <contrib/libs/mongo-c-driver/libbson/src/bson/bson.h>
#include <contrib/libs/mongo-c-driver/libmongoc/src/mongoc/mongoc.h>

namespace mongo_v3 {
    enum BsonType { BT_UNKNOWN = 0,
                    BT_DOUBLE,
                    BT_INT64,
                    BT_INT32,
                    BT_STRING,
                    BT_OID,
                    BT_DOCUMENT,
                    BT_ARRAY,
                    BT_BOOL,
                    BT_BINARY };

    template <typename T>
    bool TestCompatible(BsonType);

    IOutputStream& operator<<(IOutputStream& out, BsonType type);

    struct BinaryData {
        size_t size() const {
            return _size;
        }
        const void* data() const {
            return _data;
        }
        BinaryData()
            : _data()
            , _size()
        {
        }
        BinaryData(const void* data, size_t size)
            : _data(data)
            , _size(size)
        {
        }

    private:
        const void* _data;
        size_t _size;
    };

    class Oid {
        bson_oid_t oid;
        friend class Document;

    public:
        TString toStr() const;
        bool operator==(const Oid& anotherOid) const;
        bool operator!=(const Oid& anotherOid) const;
        Oid(const char* str);
        Oid(const bson_oid_t* _oid);
    };

    struct BsonMove {
        bson_t* bson;
        explicit BsonMove(bson_t* bson)
            : bson(bson)
        {
        }
    };

    class BsonOwner {
    public:
        BsonOwner();
        BsonOwner(const bson_t* b);
        BsonOwner(const BsonMove& b);
        BsonOwner(const BsonOwner& anotherDoc);
        BsonOwner& operator=(const BsonOwner& anotherDoc);
        virtual ~BsonOwner();

    protected:
        bson_t* bson;

    private:
        void reset(const bson_t* b);

        void init();
        void uninit();
    };

    class DocumentArray;

    class Document: public BsonOwner {
        friend class Cursor;
        friend class Bulk;
        friend class Collection;

    public:
        class Iterator {
            friend class Document;
            bson_iter_t iter;

        private:
            template <typename T>
            T get() const;

        public:
            Iterator(const Iterator& _i) {
                Iterator& c = const_cast<Iterator&>(_i);
                std::swap(iter, c.iter);
            }
            Iterator& operator=(const Iterator& _i) {
                Iterator& c = const_cast<Iterator&>(_i);
                std::swap(iter, c.iter);
                return *this;
            }
            const char* key() const;

            BsonType type() const;

            template <typename T>
            T getWithCasting() const;

            bool getBool() const;
            i32 getI32() const;
            i64 getI64() const;
            double getDouble() const;
            const char* getString() const;
            Oid getOid() const;
            Document getDocument() const;
            DocumentArray getArray() const;
            BinaryData getBinary() const;

            bool next();
            Iterator(const Document& bson);
        };

        const unsigned char* objdata() const;
        size_t objsize() const;

        Iterator iter() const;

        std::pair<bool, Iterator> find(const char* key) const;

        Document& append(const char* key, bool doc);
        Document& append(const char* key, i32 doc);
        Document& append(const char* key, i64 doc);
        Document& append(const char* key, double doc);
        Document& append(const char* key, const char* doc);
        Document& append(const char* key, const Oid& doc);
        Document& append(const char* key, const Document& doc);
        Document& append(const char* key, const DocumentArray& doc);
        Document& append(const char* key, const BinaryData& doc);

        TString toStr() const;

        bool isEmpty() const;

        static Document lessThen(int treshold);
        static Document greaterThen(int treshold);

        static Document createFromJSON(const TString& jsonStr);

        Document();
        Document(const bson_t* b);
        Document(const BsonMove& b);
        Document(const Document& anotherDoc);

        template <typename T>
        Document(const char* key, const T& doc)
            : BsonOwner(0)
        {
            append(key, doc);
        }

        virtual ~Document(){};

    private:
    };

    IOutputStream& operator<<(IOutputStream& stream, const Document& doc);

    class DocumentArray: public Document {
    public:
        struct ArrayIterator: public std::iterator<std::random_access_iterator_tag, Iterator> {
            friend class DocumentArray;
            ArrayIterator& operator++() {
                index++;
                return *this;
            }
            bool operator!=(const ArrayIterator& it) const {
                return index != it.index;
            }
            Iterator operator*() {
                const std::pair<bool, Iterator>& it = parent.find(parent.indexToStr(index));
                return it.second;
            }
            Iterator operator->() {
                return this->operator*();
            }

        private:
            const DocumentArray& parent;
            size_t index;

            ArrayIterator(const DocumentArray& parent, size_t index)
                : parent(parent)
                , index(index)
            {
            }
        };

        ArrayIterator begin() const {
            return ArrayIterator(*this, 0);
        }
        ArrayIterator end() const {
            return ArrayIterator(*this, size());
        }

        template <typename T>
        T at(size_t i) const {
            const std::pair<bool, Iterator>& res = this->find(this->indexToStr(i));
            if (!res.first)
                return T();
            return res.second.getWithCasting<T>();
        }

        void push_back(const char* doc) {
            append(inc(), doc);
        }

        size_t size() const {
            return m_size;
        }
        template <typename T>
        void push_back(const T& doc) {
            append(inc(), doc);
        }

        DocumentArray()
            : m_size()
        {
        }
        DocumentArray(const bson_t* b)
            : Document(b)
        {
            m_size = bson_count_keys(bson);
        };
        DocumentArray(const BsonMove& b)
            : Document(b)
        {
            m_size = bson_count_keys(bson);
        };

    private:
        const char* indexToStr(size_t i) const {
            const char* key = 0;
            bson_uint32_to_string(i, &key, tmp, sizeof(tmp));
            return key;
        }
        const char* inc() {
            return indexToStr(m_size++);
        }

    private:
        mutable char tmp[64];
        size_t m_size;
    };

} /* namespace mongo_v3 */
