#include <contrib/libs/mongo-c-driver/libbson/src/bson/bson.h>
#include <library/cpp/config/config.h>
#include <util/system/env.h>
#include <util/string/subst.h>
#include "StorageMongo.h"
#include "Document.h"
#include "Client.h"
#include "Bulk.h"

void appendAnyValueToArray(mongo_v3::DocumentArray& doc, const NAnyValue::TAnyValue& anyValue);
void appendAnyValueToDoc(mongo_v3::Document& doc, const TString& key, const NAnyValue::TAnyValue& anyValue);
void appendAnyValueToDoc(mongo_v3::Document& doc, const TString& key, const NAnyValue::TScalar& anyValue);

mongo_v3::Document createDocFromHashMap(const NAnyValue::TMap& hashMap);
mongo_v3::Document createDocFromHashMap(const NAnyValue::TScalarMap & hashMap);
void createHashMapFromDoc(const mongo_v3::Document& doc, NAnyValue::TMap& res);
void createHashMapFromDoc(const mongo_v3::Document& doc, NAnyValue::TScalarMap & res);

mongo_v3::Document createQueryFromAction(const TQueryData&query);
mongo_v3::Document createUpdateFromAction(const TUpdateAction& action);

static IOutputStream & operator<<(IOutputStream & stream, const NConfig::TConfig & config) {
    stream << '\n';
    config.DumpJson(stream);
    return stream;
}

namespace mongo_v3 {
    void TStorageMongo::SetReadTimeout(const TDuration& timeout) {
        m_mongo_pool->setReadTimeout(timeout);
    };
    TDuration TStorageMongo::GetReadTimeout() const {
        return m_mongo_pool->getReadTimeout();
    };
    void TStorageMongo::SetWriteTimeout(const TDuration& timeout) {
        m_mongo_pool->setWriteTimeout(timeout);
    };
    TDuration TStorageMongo::GetWriteTimeout() const {
        return m_mongo_pool->getWriteTimeout();
    };

    size_t TStorageMongo::Count(
        const TString& collectionName,
        const TFindAction& action) {
        const mongo_v3::Document& query = createQueryFromAction(action.query);

        TSimpleSharedPtr<mongo_v3::Client> client = m_mongo_pool->getConnection();
        mongo_v3::Collection collection = client->getCollection(dbName.c_str(), collectionName.c_str());

        return collection.count(query);
    }

    size_t TStorageMongo::Remove(
        const TString& collectionName,
        const TFindAction& action) {
        const mongo_v3::Document& query = createQueryFromAction(action.query);

        TSimpleSharedPtr<mongo_v3::Client> client = m_mongo_pool->getConnection();
        mongo_v3::Collection collection = client->getCollection(dbName.c_str(), collectionName.c_str());

        collection.remove(query);
        return 0;
    }

    size_t TStorageMongo::Update(
        const TString& collectionName,
        const TUpdateAction& action) {
        const mongo_v3::Document& query = createQueryFromAction(action.query);
        const mongo_v3::Document& update = createUpdateFromAction(action);

        TSimpleSharedPtr<mongo_v3::Client> client = m_mongo_pool->getConnection();
        mongo_v3::Collection collection = client->getCollection(dbName.c_str(), collectionName.c_str());

        collection.update(query, update, action.upsert);
        return 0;
    }

    void TStorageMongo::UpdateSeries(
        const TString& collectionName,
        const TActionSeries & series,
        bool ordered) {
        TSimpleSharedPtr<mongo_v3::Client> client = m_mongo_pool->getConnection();
        mongo_v3::Collection collection = client->getCollection(dbName.c_str(), collectionName.c_str());
        mongo_v3::Bulk bulk(collection, ordered);

        for (auto it = series.cbegin(); it != series.cend(); ++it) {
            const mongo_v3::Document& query = createQueryFromAction(it->query);
            const mongo_v3::Document& update = createUpdateFromAction(*it);

            bulk.update(query, update, it->upsert);
        }

        bulk.exec();
    }

    void TStorageMongo::Find(
        const TString& collectionName,
        const TFindAction& action,
        TFindResults& result) {
        const mongo_v3::Document& query = createQueryFromAction(action.query);

        TSimpleSharedPtr<mongo_v3::Client> client = m_mongo_pool->getConnection();
        mongo_v3::Collection collection = client->getCollection(dbName.c_str(), collectionName.c_str());

        mongo_v3::Cursor cursor = collection.find(query);

        mongo_v3::Document doc;
        while (cursor.nextDoc(doc)) {
            createHashMapFromDoc(doc, result.emplace_back());
        }
    }

    void TStorageMongo::FindOne(
        const TString& collectionName,
        const TFindAction& action,
        NAnyValue::TScalarMap & result) {
        const mongo_v3::Document& query = createQueryFromAction(action.query);

        TSimpleSharedPtr<mongo_v3::Client> client = m_mongo_pool->getConnection();
        mongo_v3::Collection collection = client->getCollection(dbName.c_str(), collectionName.c_str());

        mongo_v3::Cursor cursor = collection.find(query, 1);

        mongo_v3::Document doc;
        if (cursor.nextDoc(doc)) {
            createHashMapFromDoc(doc, result);
        }
    }

    void TStorageMongo::Connect(const TString& uri) {
        mongo_v3::URI mongoUri(uri);
        m_mongo_pool.Reset(new mongo_v3::ClientPool(mongoUri));
    }

    void TStorageMongo::Connect(const NConfig::TConfig & config) {
        const auto pass = [&config]() -> TMaybe<std::tuple<TString, TString>> {
            if(!config.Has("pass_template"))
                return Nothing();

            auto envVarName = config["pass_template"].As<TString>();
            auto envVar = GetEnv(envVarName, nullptr);
            if(!envVar)
                return Nothing();

            return std::make_tuple(std::move(envVarName), std::move(envVar));
        }();


        if(config.Has("db")) {
            const auto dbConf = config["db"];
            if(!dbConf.IsA<TString>()) {
                ythrow TWithBackTrace<yexception>() << "db field must be string: " << dbConf;
            }

            dbName = dbConf.Get<TString>();
        }

        if(config.Has("uri")) {
            const auto localConf = config["uri"];
            if(!localConf.IsA<TString>()) {
                ythrow TWithBackTrace<yexception>() << "uri field must be string: " << localConf;
            }

            TString uri = localConf.Get<TString>();
            if(pass.Defined())
                SubstGlobal(uri, std::get<0>(*pass), std::get<1>(*pass));

            else
                Connect(localConf.Get<TString>());
        } else
            ythrow TWithBackTrace<yexception>() << "config must contain uri field: " << config;
    }
    TStorageMongo::TStorageMongo() = default;

    TStorageMongo::TStorageMongo(const TString& dbName) : dbName(dbName) {}

    TStorageMongo::~TStorageMongo() = default;

} /* namespace mongo_v3 */

template <typename TBase>
struct TMongoVisitor : private TBase {
    template<typename ... Types>
    explicit TMongoVisitor(Types&& ... args) : TBase(std::forward<Types>(args) ...) {}

    void operator()(const TNoType&) { }

    void operator()(const NAnyValue::TMap& map) {
        this->push(createDocFromHashMap(map));
    }

    void operator()(const NAnyValue::TArray& array) {
        mongo_v3::DocumentArray arr;
        for (const auto & val : array) {
            appendAnyValueToArray(arr, val);
        }
        this->push(arr);
    }

    void operator()(const NAnyValue::TScalar& scalar) {
        scalar.Visit(*this);
    }

    void operator()(const TString& value) {
        this->push(value.c_str());
    }

    void operator()(const NAnyValue::TOid& oid) {
        (*this)(oid.get());
    }

    template<typename T>
    void operator()(const T& value) {
        this->push(static_cast<NAnyValue::make_signed_t<T>>(value));
    }
};

class TDocumentArrayBase {
public:
    TDocumentArrayBase(mongo_v3::DocumentArray& doc) : doc(doc) { }

    template<typename T>
    void push(const T& value) {
        doc.push_back(value);
    }

private:
    mongo_v3::DocumentArray& doc;
};

void appendAnyValueToArray(mongo_v3::DocumentArray& doc, const NAnyValue::TAnyValue& anyValue) {
    TMongoVisitor<TDocumentArrayBase> visitor(doc);
    anyValue.Visit(visitor);
}

class TDocumentBase {
public:
    TDocumentBase(mongo_v3::Document& doc, const TString& key) : doc(doc), key(key) { }

    template<typename T>
    void push(const T& value) {
        doc.append(key.c_str(), value);
    }

private:
    mongo_v3::Document& doc;
    const TString& key;
};

void appendAnyValueToDoc(mongo_v3::Document& doc, const TString& key, const NAnyValue::TAnyValue& anyValue) {
    TMongoVisitor<TDocumentBase> visitor(doc, key);
    anyValue.Visit(visitor);
}

void appendAnyValueToDoc(mongo_v3::Document& doc, const TString& key, const NAnyValue::TScalar& scalar) {
    TMongoVisitor<TDocumentBase> visitor(doc, key);
    scalar.Visit(visitor);
}

void appendDocToList(NAnyValue::TArray & list, const mongo_v3::Document::Iterator& it) {
    mongo_v3::BsonType bsonType = it.type();
    switch (bsonType) {
        case mongo_v3::BT_DOUBLE:
            list.emplace_back(it.getDouble());
            break;
        case mongo_v3::BT_INT64:
            list.emplace_back(static_cast<ui64>(it.getI64()));
            break;
        case mongo_v3::BT_INT32:
            list.emplace_back(static_cast<ui32>(it.getI32()));
            break;
        case mongo_v3::BT_STRING:
            list.emplace_back(TString(it.getString()));
            break;
        case mongo_v3::BT_OID:
            list.emplace_back(NAnyValue::TOid(it.getOid().toStr()));
            break;
        case mongo_v3::BT_DOCUMENT:
            createHashMapFromDoc(it.getDocument(), list.emplace_back(NAnyValue::TMap()).AsMap());
            break;
        case mongo_v3::BT_ARRAY: {
            auto & subList = list.emplace_back(NAnyValue::TArray()).AsArray();
            const mongo_v3::DocumentArray& arr = it.getArray();
            subList.reserve(arr.size());
            for (const auto & v : arr) {
                appendDocToList(subList, v);
            }
            break;
        }
        case mongo_v3::BT_BOOL:
            list.emplace_back(it.getBool());
            break;
        case mongo_v3::BT_BINARY:
            ythrow TWithBackTrace<yexception>() << "unsupported";
        case mongo_v3::BT_UNKNOWN:

            break;
    }
}

mongo_v3::Document createDocFromHashMap(const NAnyValue::TMap& hashMap) {
    mongo_v3::Document doc;
    for (const auto &it : hashMap) {
        const TString& key = it.first;
        const NAnyValue::TAnyValue& anyValue = it.second;

        appendAnyValueToDoc(doc, key, anyValue);
    }
    return doc;
}

mongo_v3::Document createDocFromHashMap(const NAnyValue::TScalarMap & hashMap) {
    mongo_v3::Document doc;
    for (const auto &it : hashMap) {
        const TString& key = it.first;
        const auto& anyValue = it.second;

        appendAnyValueToDoc(doc, key, anyValue);
    }
    return doc;
}

void createHashMapFromDoc(const mongo_v3::Document& doc, NAnyValue::TMap& res) {
    mongo_v3::Document::Iterator it(doc);
    while (it.next()) {
        const TString& key = it.key();

        mongo_v3::BsonType bsonType = it.type();
        switch (bsonType) {
            case mongo_v3::BT_DOUBLE:
                res[key] = it.getDouble();
                break;
            case mongo_v3::BT_INT64:
                res[key] = static_cast<ui64>(it.getI64());
                break;
            case mongo_v3::BT_INT32:
                res[key] = static_cast<ui32>(it.getI32());
                break;
            case mongo_v3::BT_STRING:
                res[key] = TString(it.getString());
                break;
            case mongo_v3::BT_OID:
                res[key] = NAnyValue::TOid(it.getOid().toStr());
                break;
            case mongo_v3::BT_DOCUMENT: {
                auto & val = (res[key] = NAnyValue::TMap()).AsMap();
                createHashMapFromDoc(it.getDocument(), val);
                break;
            }
            case mongo_v3::BT_ARRAY: {
                auto & subList = (res[key] = NAnyValue::TArray()).AsArray();
                const mongo_v3::DocumentArray& arr = it.getArray();
                subList.reserve(arr.size());
                for (const auto & v : arr) {
                    appendDocToList(subList, v);
                }
                break;
            }
            case mongo_v3::BT_BOOL:
                res[key] = it.getBool();
                break;
            case mongo_v3::BT_BINARY: {
                ythrow TWithBackTrace<yexception>() << "unsupported";
            }
            case mongo_v3::BT_UNKNOWN:

                break;
        }
    }
}

void createHashMapFromDoc(const mongo_v3::Document& doc, NAnyValue::TScalarMap & res) {
    mongo_v3::Document::Iterator it(doc);
    while (it.next()) {
        const TString& key = it.key();

        mongo_v3::BsonType bsonType = it.type();
        switch (bsonType) {
            case mongo_v3::BT_DOUBLE:
                res[key] = it.getDouble();
                break;
            case mongo_v3::BT_INT64:
                res[key] = static_cast<ui64>(it.getI64());
                break;
            case mongo_v3::BT_INT32:
                res[key] = static_cast<ui32>(it.getI32());
                break;
            case mongo_v3::BT_STRING:
                res[key] = TString(it.getString());
                break;
            case mongo_v3::BT_OID:
                res[key] = NAnyValue::TOid(it.getOid().toStr());
                break;
            case mongo_v3::BT_BOOL:
                res[key] = it.getBool();
                break;
            case mongo_v3::BT_DOCUMENT:
            case mongo_v3::BT_ARRAY:
            case mongo_v3::BT_BINARY: {
                ythrow TWithBackTrace<yexception>() << "unsupported";
            }
            case mongo_v3::BT_UNKNOWN:

                break;
        }
    }
}

NAnyValue::TMap createHashFromAction(const TQueryData& query) {
    NAnyValue::TMap hash;

    for (const auto &equal : query.equals) {
        NAnyValue::TAnyValue& any = hash[equal.first];
        if (any.GetType() != NAnyValue::TType::Map)
            any = NAnyValue::TMap();
        any.AsMap()["$eq"] = equal.second;
    }

    for (const auto &it : query.gt) {
        NAnyValue::TAnyValue& any = hash[it.first];
        if (any.GetType() != NAnyValue::TType::Map)
            any = NAnyValue::TMap();
        any.AsMap()["$gt"] = it.second;
    }

    for (const auto &it : query.lt) {
        NAnyValue::TAnyValue& any = hash[it.first];
        if (any.GetType() != NAnyValue::TType::Map)
            any = NAnyValue::TMap();
        any.AsMap()["$lt"] = it.second;
    }

    if (!query.in.empty()) {

        auto & orList = (hash["$or"] = NAnyValue::TArray()).AsArray();

        const auto size = query.in.cbegin()->second.size();

        for(size_t i : xrange(size)) {
            NAnyValue::TMap part;

            for(const auto & p : query.in)
                part[p.first] = p.second[i];

            orList.emplace_back(part);
        }
    }

    if (!query.ors.empty()) {

        if(!hash.contains("$or"))
            hash["$or"] = NAnyValue::TArray();

        auto & orList = hash["$or"].AsArray();

        for (const auto &it : query.ors) {
            orList.emplace_back(createHashFromAction(it));
        }
    }

    return hash;
}

mongo_v3::Document createQueryFromAction(const TQueryData& query) {
    return createDocFromHashMap(createHashFromAction(query));
}

mongo_v3::Document createUpdateFromAction(const TUpdateAction& action) {
    mongo_v3::Document update;

    if (!action.update.sets.empty())
        update.append("$set", createDocFromHashMap(action.update.sets));

    if (!action.update.incrs.empty())
        update.append("$inc", createDocFromHashMap(action.update.incrs));

    if (!action.update.setsOnInsert.empty())
        update.append("$setOnInsert", createDocFromHashMap(action.update.setsOnInsert));

    if (!action.update.ors.empty()) {
        mongo_v3::Document doc(createDocFromHashMap(action.update.ors));
        mongo_v3::Document bitDoc;

        for (mongo_v3::Document::Iterator it(doc.iter()); it.next();) {
            const TString& key = it.key();

            mongo_v3::BsonType bsonType = it.type();
            switch (bsonType) {
                case mongo_v3::BT_INT64:
                    bitDoc.append(key.c_str(), mongo_v3::Document("or", it.getI64()));
                    break;
                case mongo_v3::BT_INT32:
                    bitDoc.append(key.c_str(), mongo_v3::Document("or", it.getI32()));
                    break;
                case mongo_v3::BT_STRING:
                    bitDoc.append(key.c_str(), mongo_v3::Document("or", it.getDouble()));
                    break;
                case mongo_v3::BT_BOOL:
                    bitDoc.append(key.c_str(), mongo_v3::Document("or", it.getBool()));
                    break;
                default:
                    throw yexception() << "cannot use bit or for " << bsonType;
                    break;
            }
        }

        update.append("$bit", bitDoc);
    }

    return update;
}
