/*
 * mongoclient_v3_ut.cpp
 *
 *  Created on: 13 сент. 2016 г.
 *      Author: luckybug
 */

#include <library/cpp/testing/unittest/registar.h>
#include <mail/spamstop/tools/so-common/StorageBase.h>
#include <mail/spamstop/tools/so-common/StorageBaseUT.h>
#include "StorageMongo.h"
#include "mongoclient.h"

class MongoTest: public TTestBase {
    UNIT_TEST_SUITE(MongoTest);
    UNIT_TEST(TestBson);
    UNIT_TEST(TestUpsert);
    UNIT_TEST(TestFindAndModify);
    UNIT_TEST(TestRemove);
    UNIT_TEST(TestBulk);
    UNIT_TEST(TestArray);
    UNIT_TEST(TestBinary);
    UNIT_TEST_SUITE_END();

    mongo_v3::ClientPool pool;
    mongo_v3::Collection collection;

public:
    MongoTest()
        : pool((mongo_v3::URI("mongodb://localhost")))
        , collection(pool.getConnection()->getCollection("test_db", "test_collection"))
    {
        collection.remove(mongo_v3::Document());
    }

    void TestBson() {
        mongo_v3::Document doc("ID", 42);
        doc.append(
            "value", mongo_v3::Document().append("f", 666.5).append("b", "str").append("c", (i64)123).append("d", 456));

        std::pair<bool, mongo_v3::Document::Iterator> it = doc.find("value");

        UNIT_ASSERT_EQUAL(it.first, true);

        mongo_v3::Document innerDoc(it.second.getDocument());

        it = innerDoc.find("f");
        UNIT_ASSERT_EQUAL(it.first, true);
        UNIT_ASSERT_DOUBLES_EQUAL(it.second.getDouble(), 666.5, 1e-6);

        it = innerDoc.find("b");
        UNIT_ASSERT_EQUAL(it.first, true);
        UNIT_ASSERT_STRINGS_EQUAL(it.second.getString(), "str");

        it = innerDoc.find("c");
        UNIT_ASSERT_EQUAL(it.first, true);
        UNIT_ASSERT_EQUAL(it.second.getI64(), (i64)123);

        it = innerDoc.find("d");
        UNIT_ASSERT_EQUAL(it.first, true);
        UNIT_ASSERT_EQUAL(it.second.getI32(), 456);
    }

    void TestUpsert() {
        collection.remove(mongo_v3::Document());
        UNIT_ASSERT_EQUAL(collection.size(), 0);

        collection.update(
            mongo_v3::Document("ID", 42),
            mongo_v3::Document("$set",
                               mongo_v3::Document("val", 666)),
            false);
        UNIT_ASSERT_EQUAL(collection.size(), 0);

        collection.update(
            mongo_v3::Document("ID", 42),
            mongo_v3::Document("$set",
                               mongo_v3::Document("val", 666)),
            true);
        UNIT_ASSERT_EQUAL(collection.size(), 1);
    }

    void TestFindAndModify() {
        collection.update(
            mongo_v3::Document("ID", 42),
            mongo_v3::Document("$set",
                               mongo_v3::Document("val", 666)),
            true);

        const mongo_v3::Document& beforeFAM = collection.findAndModify(
            mongo_v3::Document("ID", 42),
            mongo_v3::Document("$set",
                               mongo_v3::Document("val", 456)),
            true,
            false);

        std::pair<bool, mongo_v3::Document::Iterator> it = beforeFAM.find("val");

        UNIT_ASSERT_EQUAL(it.first, true);
        UNIT_ASSERT_EQUAL(it.second.getI32(), 666);

        mongo_v3::Cursor cursor = collection.find(mongo_v3::Document("ID", 42), 1);

        mongo_v3::Document afterFAM;
        UNIT_ASSERT_EQUAL(cursor.nextDoc(afterFAM), true);
        it = afterFAM.find("val");

        UNIT_ASSERT_EQUAL(it.first, true);
        UNIT_ASSERT_EQUAL(it.second.getI32(), 456);
    }

    void TestRemove() {
        mongo_v3::Document query("ID", 42);
        collection.remove(query);

        mongo_v3::Cursor cursor = collection.find(query, 1);

        mongo_v3::Document res;
        UNIT_ASSERT_EQUAL(cursor.nextDoc(res), false);
    }

    void TestBulk() {
        collection.remove(mongo_v3::Document());
        UNIT_ASSERT_EQUAL(collection.size(), 0);

        mongo_v3::Bulk bulk(collection);

        bulk
            .update(
                mongo_v3::Document("ID", 42),
                mongo_v3::Document("$set",
                                   mongo_v3::Document("val", 666)))
            .update(
                mongo_v3::Document("ID", 43),
                mongo_v3::Document("$set",
                                   mongo_v3::Document("val", 667)))
            .exec();

        UNIT_ASSERT_EQUAL(collection.size(), 2);

        {
            mongo_v3::Cursor cursor = collection.find(
                mongo_v3::Document("ID", 42), 1);

            mongo_v3::Document dbVar;
            UNIT_ASSERT_EQUAL(cursor.nextDoc(dbVar), true);

            std::pair<bool, mongo_v3::Document::Iterator> it = dbVar.find("val");

            UNIT_ASSERT_EQUAL(it.first, true);
            UNIT_ASSERT_EQUAL(it.second.getI32(), 666);
        }

        {
            mongo_v3::Cursor cursor = collection.find(
                mongo_v3::Document("ID", 43), 1);

            mongo_v3::Document dbVar;
            UNIT_ASSERT_EQUAL(cursor.nextDoc(dbVar), true);

            std::pair<bool, mongo_v3::Document::Iterator> it = dbVar.find("val");

            UNIT_ASSERT_EQUAL(it.first, true);
            UNIT_ASSERT_EQUAL(it.second.getI32(), 667);
        }
    }

    void TestArray() {
        {
            mongo_v3::DocumentArray array;

            array.push_back(2.0);
            array.push_back(4.3);

            collection.update(
                mongo_v3::Document("ID", 42),
                mongo_v3::Document("$set",
                                   mongo_v3::Document("array", array)),
                true);
        }

        {
            mongo_v3::Cursor cursor = collection.find(mongo_v3::Document("ID", 42), 1);

            mongo_v3::Document doc;
            UNIT_ASSERT_EQUAL(cursor.nextDoc(doc), true);

            std::pair<bool, mongo_v3::Document::Iterator> res = doc.find("array");
            UNIT_ASSERT_EQUAL(res.first, true);

            mongo_v3::DocumentArray arr = res.second.getArray();

            TVector<double> temp;
            temp.reserve(arr.size());

            for (mongo_v3::DocumentArray::ArrayIterator it = arr.begin(); it != arr.end(); ++it) {
                temp.push_back((*it).getDouble());
            }

            UNIT_ASSERT_EQUAL(arr.size(), 2);
            UNIT_ASSERT_EQUAL(arr.at<double>(0), 2.0);
            UNIT_ASSERT_EQUAL(arr.at<double>(1), 4.3);
        }
    }

    void TestBinary() {
        TVector<ui64> data;

        data.push_back(42);
        data.push_back(666);
        {
            mongo_v3::BinaryData binaryData(&data[0], data.size() * sizeof(ui64));

            collection.update(
                mongo_v3::Document("ID", 42),
                mongo_v3::Document("$set",
                                   mongo_v3::Document("binaryData", binaryData)),
                true);
        }

        {
            mongo_v3::Cursor cursor = collection.find(mongo_v3::Document("ID", 42), 1);

            mongo_v3::Document doc;
            UNIT_ASSERT_EQUAL(cursor.nextDoc(doc), true);

            std::pair<bool, mongo_v3::Document::Iterator> res = doc.find("binaryData");
            UNIT_ASSERT_EQUAL(res.first, true);

            const mongo_v3::BinaryData& binaryData = res.second.getBinary();

            UNIT_ASSERT_EQUAL(data.size() * sizeof(ui64), binaryData.size());

            UNIT_ASSERT_EQUAL(memcmp(&data[0], binaryData.data(), binaryData.size()), 0);
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(MongoTest);

class MongoTestAbstract: public StorageBaseTest {
    mongo_v3::TStorageMongo db;

public:
    MongoTestAbstract()
        : StorageBaseTest(db, "test_collection")
        , db("test_db")
    {
        db.Connect("mongodb://localhost");
    }
};

UNIT_TEST_SUITE_REGISTRATION(MongoTestAbstract);
