#include <library/cpp/logger/log.h>
#include <library/cpp/config/config.h>
#include <util/generic/yexception.h>
#include <util/string/split.h>
#include <util/generic/set.h>
#include <util/generic/iterator_range.h>
#include <util/system/env.h>
#include <util/string/subst.h>

#include <algorithm>
#include <utility>
#include <util/string/join.h>
#include "PostgreBase.h"
#include "pq/Result.h"
#include "pq/Work.h"

static IOutputStream & operator<<(IOutputStream & stream, const NConfig::TConfig & config) {
    stream << '\n';
    config.DumpJson(stream);
    return stream;
}

namespace sql {
    void ResultToHashMap(const TResult::TLine& line, NAnyValue::TScalarMap & hash);
    void ResultToHashMaps(const TResult& res, TFindResults & hashes);
    void queryBindAnyval(Query& query, const NAnyValue::TScalar& val);

    NAnyValue::TScalarMap getUpsertActions(const TUpdateData &update);
    NAnyValue::TScalarMap getUdateActions(const TUpdateData& update);
    NAnyValue::TScalarMap getEqualActions(const TQueryData& query);

    struct TAnyScalarSQLView {
        const NAnyValue::TScalar& val;

        explicit TAnyScalarSQLView(const NAnyValue::TScalar& val)
                : val(val)
        {
        }
        friend IOutputStream& operator<<(IOutputStream& stream, const TAnyScalarSQLView& view);
    };

    struct TIndexSQLView {
        size_t index;

        explicit TIndexSQLView(size_t index)
                : index(index)
        {
        }
        friend IOutputStream& operator<<(IOutputStream& stream, const TIndexSQLView& view);
    };

    TPostgreBase::TPostgreBase(TAtomicSharedPtr<IThreadPool> pool)
            : TPostgreBase(TPoolParams(32, TDuration::Seconds(1)), TPoolParams(32, TDuration::Seconds(1)), std::move(pool))
    { };

    TPostgreBase::TPostgreBase(const TPoolParams& masterSettings, const TPoolParams& replicaSettings, TAtomicSharedPtr<IThreadPool> pool, const TAtomicSharedPtr<TLog>& logger)
            : waitQueue(std::move(pool))
            , masterPool(MakeHolder<TPGPool>(masterSettings))
            , replicaPool(MakeHolder<TPGPool>(replicaSettings))
            , logger(logger? logger: new TLog)
            , readTimeout(TDuration::MilliSeconds(100))
            , connectTimeout(TDuration::MilliSeconds(100))
            , scheduler(*waitQueue)
    {}

    void TPostgreBase::Connect(const TString& uri) {
        TVector<TString> conStrings;
        Split(uri, ",", conStrings);

        roSettings.SetConnectionString(conStrings);
        wSettings.SetConnectionString(conStrings);

        scheduler.Add(([this]() {
//            (*logger) << ELogPriority::TLOG_INFO << HistPrinter(replicaPool.MoveHist()) << Endl;
            wSettings.GetTraits()->UpdateTraits();
            roSettings.GetTraits()->UpdateTraits();
        }), checkROPeriod, "check read only");
    }

    void TPostgreBase::Connect(const NConfig::TConfig & config) {

        TString passTemplate, pass;

        if(config.Has("ro_check_period")) {
            const auto & periodConfig = config["ro_check_period"];
            if (!periodConfig.IsA<TString>())
                ythrow TWithBackTrace<yexception>() << "ro_check_period field must be string: " << periodConfig;

            checkROPeriod = TDuration::Parse(periodConfig.Get<TString>());
        }

        if (config.Has("lock_timeout")) {
            const auto & timeout = config["lock_timeout"];
            if (!timeout.IsNumeric())
                ythrow TWithBackTrace<yexception>() << "lock_timeout field must be numeric: " << timeout;

            lockTimeout = timeout.As<size_t>();
        }

        if (config.Has("update_chunk_size")) {
            const auto & size = config["update_chunk_size"];
            if (!size.IsNumeric())
                ythrow TWithBackTrace<yexception>() << "lock_timeout field must be numeric: " << size;

            if (size.As<size_t>() != 0)
                updateChunkSize = size.As<size_t>();
        }

        if(config.Has("pass_template")) {
            if(!config["pass_template"].IsA<TString>())
                ythrow TWithBackTrace<yexception>() << "pass_template field must be string: " << config;

            passTemplate = config["pass_template"].As<TString>();
            pass = GetEnv(passTemplate);
            if(pass.empty())
                ythrow TWithBackTrace<yexception>() << passTemplate << " must specify not empty password";
        }

        if(config.Has("master")) {

            const auto & masterConf = config["master"];
            if(!masterConf.IsA<NConfig::TDict>()) {
                ythrow TWithBackTrace<yexception>() << "master field must be dict: " << masterConf;
            }

            if(masterConf.Has("pool")) {
                masterPool = MakeHolder<TPGPool>(TPoolParams(masterConf["pool"]));
            }

        }

        if(config.Has("replica")) {
            const auto & replicaConf = config["replica"];
            if(!replicaConf.IsA<NConfig::TDict>()) {
                ythrow TWithBackTrace<yexception>() << "replica field must be dict: " << replicaConf;
            }

            if(replicaConf.Has("pool")) {
                replicaPool = MakeHolder<TPGPool>(TPoolParams(replicaConf["pool"]));
            }

        }

        if(config.Has("read_timeout")) {
            const auto & timeoutConf = config["read_timeout"];
            if(timeoutConf.IsNumeric())
                readTimeout = TDuration::MilliSeconds(timeoutConf.As<size_t>());
            else if(timeoutConf.IsA<TString>()) {
                if(!TDuration::TryParse(timeoutConf.Get<TString>(), readTimeout))
                    ythrow TWithBackTrace<yexception>() << "cannot parse duration from : " << timeoutConf;
            } else
                ythrow TWithBackTrace<yexception>() << "read_timeout field must be numeric or string: " << timeoutConf;
        }

        if(config.Has("connect_timeout")) {
            const auto & timeoutConf = config["connect_timeout"];
            if(timeoutConf.IsNumeric())
                connectTimeout = TDuration::MilliSeconds(timeoutConf.As<size_t>());
            else if(timeoutConf.IsA<TString>()) {
                if(!TDuration::TryParse(timeoutConf.Get<TString>(), connectTimeout))
                    ythrow TWithBackTrace<yexception>() << "cannot parse duration from : " << timeoutConf;
            } else
                ythrow TWithBackTrace<yexception>() << "connectTimeout field must be numeric or string: " << timeoutConf;
        }

        if(config.Has("uri")) {
            const auto localConf = config["uri"];
            if(!localConf.IsA<TString>()) {
                ythrow TWithBackTrace<yexception>() << "uri field must be string: " << localConf;
            }

            TString uri = localConf.Get<TString>();

            if(!passTemplate.empty()) {
                SubstGlobal(uri, passTemplate, pass);
            }
            Connect(uri);
        } else {
            ythrow TWithBackTrace<yexception>() << "config must contain uri field: " << config;
        }

        Cerr << "connect to pg with " << readTimeout << " timeout" << Endl;
    }

    size_t TPostgreBase::Count(
            const TString& collectionName,
            const TFindAction& action) {
        Query query;

        const auto end = Now() + readTimeout;
        query.getStream() << "select count(*) from " << collectionName;

        if (!action.query.empty()) {
            query.getStream() << " where ";
            CreateFindQuery(query, action.query);
        }
        query.getStream() << ';';

        auto holder = GetReplicaConnection(end);

        Work work(*holder);
        auto res = work.exec(query, end, true);

        if (!res)
            ythrow TInterfaceError(res.error);

        NAnyValue::TScalarMap count;
        ResultToHashMap(res.res[0], count);

        return count["count"].AsUi64();
    }

    size_t TPostgreBase::Remove(
            const TString& collectionName,
            const TFindAction& action) {
        Query query;

        if (lockTimeout.Defined())
            query.getStream() << "set lock_timeout = " << *lockTimeout << ';';

        query.getStream() << "delete from " << collectionName;

        if (!action.query.empty()) {
            query.getStream() << " where ";
            CreateFindQuery(query, action.query, 1, false);
        }
        query.getStream() << ';';

        return ExecModifyQuery(query, false);
    }

    size_t TPostgreBase::Update(
            const TString& collectionName,
            const TUpdateAction& action) {
        return action.upsert?
               Upsert(collectionName, action, true):
               JustUpdate(collectionName, action, true);
    }

    void TPostgreBase::UpdateSeries(
            const TString& collectionName,
            const TActionSeries & actions,
            bool /*ordered*/) {

        auto begin = actions.cbegin(), end = actions.cend();

        if (!updateChunkSize.Defined())
            return UpdateSeriesImpl(collectionName, begin, end);

        while (begin != end) {
            auto chunk_end = std::min(end, std::next(begin, *updateChunkSize));
            UpdateSeriesImpl(collectionName, begin, chunk_end);
            begin = chunk_end;
        }
    }

    void TPostgreBase::UpdateSeriesImpl(
            const TString& collectionName,
            TActionSeries::const_iterator begin,
            TActionSeries::const_iterator end) {

        //Cerr << "UpdateSeries" << Endl;
        Query query;
        if (lockTimeout.Defined())
            query.getStream() << "set lock_timeout = " << *lockTimeout << ';';

        for (auto actionIt = begin; actionIt != end; ++actionIt) {
            const TUpdateAction& action = *actionIt;

            if (action.upsert) {
                FillUpsertQuery(collectionName, action, query);
            } else {
                FillUpdateQuery(collectionName, action, query);
            }
        }

//        Cdbg << query.getStream().Str() << Endl;

        ExecModifyQuery(query, false);
    }

    void TPostgreBase::UpdateBulk(
            const TString& collectionName,
            const TUpdateAction& action) {
        if (action.upsert)
            Upsert(collectionName, action, false);
        else
            JustUpdate(collectionName, action, false);
    }

    void TPostgreBase::Find(
            const TString& collectionName,
            const TFindAction& action,
            TFindResults& result) {
        Query query;

        const auto end = Now() + readTimeout;
        query.getStream() << "select * from " << collectionName;

        if (!action.query.empty()) {
            query.getStream() << " where ";
            CreateFindQuery(query, action.query);
        }
        query.getStream() << ';';

        auto holder = GetReplicaConnection(end);

        Work work(*holder);
        auto res = work.exec(query, end, true);

        if (!res)
            ythrow TInterfaceError(res.error, res.getErrorString());

        ResultToHashMaps(res.res, result);
    }

    void TPostgreBase::FindOne(
            const TString& collectionName,
            const TFindAction& action,
            NAnyValue::TScalarMap & result) {
        Query query;

        const auto end = Now() + readTimeout;
        query.getStream() << "select * from " << collectionName;

        if (!action.query.empty()) {
            query.getStream() << " where ";
            CreateFindQuery(query, action.query);
        }
        query.getStream() << " limit 1;";

        auto holder = GetReplicaConnection(end);
        Work work(*holder);
        auto res = work.exec(query, end, true);

        if (!res)
            ythrow TInterfaceError(res.error);

        return ResultToHashMap(res.res[0], result);
    }

    template<Connection::InitStrategy strategy> class TFuture : public TPostgreBase::IFuture{
    public:
        TFindResults Get(TCont* cont) override {
            TConnectionHolder connect;

            NProf::Profiler profiler;

            try {
                profiler.Prof("pool").Start();
                if (!pool.get(settings, connect, cont))
                    throw TInterfaceError(TInterfaceError::PoolTimeout);
                profiler.Prof("pool").Stop();

                Connection &instance = connect.Get();
                profiler.Prof("init").Start();

                if (!instance.ok()) {
                    instance.reset();
                    const auto &res = instance.Init<strategy>(cont, connectTimeout.ToDeadLine());
                    if (!res)
                        throw TInterfaceError(res.error.type, res.getErrorString());
                }

                profiler.Prof("init").Stop();

                const auto localDeadline = readTimeout.ToDeadLine();

                if (!connect.Get().ok())
                    throw TInterfaceError(TInterfaceError::ConnectError, instance.Info());


                profiler.Prof("send").Start();
                instance.SendQuery(query, binary);
                profiler.Prof("send").Stop();

                profiler.Prof("flush").Start();
                instance.Flush(cont, localDeadline);
                profiler.Prof("flush").Stop();

                profiler.Prof("getres").Start();
                TResult res(instance.GetResult(localDeadline, cont));
                profiler.Prof("getres").Stop();

                connect = {};
//                Cdbg << profiler.ToLog() << Endl;

                TFindResults results;
                ResultToHashMaps(res, results);
                return results;
            } catch (...) {
                if(connect.Defined())
                    connect.Get().reset();
                std::rethrow_exception(std::current_exception());
            }
        }

        explicit TFuture(
                Query query,
                bool binary,
                TInstant deadline,
                TPGPool & pool,
                const PGSettings & settings,
                TDuration readTimeout,
                TDuration connectTimeout)
                : query(std::move(query)),
                  binary(binary),
                  deadline(deadline),
                  pool(pool),
                  settings(settings),
                  readTimeout(readTimeout),
                  connectTimeout(connectTimeout) {}
    private:
        Query query;
        bool binary;
        TInstant deadline;
        TPGPool & pool;
        const PGSettings & settings;
        TDuration readTimeout, connectTimeout;
    };

    THolder<TPostgreBase::IFuture> TPostgreBase::FindNonblock(
            TInstant deadline,
            const TString& collectionName,
            const TFindAction& action) {

        Query query;
        query.getStream() << "select * from " << collectionName;

        if (!action.query.empty()) {
            query.getStream() << " where ";
            CreateFindQuery(query, action.query);
        }
        query.getStream() << ';';

        return MakeHolder<TFuture<Connection::RandomReplica>>(
                std::move(query), true, deadline, *replicaPool, roSettings, readTimeout, connectTimeout
        );
    }


    void TPostgreBase::FillUpdateQuery(
            const TString& collectionName,
            const TUpdateAction& action,
            Query& query) {
        query.getStream() << "update " << collectionName << " as target set ";

        CreateUpdateQuery(query, action, "change");

        const auto & eqFields = getEqualActions(action.query);
        const auto & updateActions = getUdateActions(action.update);

        query.getStream() << " from (values (";

        {
            const char* sep = "";
            for (const auto &eqField : eqFields) {
                if (updateActions.end() == updateActions.find(eqField.first)) {
                    query.getStream() << sep << TAnyScalarSQLView(eqField.second);
                    sep = ",";
                }


                query.getStream() << sep << TAnyScalarSQLView(eqField.second);
                sep = ",";
            }

            for (const auto &updateAction : updateActions) {
                query.getStream() << sep << TAnyScalarSQLView(updateAction.second);
                sep = ",";
            }
        }

        query.getStream() << ")) as change(";
        {
            const char* sep = "";
            for (const auto &eqField : eqFields) {

                if (updateActions.end() == updateActions.find(eqField.first)) {
                    query.getStream() << sep << eqField.first;
                    sep = ",";
                }

                query.getStream() << sep << "search_" << eqField.first;
                sep = ",";
            }
            for (const auto &updateAction : updateActions) {
                query.getStream() << sep << updateAction.first;
                sep = ",";
            }
        }

        query.getStream() << ") where (";

        CreateFindQueryTemplate(query, action.query, "change.search_", "target.");

        query.getStream() << ");";
    }

    void TPostgreBase::CreateFindQueryTemplate(
            Query& query,
            const TQueryData& queryData,
            const TString& target,
            const TString& change) {
        const char* sep = "";
        for (const auto &equal : queryData.equals) {
            query.getStream() << sep << change << equal.first << '=' << target << equal.first;
            sep = " and ";
        }
        for (const auto &it : queryData.lt) {
            query.getStream() << sep << change << it.first << '<' << target << it.first;
            sep = " and ";
        }
        for (const auto &it : queryData.gt) {
            query.getStream() << sep << change << it.first << '>' << target << it.first;
            sep = " and ";
        }

        if (!queryData.ors.empty()) {
            query.getStream() << sep << '(';
            const char* orSep = "";
            for (const auto &it : queryData.ors) {
                query.getStream() << orSep << '(';
                CreateFindQueryTemplate(query, it, target, change);
                query.getStream() << ')';
                orSep = " or ";
            }
            query.getStream() << ')';
        }
    }

    size_t TPostgreBase::JustUpdate(
            const TString& collectionName,
            const TUpdateAction& action,
            bool binary) {
        Query query;

        FillUpdateQuery(collectionName, action, query);

//            Cout << query.getStream().Str() << Endl;

        return ExecModifyQuery(query, binary);
    }

    void TPostgreBase::FillUpsertQuery(
            const TString& collectionName,
            const TUpdateAction& action,
            Query& query) {
        query.getStream() << "insert into " << collectionName << " as target (";

        const auto & eqFields = getEqualActions(action.query);
        const auto & upsertActions = getUpsertActions(action.update);

        {
            const char* sep = "";
            for (const auto &eqField : eqFields) {
                query.getStream() << sep << eqField.first;
                sep = ",";
            }
            for (const auto &upsertAction : upsertActions) {
                if (eqFields.end() == eqFields.find(upsertAction.first)) {
                    query.getStream() << sep << upsertAction.first;
                    sep = ",";
                }
            }
        }
        query.getStream() << ") values (";
        {
            const char* sep = "";
            for (const auto &eqField : eqFields) {
                query.getStream() << sep << TAnyScalarSQLView(eqField.second);
                sep = ",";
            }
            for (const auto &upsertAction : upsertActions) {
                if (eqFields.end() == eqFields.find(upsertAction.first)) {
                    query.getStream() << sep << TAnyScalarSQLView(upsertAction.second);
                    sep = ",";
                }
            }
        }
        query.getStream() << ")";

        {
            query.getStream() << " on conflict(";

            const char* sep = "";
            for (const auto &eqField : eqFields) {
                const TString& field = eqField.first;
                query.getStream() << sep << field;
                sep = ",";
            }
            query.getStream() << ')';
        }

        query.getStream() << " do update set ";

        {
            CreateUpdateQuery(query, action, "excluded");
            query.getStream() << " where (";

            CreateFindQueryTemplate(query, action.query, "excluded.", "target.");

            query.getStream() << ");";
        }

        query.getStream() << ';';
    }

    size_t TPostgreBase::Upsert(
            const TString& collectionName,
            const TUpdateAction& action,
            bool binary) {

        Query query;
        FillUpsertQuery(collectionName, action, query);

//            Cout << query.getStream().Str() << Endl;

        return ExecModifyQuery(query, binary);
    }

    size_t TPostgreBase::ExecModifyQuery(const Query& query, bool binary) {
        TConnectionHolder connectionHolder = GetMasterConnection();

        const auto & res = Work(connectionHolder.Get()).exec(query, binary);

        if (!res) {
            connectionHolder.Reset();
            *logger << TLOG_DEBUG << "exec failed. error: " << res.error.what() << "\nrequest: " << query.getStream().Str();
            ythrow TInterfaceError(res.error);
        }

        return res.res.AffectedLinesCount();
    }

    class TOutputStreamVisitor {
    public:
        explicit TOutputStreamVisitor(IOutputStream& stream) : stream(stream) {};

        IOutputStream& operator()(const TNoType&) {
            return stream;
        }

        IOutputStream& operator()(const bool& value) {
            return stream << (value? "true": "false");
        }

        IOutputStream& operator()(const TString& value) {
            return stream << '\'' << value << '\'';
        }

        IOutputStream& operator()(const NAnyValue::TOid& oid) {
            return stream << '\'' << oid.get() << '\'';
        }

        template<typename T>
        IOutputStream& operator()(const T& value) {
            return stream << static_cast<NAnyValue::make_signed_t<T>>(value);
        }

    private:
        IOutputStream& stream;
    };

    IOutputStream& operator<<(IOutputStream& stream, const TAnyScalarSQLView& view) {
        TOutputStreamVisitor visitor(stream);
        return view.val.Visit(visitor);
    }

    IOutputStream& operator<<(IOutputStream& stream, const TIndexSQLView& view) {
        stream << '$' << view.index;

        return stream;
    }

    void AddValue(const NAnyValue::TScalar& value, Query& resultQuery, size_t& templateNum, bool binary) {
        if (binary) {
            resultQuery.getStream() << TIndexSQLView(templateNum++);
            queryBindAnyval(resultQuery, value);
        } else {
            resultQuery.getStream() << value;
        }
    }

    void CreateFindQuerySection(Query& resultQuery, const NAnyValue::TScalarMap& query, char action, size_t& templateNum, TStringBuf& separator, bool binary) {
        auto & s = resultQuery.getStream();
        for (const auto &item : query) {
            const TString& key = item.first;
            const auto & value = item.second;

            s << separator << key << action;
            AddValue(value, resultQuery, templateNum, binary);
            separator = " and ";
        }
    }

    size_t TPostgreBase::CreateFindQuery(Query& resultQuery, const TQueryData& query, size_t templateNum, bool binary) {
        TStringBuf separator;
        CreateFindQuerySection(resultQuery, query.equals, '=', templateNum, separator, binary);
        CreateFindQuerySection(resultQuery, query.gt, '>', templateNum, separator, binary);
        CreateFindQuerySection(resultQuery, query.lt, '<', templateNum, separator, binary);

        auto & s = resultQuery.getStream();
        if(!query.in.empty()) {
            s << separator << '(';
            separator = " and ";
            const char *inSep = "";

            const size_t maxSize = query.in.cbegin()->second.size();
            for (const auto &in : query.in) {
                s << inSep << in.first;
                inSep = ",";
            }

            s << ") in (values ";

            inSep = "";
            for (size_t i = 0; i < maxSize; i++) {

                s << inSep << '(';
                inSep = ",";

                const char *valSep = "";
                for (const auto &in : query.in) {
                    s << valSep;
                    AddValue(in.second[i], resultQuery, templateNum, binary);

                    valSep = ",";
                }

                s << ')';
            }

            s << ')';
        }

        if (!query.ors.empty()) {
            s << separator << '(';
            const char* orSep = "";
            for (const auto &it : query.ors) {
                s << orSep << '(';
                templateNum = CreateFindQuery(resultQuery, it, templateNum, binary);
                s << ')';
                orSep = " or ";
            }
            s << ')';
        }

        return templateNum;
    }

    void TPostgreBase::CreateUpdateQuery(Query& query, const TUpdateAction& updateAction, const TString& updateTableName) {
        const char* sep = "";
        auto & s = query.getStream();
        for (const auto &m : updateAction.update.fieldsMax) {
            s << sep << m.first << "=GREATEST(target." << m.second.first << ",target." << m.second.second << ") ";
            sep = ",";
        }
        for (const auto &set : updateAction.update.sets) {
            s << sep << set.first << '=' << updateTableName << '.' << set.first << ' ';
            sep = ",";
        }
        for (const auto &incr : updateAction.update.incrs) {
            s << sep << incr.first << '=' << updateTableName << '.' << incr.first << '+' << "target." << incr.first << ' ';
            sep = ",";
        }
        for (const auto &it : updateAction.update.ors) {
            s << sep << it.first << '=' << updateTableName << '.' << it.first << '|' << "target." << it.first << ' ';
            sep = ",";
        }

        for (auto it = updateAction.query.equals.begin(); it != updateAction.query.equals.end(); ++it) {
            if (updateAction.update.sets.contains(it->first) || updateAction.update.setsOnInsert.contains(it->first))
                continue;
            s << sep << it->first << '=' << updateTableName << '.' << it->first << ' ';
            sep = ",";
        }
    }

    void ResultToHashMap(const TResult::TLine& line, NAnyValue::TScalarMap & hash) {

        for (size_t i = 0; i < line.Size(); i++) {
            const auto & cell = line[i];

            const auto & fieldName = cell.GetName();
            switch (cell.GetType()) {
                case T_int8:
                    hash[fieldName] = cell.As<ui64>();
                    break;
                case T_int4:
                    hash[fieldName] = cell.As<ui32>();
                    break;
                case T_int2:
                    hash[fieldName] = cell.As<ui16>();
                    break;
                case T_float4:
                    hash[fieldName] = static_cast<double>(cell.As<float>());
                    break;
                case T_float8:
                    hash[fieldName] = cell.As<double>();
                    break;
                case T_text:
                case T_varchar:
                    hash[fieldName] = cell.As<TString>();
                    break;
                case T_date:
                    hash[fieldName] = cell.As<TInstant>().Seconds();
                    break;
                case T_bool:
                    hash[fieldName] = cell.As<ui8>() != 0;
                    break;
                default:
                    ythrow yexception() << "unsupported type " << static_cast<int>(cell.GetType());
            }
        }
    }

    void ResultToHashMaps(const TResult& res, TFindResults & hashes) {
        if (!res.Size())
            return;

        hashes.reserve(res.Size());
        for (size_t line = 0; line < res.Size(); line++) {
            ResultToHashMap(res[line], hashes.emplace_back());
        }
    }

    class TQueryVisitor {
    public:
        TQueryVisitor(Query& query) : query(query) {};

        void operator()(TNoType) {
            ythrow TWithBackTrace<yexception>() << "notype";
        }

        void operator()(const NAnyValue::TOid& oid) {
            return (*this)(oid.get());
        }

        template<typename T>
        void operator()(const T& value) {
            query.bind<NAnyValue::make_signed_t<T>>(value);
        }

    private:
        Query& query;
    };

    void queryBindAnyval(Query& query, const NAnyValue::TScalar& val) {
        TQueryVisitor visitor(query);
        val.Visit(visitor);
    }

    NAnyValue::TScalarMap getUpsertActions(const TUpdateData &update) {
        NAnyValue::TScalarMap upsertFields;

        upsertFields.insert(update.incrs.begin(), update.incrs.end());
        upsertFields.insert(update.ors.begin(), update.ors.end());
        upsertFields.insert(update.setsOnInsert.begin(), update.setsOnInsert.end());
        upsertFields.insert(update.sets.begin(), update.sets.end());

        return upsertFields;
    }

    NAnyValue::TScalarMap getUdateActions(const TUpdateData& update) {
        NAnyValue::TScalarMap upsertFields;

        upsertFields.insert(update.incrs.begin(), update.incrs.end());
        upsertFields.insert(update.ors.begin(), update.ors.end());
        upsertFields.insert(update.sets.begin(), update.sets.end());

        return upsertFields;
    }

    NAnyValue::TScalarMap getEqualActions(const TQueryData& query) {
        NAnyValue::TScalarMap searchFields;

        searchFields.insert(query.equals.begin(), query.equals.end());
        searchFields.insert(query.gt.begin(), query.gt.end());
        searchFields.insert(query.lt.begin(), query.lt.end());
        for(const auto & o : query.ors) {
            for(const auto & v : getEqualActions(o)) {
                searchFields.emplace(v);
            }
        }

        return searchFields;
    }
} /* namespace sql */
