#pragma once

#include <util/system/rwlock.h>
#include <util/thread/pool.h>
#include <mail/so/spamstop/tools/so-common/StorageBase.h>
#include <mail/so/libs/scheduler/scheduler.h>
#include "settings.h"

class TLog;

namespace sql {
    class TPostgreBase: public TStorageBase {
    public:
        void SetReadTimeout(const TDuration& timeout) override {
            readTimeout = timeout;
        };

        TDuration GetReadTimeout() const override {
            return readTimeout;
        };

        void SetWriteTimeout(const TDuration& /*timeout*/) override {
            ythrow yexception() << "not implemented";
        };

        TDuration GetWriteTimeout() const override {
            ythrow yexception() << "not implemented";
        };

        size_t Count(
                const TString& collectionName,
                const TFindAction& action) override;

        size_t Remove(
                const TString& collectionName,
                const TFindAction& action) override;

        size_t Update(
                const TString& collectionName,
                const TUpdateAction& action) override;

        void UpdateSeries(
                const TString& collectionName,
                const TActionSeries & actions,
                bool ordered) override;

        void UpdateBulk(
                const TString& collectionName,
                const TUpdateAction& action) override;

        void Find(
                const TString& collectionName,
                const TFindAction& action,
                TFindResults& result) override;

        void FindOne(
                const TString& collectionName,
                const TFindAction& action,
                NAnyValue::TScalarMap & result) override;

        THolder<IFuture> FindNonblock(
                TInstant deadline,
                const TString& collectionName,
                const TFindAction& action) override ;

        void Connect(const TString& uri) override;
        void Connect(const NConfig::TConfig & config) override;

        void SetCheckingROPeriod(TDuration period) {
            checkROPeriod = period;
        }

        explicit TPostgreBase(TAtomicSharedPtr<IThreadPool> pool);
        TPostgreBase(const TPoolParams& masterSettings, const TPoolParams& replicaSettings, TAtomicSharedPtr<IThreadPool> pool, const TAtomicSharedPtr<TLog>& logger = nullptr);

        ~TPostgreBase() override = default;

    private:
        size_t JustUpdate(
                const TString& collectionName,
                const TUpdateAction& action,
                bool binary);

        size_t Upsert(
                const TString& collectionName,
                const TUpdateAction& action,
                bool binary);

        void UpdateSeriesImpl(
                const TString& collectionName,
                TActionSeries::const_iterator begin,
                TActionSeries::const_iterator end);

        size_t ExecModifyQuery(const Query& query, bool binary);

        static void FillUpdateQuery(
                const TString& collectionName,
                const TUpdateAction& action,
                Query& query);

        static void FillUpsertQuery(
                const TString& collectionName,
                const TUpdateAction& action,
                Query& query);

        static void CreateFindQueryTemplate(
                Query& query,
                const TQueryData& queryData,
                const TString& target,
                const TString& change);

        static size_t CreateFindQuery(
                Query& resultQuery,
                const TQueryData& query,
                size_t templateNum = 1,
                bool binary = true);

        static void CreateUpdateQuery(
                Query& query,
                const TUpdateAction& updateAction,
                const TString& updateTableName);

        template<Connection::InitStrategy strategy> TConnectionHolder GetConnection(TPGPool & pool, const PGSettings & settings, TInstant deadline) {
            TConnectionHolder connect;

            if (!pool.get(settings, connect))
                ythrow TInterfaceError(TInterfaceError::PoolTimeout);

            auto & instance = connect.Get();

            if (!instance.ok()) {
                instance.reset();
                const auto & res = instance.Init<strategy>(deadline);
                if (!res)
                    ythrow TInterfaceError(res.error.type, res.getErrorString());
            }

            if (!connect.Get().ok())
                ythrow TInterfaceError(TInterfaceError::ConnectError, instance.Info());

            return connect;
        }

        TConnectionHolder GetReplicaConnection(TInstant deadline) {
            return GetConnection<Connection::InitStrategy::RandomReplica>(*replicaPool, roSettings, deadline);
        }

        TConnectionHolder GetMasterConnection(TInstant deadline = TInstant::Max()) {
            return GetConnection<Connection::InitStrategy::Master>(*masterPool, wSettings, deadline);
        }
    private:
        TAtomicSharedPtr<IThreadPool> waitQueue;
        THolder<TPGPool> masterPool, replicaPool;
        PGSettings wSettings, roSettings;
        TAtomicSharedPtr<TLog> logger;
        TDuration readTimeout, connectTimeout;
        TMaybe<size_t> lockTimeout;
        TMaybe<size_t> updateChunkSize;
        TDuration checkROPeriod = TDuration::Minutes(2);
        TSimpleScheduler scheduler;

    };

} /* namespace sql */
