/*
 * interface.cpp
 *
 *  Created on: 06.11.2015
 *      Author: luckybug
 */

#include "interface.h"

#include <iostream>
#include <functional>

#include <util/datetime/base.h>
#include <util/system/tls.h>
#include <util/system/thread.h>
#include <util/system/defaults.h>

#include <mail/so/libs/scheduler/scheduler.h>

#include "pq/Error.h"
#include "pq/Query.h"
#include "pq/Result.h"
#include "pq/Work.h"
#include "pq/Connection.h"
#include "pq/QueryGenerator.h"
#include "settings.h"

namespace sql {
    namespace {
        static PGSettings settings;
        int checkingReplicaPeriodS = -1;

        static auto waitQueue = MakeAtomicShared<TAdaptiveThreadPool>();
        static THolder<TSimpleScheduler> scheduler;

        static const TString defaults[COUNT] = {
                "DEFAULT 0",
                "DEFAULT 0",
                "DEFAULT 0",
                "DEFAULT 0.0",
                "DEFAULT ''",
        };

        typedef char static_assertion_defaults_size[(sizeof(defaults) / sizeof(defaults[0]) == COUNT) ? 1 : -1];

        static const TString types[COUNT] = {
                "int8",
                "int4",
                "int2",
                "double precision",
                "text",
        };

        static auto masterPool = MakeHolder<TPGPool>(TPoolParams());
        static auto replicaPool = MakeHolder<TPGPool>(TPoolParams());

        typedef char static_assertion_types_size[(sizeof(types) / sizeof(types[0]) == COUNT) ? 1 : -1];
    }

    Field::Field(Type type, const TString& name, bool primary)
            : type(type)
            , name(name)
            , primary(primary)
    {
        pgView = name + " " + typeToPGType(type) + " " + primaryPG(primary) + " " + defaultValue(type);
    }

    const TString& Field::primaryPG(bool primary) {
        static const TString primaryView("primary key not null");
        static const TString empty("");
        return primary ? primaryView : empty;
    }
    const TString& Field::defaultValue(Type type) {
        if (type >= COUNT)
            ythrow yexception() << "Unknown type";

        return defaults[type];
    }
    const TString& Field::typeToPGType(Type type) {
        if (type >= COUNT)
            ythrow yexception() << "Unknown type";

        return types[type];
    }

    const TString& Field::toString() const {
        //"ID INT PRIMARY KEY     NOT NULL,"
        return pgView;
    }

    template <sql::Connection::InitStrategy strategy>
    TResWithError<void> reconnect(sql::Connection& connection, const TInstant& mustEndUntil) {
        if (Now() > mustEndUntil)
            return TResWithError<void>(TInterfaceError(TInterfaceError::TotalTimeout));

        static const size_t maxAttempt = 100;
        size_t attempts = 0;

        CProf prof;

        prof.Prof("checkconok").Start();
        bool ok = connection.ok();
        prof.Prof("checkconok").Stop();
        if (ok)
            return TResWithError<void>();

        prof.Prof("initconn").Start();
        TResWithError<void> res = connection.Init<strategy>(mustEndUntil);
        while (!connection.ok() && attempts++ < maxAttempt) {
            if (Now() > mustEndUntil)
                return TResWithError<void>(TInterfaceError(TInterfaceError::TotalTimeout));

            Sleep(TDuration::MilliSeconds(10));
        }
        prof.Prof("checkconok").Start();
        if (!connection.ok()) {
            prof.Prof("checkconok").Stop();
            return res;
        }
        prof.Prof("checkconok").Stop();
        return TResWithError<void>(std::move(prof));
    }

    template <sql::Connection::InitStrategy strategy>
    TVoidWithError checkStrategy(sql::Connection& connection, const TInstant& mustEndUntil) {
        Y_STATIC_THREAD(TInstant)
                lastTimeCheck;

        const TInstant now = Now();

        if (checkingReplicaPeriodS > 0 && (long)(now - lastTimeCheck).Seconds() > checkingReplicaPeriodS) {
            const TResWithError<bool>& readOnly = connection.isReplica();
            if (!readOnly)
                return readOnly.error;

            if (!Connection::goodMode<strategy>(readOnly.res)) {
                TVoidWithError res = reconnect<strategy>(connection, mustEndUntil);
                if (!res)
                    return res;
            }
            lastTimeCheck = now;
        }
        return TVoidWithError();
    }

    template <sql::Connection::InitStrategy strategy>
    TResWithError<TConnectionHolder> GetConnection(TPGPool& pool, const TInstant& mustEndUntil) {
        TConnectionHolder holder;

        CProf prof;

        {
            TInstant now = Now();
            if (now >= mustEndUntil)
                return TResWithError<TConnectionHolder>(TInterfaceError(TInterfaceError::TotalTimeout));

            CProfItemGuard g(prof.Prof("getfrompool"));
            if (mustEndUntil == TInstant::Max()) {
                if (!pool.get(settings, holder))
                    return TResWithError<TConnectionHolder>(TInterfaceError(TInterfaceError::PoolTimeout));
            } else {
                if (!pool.get(settings, holder, mustEndUntil))
                    return TResWithError<TConnectionHolder>(TInterfaceError(TInterfaceError::PoolTimeout));
            }
        }

        sql::Connection& connection = holder.Get();

        prof.Prof("reconnect").Start();
        TResWithError<void> res = reconnect<strategy>(connection, mustEndUntil);
        prof.Prof("reconnect").Stop();

        prof.SetSub("reconnect", std::move(res.prof));
        if (!res)
            return res.error;

        if (settings.GetConnectionString().size() > 1) {
            prof.Prof("checkStrategy").Start();
            TVoidWithError r = checkStrategy<strategy>(connection, mustEndUntil);
            prof.Prof("checkStrategy").Stop();
            prof.SetSub("reconnect", std::move(r.prof));
            if (!r)
                return r.error;
        }

        return ResWithError(std::move(holder), std::move(prof));
    }

    TResWithError<TConnectionHolder> GetReplicaConnection(TInstant mustEndUntil) {
        return GetConnection<sql::Connection::RandomReplica>(*replicaPool, mustEndUntil);
    }

    TResWithError<TConnectionHolder> GetMasterConnection(TInstant mustEndUntil) {
        return GetConnection<sql::Connection::Master>(*masterPool, mustEndUntil);
    }

    std::string makeUri(
            const char* db,
            const char* user,
            const char* pwd,
            const char* hostname,
            size_t port) {
        TStringStream ss;

        ss << "dbname=" << db
           << " user=" << user
           << " host=" << hostname
           << " port=" << port;

        if (pwd && strlen(pwd)) {
            ss << " password=" << pwd;
        }

        return ss.Str();
    }

    TPoolParams PGMasterPoolParams() {
        return masterPool->getParams();
    }

    TPoolParams PGReplicaPoolParams() {
        return replicaPool->getParams();
    }

    void PGInitPool(TPoolParams masterParams, TPoolParams replicaParams) {
//        static const TDuration levels[] = {
//            TDuration::MilliSeconds(10),
//            TDuration::MilliSeconds(20),
//            TDuration::MilliSeconds(50),
//            TDuration::MilliSeconds(100),
//            TDuration::MilliSeconds(150),
//            TDuration::MilliSeconds(190),
//            TDuration::MilliSeconds(500),
//        };
//        static const size_t levelsCount = sizeof(levels) / sizeof(levels[0]);
//        if (masterParams.histLevels.empty()) {
//            std::copy(levels, levels + levelsCount, std::back_inserter(masterParams.histLevels));
//        }
//        if (replicaParams.histLevels.empty()) {
//            std::copy(levels, levels + levelsCount, std::back_inserter(replicaParams.histLevels));
//        }

        MakeHolder<TPGPool>(TPoolParams(masterParams)).Swap(masterPool);
        MakeHolder<TPGPool>(TPoolParams(replicaParams)).Swap(replicaPool);
    }

    TResWithError<bool> PGInit(const TVector<TString>& connStrings, const TDuration& execTimeout) {
        TVector<TString> cons;
        waitQueue->Start(0, 0);
        scheduler = MakeHolder<TSimpleScheduler>(*waitQueue);

        CopyIf(connStrings.cbegin(), connStrings.cend(), std::back_inserter(cons), [](const TString& host){
            return bool(host);
        });

        Cdbg << "PGinit try connect to: " << JoinStrings(cons, ",") << Endl;

        if (cons.empty())
            return TResWithError<bool>(TInterfaceError(TInterfaceError::ConnectError, "hosts empty"));

        settings.SetConnectionString(cons);

        Cdbg << "PGinit found connects: " << Endl;
        for(const auto & trait : settings.GetTraits()->GetTraits()) {
            Cdbg << trait.second << Endl;
        }

        settings.setExecTimeout(execTimeout);

        if(scheduler)
            scheduler->Add(([traits = settings.GetTraits()]() {
                traits->UpdateTraits();
            }), TDuration::Seconds(15), "check read only");

        return ResWithError(true);
    }

    TResWithError<bool> PGInit(
            const char* db,
            const char* user,
            const char* pwd,
            const char* hostname,
            size_t port,
            const TDuration& execTimeout) {
        TVector<TString> conn;
        conn.push_back(makeUri(db, user, pwd, hostname, port).c_str());

        return PGInit(conn, execTimeout);
    }

    TResWithError<bool> PGInitWOPass(
            const char* db,
            const char* user,
            const char* hostname,
            size_t port,
            const TDuration& execTimeout) {
        return PGInit(db, user, NULL, hostname, port, execTimeout);
    }

    void PGSetPeriodForCheckingReplicas(int seconds) {
        checkingReplicaPeriodS = seconds;
    }

    TResWithError<void> PGCreateIndex(const char* indexName, const char* table, const TVector<TString>& fields) {
        try {
            TResWithError<TConnectionHolder> connHolder = GetMasterConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();
            sql::QueryGenerator gen;

            gen("create unique index if not exists ")(indexName)(" on ")(table)('(');

            for (size_t i = 0; i < fields.size(); i++) {
                if (i > 0)
                    gen(',');
                gen(fields[i].c_str());
            }
            gen(");");

            sql::Work work(*conn);

            auto res = work.exec(gen);

            if (res)
                return TResWithError<void>();

            return res.error;
        } catch (const std::exception& e) {
            return TResWithError<void>(TInterfaceError(TInterfaceError::Unknown, e.what()));
        }
    }

    TResWithError<void> PGCheckConnection() {
        try {
            TResWithError<TConnectionHolder> res = GetMasterConnection();
            if (res)
                return TResWithError<void>();

            TStringBuf err = res.error.AsStrBuf();

            const char* src = err.cbegin();
            const char* srcEnd = err.cend();

            static const char passTag[] = "password=";

            const char* start = strstr(src, passTag);

            if (!start)
                return TInterfaceError(res.error.type, TString{err});

            start += sizeof(passTag) - 1;

            const char* end = strchr(start, ' ');

            if (!end)
                end = srcEnd;

            return TInterfaceError(
                    TInterfaceError::Unknown,
                    TString(src, start) + "***" + TString(end, srcEnd));

        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> PGCreateTable(const char* name, const table_t& table) {
        //"CREATE TABLE COMPANY();"

        try {
            TResWithError<TConnectionHolder> connHolder = GetMasterConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();
            sql::QueryGenerator gen;

            gen.assign("CREATE TABLE IF NOT EXISTS ");
            gen.assign(name);
            gen.assign('(');

            table_t::const_iterator it = table.begin();
            gen.assign(it->second.toString().c_str());
            ++it;
            for (; it != table.end(); ++it) {
                gen.assign(',');
                gen.assign(it->second.toString().c_str());
            }
            gen.assign(");");

            sql::Work work(*conn);

            work.exec(gen);

            return TResWithError<void>();
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> runQuery(sql::QueryGenerator& gen, bool timedExec) {
        try {
            if (!gen.length())
                return TResWithError<void>();
            CProf prof;

            const TInstant mustEndUntil = Now() + settings.GetTimeout() + masterPool->getParams().timeout;
            TResWithError<TConnectionHolder> connHolder = GetMasterConnection(mustEndUntil);
            prof.SetSub("GetMasterConnection", std::move(connHolder.prof));
            if (!connHolder)
                return connHolder.error;

            prof.Prof(ProfKeys::exec).Start();

            sql::Work work(connHolder.res.Get());
            auto res = timedExec ? work.exec(gen, mustEndUntil) : work.exec(gen);
            prof.Prof(ProfKeys::exec).Stop();

            if (!res) {
                res.prof.Extend(std::move(prof));
                return res.error;
            }

            return TResWithError<void>(std::move(prof));
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> PGContiniusUpdate(
            const char* table,
            table_t& tableScheme,
            ui64 shingle,
            const nosql::HashMap& incrs,
            const nosql::HashMap& sets,
            sql::QueryGenerator& generator) {
        try {
            if (tableScheme.empty())
                return TResWithError<void>(TInterfaceError(TInterfaceError::ExecError, "Empty scheme"));

            generator.setFirst();

            generator("insert into ")(table)(" (id) select ")(static_cast<i64>(shingle))(" where not exists(select id from ")(table)(" where id=")(static_cast<i64>(shingle))(");");

            generator("UPDATE ")(table)(" set ");

            for (const auto &incr : incrs) {
                const TString& name = incr.first;
                const nosql::AnyValue& val = incr.second;

                switch (val.Type()) {
                    case nosql::AnyValue::INTEGER:
                        generator.assignIncr(name.c_str(), i32(val.Integer()));
                        break;
                    case nosql::AnyValue::INTEGER64:
                        generator.assignIncr(name.c_str(), i64(val.Long()));
                        break;
                    case nosql::AnyValue::DOUBLE:
                        generator.assignIncr(name.c_str(), val.Double());
                        break;
                    case nosql::AnyValue::STRING:
                        generator.assignIncr(name.c_str(), val.String().c_str());
                        break;
                    default:
                        break;
                }
            }
            for (const auto &set : sets) {
                const TString& name = set.first;
                const nosql::AnyValue& val = set.second;

                switch (val.Type()) {
                    case nosql::AnyValue::INTEGER:
                        generator.assignSet(name.c_str(), i32(val.Integer()));
                        break;
                    case nosql::AnyValue::INTEGER64:
                        generator.assignSet(name.c_str(), i64(val.Long()));
                        break;
                    case nosql::AnyValue::DOUBLE:
                        generator.assignSet(name.c_str(), val.Double());
                        break;
                    case nosql::AnyValue::STRING:
                        generator.assignSet(name.c_str(), val.String().c_str());
                        break;
                    default:
                        break;
                }
            }
            generator(" WHERE id=")(i64(shingle))(';');

            return TResWithError<void>();
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> PGUpdate(
            const char* table,
            table_t& tableScheme,
            ui64 shingle,
            const nosql::HashMap& incrs,
            const nosql::HashMap& sets,
            bool timedExec) {
        //PGInsert(table, shingle);
        try {
            sql::QueryGenerator gen;

            CProf prof;

            CProfItemGuard totalG(prof.Prof(ProfKeys::total));

            prof.Prof(ProfKeys::createQuery).Start();
            {
                TResWithError<void> res = PGContiniusUpdate(table, tableScheme, shingle, incrs, sets, gen);
                prof.Prof(ProfKeys::createQuery).Stop();
                if (!res) {
                    res.prof.Extend(std::move(prof));
                    return res;
                }
            }
            prof.Prof(ProfKeys::runQuery).Start();
            TResWithError<void> res = runQuery(gen, timedExec);
            prof.Prof(ProfKeys::runQuery).Stop();

            res.prof.Extend(std::move(prof));
            return res;
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    void line2hash_binary(nosql::HashMap& resultHash,
                          table_t& tableScheme,
                          const sql::TResult::TLine & line) {

        for(const auto & p : tableScheme) {
            const auto & name = p.first;
            const auto & field = p.second;

            nosql::AnyValue val;
            if(line.Has(name) && !line[name].IsNull()) {
                const auto & cell = line[name];

                switch (field.type) {
                    case INT64:
                        val = cell.As<i64>();
                        break;
                    case INT32:
                        val = cell.As<i32>();
                        break;
                    case INT16:
                        val = cell.As<i16>();
                        break;
                    case DOUBLE:
                        val = cell.As<double>();
                        break;
                    case TEXT:
                        val = cell.As<TString>();
                        break;
                    case COUNT:break;
                }
            } else {
                switch (field.type) {
                    case INT64:
                        val = i64{};
                        break;
                    case INT32:
                        val = i32{};
                        break;
                    case INT16:
                        val = i16{};
                        break;
                    case DOUBLE:
                        val = double{};
                        break;
                    case TEXT:
                        val = "";
                        break;
                    case COUNT:break;
                }
            }
            resultHash[name] = std::move(val);
        }
    }

    void line2hash(nosql::HashMap& resultHash,
                   table_t& tableScheme,
                   const sql::TResult::TLine & line) {

        for(const auto & p : tableScheme) {
            const auto & name = p.first;
            const auto & field = p.second;

            nosql::AnyValue val;
            if(line.Has(name) && !line[name].IsNull()) {
                const auto & cell = line[name];
                const auto & data = cell.As<TString>();

                switch (field.type) {
                    case INT64:
                        val = FromString<i64>(data);
                        break;
                    case INT32:
                        val = FromString<i32>(data);
                        break;
                    case INT16:
                        val = FromString<i16>(data);
                        break;
                    case DOUBLE:
                        val = FromString<double>(data);
                        break;
                    case TEXT:
                        val = data;
                        break;
                    case COUNT:break;
                }
            } else {
                switch (field.type) {
                    case INT64:
                        val = i64{};
                        break;
                    case INT32:
                        val = i32{};
                        break;
                    case INT16:
                        val = i16{};
                        break;
                    case DOUBLE:
                        val = double{};
                        break;
                    case TEXT:
                        val = "";
                        break;
                    case COUNT:break;
                }
            }
            resultHash[name] = std::move(val);
        }
    }

    TResWithError<void> PGFindOne(
            const char* table,
            table_t& tableScheme,
            ui64 shingle,
            nosql::HashMap& resultHash,
            const TString& idField,
            bool timedExec) {
        TMap<ui64, nosql::HashMap*> resultHashes;
        resultHashes[shingle] = &resultHash;
        return PGFindByIDs(table, tableScheme, resultHashes, idField, timedExec);
    }

    TResWithError<void> PGFindByIDs(
            const char* table,
            table_t& tableScheme,
            TMap<ui64, nosql::HashMap*>& resultHashes,
            const TString& idField,
            bool timedExec) {
        const size_t shinglesCount = resultHashes.size();
        const TInstant mustEndUntil = Now() + settings.GetTimeout() + replicaPool->getParams().timeout;

        CProf prof;

        if (shinglesCount == 0)
            return TResWithError<void>(std::move(prof));

        try {
            CProfItemGuard totalG(prof.Prof(ProfKeys::total));

            prof.Prof(ProfKeys::getConnect).Start();
            TResWithError<TConnectionHolder> connHolder = GetReplicaConnection(mustEndUntil);
            prof.Prof(ProfKeys::getConnect).Stop();

            prof.SetSub("GetReplicaConnection", std::move(connHolder.prof));

            if (!connHolder) {
                return connHolder.error;
            }

            prof.Prof(ProfKeys::createQuery).Start();
            sql::QueryGenerator gen;
            gen("SELECT * FROM ")(table)(" WHERE ")(idField)("=any($1);");

            TVector<i64> shingles(Reserve(resultHashes.size()));

            for (const auto &resultHashe : resultHashes) {
                shingles.push_back(resultHashe.first);
            }

            sql::Query query = gen.createQuery();
            query.bindArray(&shingles[0], shingles.size());
            prof.Prof(ProfKeys::createQuery).Stop();

            prof.Prof(ProfKeys::exec).Start();

            sql::Work work(connHolder.res.Get());
            auto res = timedExec ? work.exec(query, mustEndUntil, true) : work.exec(query, true);

            prof.Prof(ProfKeys::exec).Stop();

            if (!res) {
                res.prof.Extend(std::move(prof));
                return res.error;
            }

            CProfItemGuard parseProf(prof.Prof(ProfKeys::parseRes));

            const sql::TResult& sqlRes = res.res;

            for (size_t y = 0; y < sqlRes.Size(); y++) {
                nosql::HashMap hash;
                line2hash_binary(hash, tableScheme, sqlRes[y]);

                auto idIt = hash.find(idField);
                if (idIt == hash.end()) {
                    const TString message = "cannot find id field " + idField;
                    return TResWithError<void>(
                            TInterfaceError(
                                    TInterfaceError::ParsingResult,
                                    message),
                            std::move(prof));
                }
                auto hashIt = resultHashes.find(idIt->second.Long());

                if (hashIt == resultHashes.end() || !hashIt->second) {
                    TString message = "cannot find " + UI64ToStroka(idIt->second.Long()) + " in requested hashes";
                    return TResWithError<void>(
                            TInterfaceError(
                                    TInterfaceError::ParsingResult,
                                    message),
                            std::move(prof));
                }

                *hashIt->second = hash;
            }

            return TResWithError<void>(std::move(prof));
        } catch (const std::exception& e) {
            return TResWithError<void>(TInterfaceError(TInterfaceError::Unknown, e.what()), std::move(prof));
        }
    }

    TResWithError<void> PGFind(
            const char* table,
            table_t& tableScheme,
            TVector<nosql::HashMap>& resultHashes,
            bool timedExec,
            int limit) {
        try {
            const TInstant mustEndUntil = Now() + settings.GetTimeout() + replicaPool->getParams().timeout;
            TResWithError<TConnectionHolder> connHolder = GetReplicaConnection(mustEndUntil);
            if (!connHolder)
                return connHolder.error;


            sql::QueryGenerator gen;

            gen("SELECT * FROM ")(table)(" LIMIT ")(limit)(';');

            sql::Work work(connHolder.res.Get());
            auto res = timedExec ? work.exec(gen, mustEndUntil) : work.exec(gen);

            if (!res)
                return res.error;

            resultHashes.resize(res.res.Size());
            for (size_t i = 0; i < res.res.Size(); i++) {
                line2hash(resultHashes[i], tableScheme, res.res[i]);
            }

            return TResWithError<void>();
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> PGMultiErase(
            const char* table,
            const TVector<i64>& shingles) {
        try {
            if (shingles.empty())
                return TResWithError<void>();

            TResWithError<TConnectionHolder> connHolder = GetMasterConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();

            sql::QueryGenerator gen;

            gen("DELETE FROM ")(table)(" WHERE ID = any ($1);");
            sql::Query query = gen.createQuery();
            query.bindArray(&shingles[0], shingles.size());

            sql::Work work(*conn);
            work.exec(query, true);

            return TResWithError<void>();
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> PGErase(
            const char* table,
            ui64 shingle) {
        try {
            TResWithError<TConnectionHolder> connHolder = GetMasterConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();
            sql::QueryGenerator gen;

            gen("delete from ")(table)(" where id=")(i64(shingle))(';');

            sql::Work work(*conn);
            work.exec(gen.createQuery());

            return TResWithError<void>();
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<size_t> PGTableSizeApprox(const char* table) {
        try {
            TResWithError<TConnectionHolder> connHolder = GetReplicaConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();
            sql::QueryGenerator gen;
            size_t count_v = 0;

            gen("SELECT reltuples AS approximate_row_count FROM pg_class WHERE relname = '")(table)("';");

            sql::Work work(*conn);
            auto res = work.exec(gen);
            if (!res)
                return res.error;

            if ((res.res.Size() > 0) && (res.res[0].Size() > 0)) {
                TString txt = res.res[0][0].As<TString>();
                double cnt_v = 0;

                if (sscanf(txt.c_str(), "%lf", &cnt_v) == 0)
                    cnt_v = 0;
                count_v = static_cast<ui32>(cnt_v);
                //count_v = std::atoi(results[0][0]);
            }

            return TResWithError<size_t>(count_v);
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<size_t> PGTableSize(const char* table) {
        try {
            TResWithError<TConnectionHolder> connHolder = GetReplicaConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();
            sql::QueryGenerator gen;

            gen("SELECT COUNT(*) as size FROM ")(table)(';');

            sql::Work work(*conn);
            auto res = work.exec(gen.createQuery());
            if (!res)
                return res.error;

            if (res.res.Size() == 0 || res.res[0].Size() == 0) {
                return TInterfaceError(TInterfaceError::Unknown, "zero result");
            }

            return ResWithError(FromString<size_t>(res.res[0][0].As<TString>()));
        } catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

    TResWithError<void> PGTableDrop(const char* table) {
        try {
            TResWithError<TConnectionHolder> connHolder = GetMasterConnection();
            if (!connHolder)
                return connHolder.error;
            sql::Connection* conn = &connHolder.res.Get();
            sql::QueryGenerator gen;

            gen("DROP TABLE ")(table)(';');

            sql::Work work(*conn);
            TResWithError<sql::TResult> res = work.exec(gen.createQuery());
            if (!res)
                return res.error;

            return TResWithError<void>();
        }
        catch (const std::exception& e) {
            return TInterfaceError(TInterfaceError::Unknown, e.what());
        }
    }

} //namespace sql
