/*
 * interface_ut.cpp
 *
 *  Created on: 9 мар. 2017 г.
 *      Author: luckybug
 */

#include <library/cpp/testing/unittest/registar.h>
#include "interface.h"
#include "PostgreBase.h"
#include <mail/so/spamstop/tools/so-common/StorageBaseUT.h>

#define ADDKEY(table, name, type) table[name] = sql::Field(type, name, true)
#define ADDTOTABLE(table, name, type) table[name] = sql::Field(type, name)

class PGTest: public TTestBase {
    UNIT_TEST_SUITE(PGTest);
    UNIT_TEST(TestConnection);
    UNIT_TEST(CreateTable);
    UNIT_TEST(TestTimeout);
    UNIT_TEST(TestInsert);
    UNIT_TEST(TestInterface);
    UNIT_TEST(TestRunQuery);
    UNIT_TEST_SUITE_END();

    sql::table_t table;
    static const size_t totalConnections = 5;
    sql::TPostgreBase db;

public:
    PGTest()
        : db(TPoolParams(totalConnections, TDuration::MilliSeconds(1000)), TPoolParams(totalConnections, TDuration::MilliSeconds(1000)))
    {
        sql::PGInit("testdb", "testuser", "123", "127.0.0.1", 5432, TDuration::MilliSeconds(100));
        sql::PGInitPool(TPoolParams(totalConnections, TDuration::MilliSeconds(100)), TPoolParams(totalConnections, TDuration::MilliSeconds(100)));
    }

    void TestConnection() {
        sql::TResWithError<void> res = sql::PGCheckConnection();
        if (!res)
            //Cerr << res.getErrorString() << Endl;
        UNIT_ASSERT_EQUAL(res, true);

        db.Connect("dbname=testdb host=127.0.0.1 port=5432 user=testuser password=123");
    }

    void CreateTable() {
        ADDKEY(table, "id", sql::INT64);
        ADDTOTABLE(table, "text", sql::TEXT);
        ADDTOTABLE(table, "i32", sql::INT32);
        ADDTOTABLE(table, "i16", sql::INT16);
        ADDTOTABLE(table, "double", sql::DOUBLE);

        sql::TResWithError<void> res = sql::PGCreateTable("test_table", table);
        if (!res)
            //Cerr << res.getErrorString() << Endl;
        UNIT_ASSERT_EQUAL(res, true);
    }

    void TestTimeout() {
        {
            TVector<sql::TConnectionHolder> connections;

            for (size_t i = 0; i < totalConnections; i++) {
                const sql::TResWithError<sql::TConnectionHolder>& res = sql::GetMasterConnection();
                UNIT_ASSERT_EQUAL(res, true);
                connections.push_back(res.res);
            }

            {
                const sql::TResWithError<sql::TConnectionHolder>& res = sql::GetMasterConnection();
                UNIT_ASSERT_EQUAL(res, false);
                UNIT_ASSERT_EQUAL(res.error.type, sql::TInterfaceError::PoolTimeout);
            }
            {
                const sql::TResWithError<sql::TConnectionHolder>& res = sql::GetMasterConnection();
                UNIT_ASSERT_EQUAL(res, false);
                UNIT_ASSERT_EQUAL(res.error.type, sql::TInterfaceError::PoolTimeout);
            }
        }

        const sql::TResWithError<sql::TConnectionHolder>& res = sql::GetMasterConnection();
        UNIT_ASSERT_EQUAL(res, true);
    }

    void TestInsert() {
        {
            nosql::HashMap sets, incrs;
            sets["i32"] = 48;
            const sql::TVoidWithError& res = sql::PGUpdate("test_table", table, 496898652, incrs, sets, false);
            if (!res)
                Cout << res.error.ToLog() << Endl;
            UNIT_ASSERT_EQUAL(res, true);
        }

        {
            nosql::HashMap sets, incrs;
            sets["text"] = "some another text";
            const sql::TVoidWithError& res = sql::PGUpdate("test_table", table, 496898652, incrs, sets, true);
            if (!res)
                Cout << res.error.ToLog() << Endl;
            UNIT_ASSERT_EQUAL(res, true);
        }
    }

    void TestInterface() {
        nosql::HashMap resultHashMap[2];
        TMap<ui64, nosql::HashMap*> resultHashes;

        resultHashes[4] = &resultHashMap[0];
        resultHashes[496898652] = &resultHashMap[1];

        sql::TResWithError<void> res;

        for (int i = 0; i < 1; i++) {
            res = sql::PGFindByIDs("test_table", table, resultHashes, "id", true);
            if (!res)
                Cout << res.getErrorString() << Endl;
            else {
                TString str, err;
                Cout << HashMapToStrokaBase(resultHashMap[0]) << Endl;
                Cout << HashMapToStrokaBase(resultHashMap[1]) << Endl;
            }
        }

        for (int i = 0; i < 1; i++) {
            res = sql::PGFindOne("test_table", table, 4, resultHashMap[0], "id", true);
            if (!res)
                Cout << res.getErrorString() << Endl;
            else {
                TString str, err;
                Cout << HashMapToStrokaBase(resultHashMap[0]) << Endl;
                Cout << HashMapToStrokaBase(resultHashMap[1]) << Endl;
            }
        }
        for (int i = 0; i < 1; i++) {
            TVector<nosql::HashMap> results;
            res = sql::PGFind("test_table", table, results, true);
            if (!res)
                Cout << res.getErrorString() << Endl;
            else {
                for (int k = 0; k < results.size(); k++)
                    Cout << HashMapToStrokaBase(results[k]) << Endl;
            }
        }

        {
            TVector<nosql::HashMap> hashes(2);
            TMap<ui64, nosql::HashMap*> results;
            results[1] = &hashes[0];
            results[3] = &hashes[1];

            sql::PGFindByIDs("test_table", table, results, "id", true);

            Cout << "PGFindByIDs" << Endl;
            Cout << HashMapToStrokaBase(hashes[0]) << Endl;
            Cout << HashMapToStrokaBase(hashes[1]) << Endl;
        }
    }

    void TestRunQuery() {
        sql::TResWithError<void> res;

        for (int i = 0; i < 1; i++) {
            sql::Query query("select * from test_table where id=$1;");
            query.bind<i64>(4);

            sql::TResWithError<sql::TConnectionHolder> connHolder = sql::GetMasterConnection();

            sql::TResWithError<TSharedPtr<sql::Result>> res = sql::Work(connHolder.res->get()).exec(query, true);

            Cout << int(bool(res)) << ' ' << res.getProfString() << ' ' << res.getErrorString() << Endl;

            if (res.res->linesCount() > 0) {
                for (size_t а = 0; а < res.res->fieldsCount(); а++) {
                    const TString& field = res.res->getFieldName(а);
                    switch (res.res->getType(а)) {
                        case sql::T_int8:
                            Cout << res.res->get<ui64>(0, а) << Endl;
                            break;
                        case sql::T_int4:
                            Cout << res.res->get<ui32>(0, а) << Endl;
                            break;
                        case sql::T_int2:
                            Cout << res.res->get<ui16>(0, а) << Endl;
                            break;
                        case sql::T_float8:
                            Cout << res.res->get<double>(0, а) << Endl;
                            break;
                        case sql::T_text:
                            Cout << res.res->get(0, а) << Endl;
                            break;
                    }
                }
            }
        }

        for (int i = 0; i < 1; i++) {
            sql::Query query("update test_table set i16=23 where id=4;;");

            sql::TResWithError<sql::TConnectionHolder> connHolder = sql::GetMasterConnection();

            const sql::TResWithError<TSharedPtr<sql::Result>>& res = sql::Work(connHolder.res->get()).exec(query, true);

            Cout << int(bool(res)) << ' ' << res.getProfString() << ' ' << res.getErrorString() << Endl;
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(PGTest);

class PGTestAbstract: public StorageBaseTest {
    sql::TPostgreBase db;

public:
    PGTestAbstract()
        : StorageBaseTest(db, "test_table")
        , db(TPoolParams(32, TDuration::MilliSeconds(1000)), TPoolParams(32, TDuration::MilliSeconds(1000)))
    {
        db.Connect("dbname=testdb host=127.0.0.1 port=5432 user=testuser password=123");
    }
};

UNIT_TEST_SUITE_REGISTRATION(PGTestAbstract);
