#pragma once

#include <util/generic/string.h>
#include <util/generic/maybe.h>
#include <util/datetime/base.h>
#include <util/network/poller.h>
#include <util/system/rwlock.h>
#include <mail/so/spamstop/tools/so-common/ares.h>
#include <ostream>
#include "Error.h"
#include "Result.h"
#include "connection_holder.h"

class TCont;
namespace sql {
    struct TConnectionString{
    public:
        TConnectionString & SetDb(TString v) {
            db.ConstructInPlace(std::move(v));
            return *this;
        }
        TConnectionString & SetUser(TString v) {
            user.ConstructInPlace(std::move(v));
            return *this;
        }
        TConnectionString & SetPass(TString v) {
            pass.ConstructInPlace(std::move(v));
            return *this;
        }
        TConnectionString & SetHost(TString v) {
            host.ConstructInPlace(std::move(v));
            return *this;
        }
        TConnectionString & SetPort(ui32 v) {
            port.ConstructInPlace(v);
            return *this;
        }

        const TMaybe<TString> & GetDb() const { return db; }
        const TMaybe<TString> & GetUser() const { return user; }
        const TMaybe<TString> & GetPass() const { return pass; }
        const TMaybe<TString> & GetHost() const { return host; }
        const TMaybe<ui32>    & GetPort() const { return port; }

        static TConnectionString Parse(const TString & s);

        enum class ESafe{Yes, No};
        TString MakeRawConnectionString(ESafe safe = ESafe::No) const;

    private:
        TMaybe<TString> db, user, pass, host;
        TMaybe<ui32> port;
    };

    struct TConnectionTRaits {
        TConnectionString connectionString;
        TString host;
        bool ok = false;
        bool isReplica = false;

        friend IOutputStream &operator<<(IOutputStream &os, const TConnectionTRaits &raits) {
            os << "connectionString: " << raits.connectionString.MakeRawConnectionString(TConnectionString::ESafe::Yes) << " host: " << raits.host << " ok: " << raits.ok
               << " isReplica: " << raits.isReplica;
            return os;
        }

        TConnectionTRaits() = default;
        TConnectionTRaits(const TConnectionTRaits&) = default;
        TConnectionTRaits & operator=(const TConnectionTRaits&) = default;
        TConnectionTRaits(TConnectionString connectionString, TString host, bool ok, bool isReplica)
                : connectionString(std::move(connectionString)), host(std::move(host)), ok(ok), isReplica(isReplica) {}
    };

    struct TConnectionsTRaits {
        void UpdateTraits();

        void AddAndCheck(const TString & connectionString);

        TConnectionTRaits GetTraits(const TString & connectionString) const;
        THashMap<size_t, TConnectionTRaits> GetTraits() const;

        TConnectionsTRaits();
    private:
        static std::tuple<bool, bool> Check(const TString & connectionString);
    private:
        THashMap<size_t, TConnectionTRaits> connectionsTraits;
        NAres::TAres ares;
        mutable TRWMutex traitsMutex;
    };

    class Connection : TMoveOnly {
    public:
        enum InitStrategy {
            RandomReplica,
            Master
        };

        void SendQuery(const Query& query, bool binary);

        TPgResultHolder ExecQuery(const Query& query, bool binary);

        void Flush();
        void Flush(TInstant deadline);
        void Flush(TCont *cont, TInstant deadline);

        TPgResultHolder GetResult(TInstant deadline);
        TPgResultHolder GetResult(TInstant deadline, TCont* cont);

        void setExecTimeout(const TDuration& timeout);
        TDuration getExecTimeout() const;

        TString Info() const;

        bool ok() const;

        void reset();

        TResWithError<bool> isReplica();

        TInstant GetLastCheckRO() const { return lastROCheck; }

        template <InitStrategy strategy> static bool goodMode(bool ro) {
            return (strategy == sql::Connection::Master) ? !ro : ro;
        }

        template <InitStrategy strategy> bool CheckReadOnly() {
            const auto & readOnly = isReplica();
            return readOnly && goodMode<strategy>(readOnly.res);
        }

        template <InitStrategy strategy> TVoidWithError Init(TInstant deadline);
        template <InitStrategy strategy> TVoidWithError Init(TCont * cont, TInstant deadline);

        TVoidWithError setNonBlockng(bool nonblocking = true);

        void SetConnectionTraits(TAtomicSharedPtr<TConnectionsTRaits> newTraits) {
            traits = std::move(newTraits);
        }

        Connection() = default;

        Connection(Connection&&) noexcept = default;
        Connection &operator=(Connection&&) noexcept = default;

        explicit Connection(const TString & connectionString);

        explicit Connection(const std::vector<std::string>& _hosts);

        explicit Connection(const TVector<TString>& _hosts);
    private:
        Connection(const Connection&);
        Connection& operator=(const Connection&);

        TVector<TString> hosts;
        TPgConnectionHolder conn;
        TDuration execTimeout;
        TInstant lastROCheck;
        TAtomicSharedPtr<TConnectionsTRaits> traits;
    };

} /* namespace sql */
