#pragma once

#include <util/generic/string.h>
#include <util/string/builder.h>
#include <mail/so/spamstop/tools/so-common/prof.h>
#include <mail/so/spamstop/tools/so-common/tstoragerequeststat.h>
#include <util/generic/bt_exception.h>

#include "Query.h"

namespace sql {
    struct TInterfaceError: public TWithBackTrace<yexception> {
        enum Type {
            Unknown = 0,
            PoolTimeout,
            TotalTimeout,
            ConnectError,
            ConnectTimeout,
            ExecError,
            QueryError,
            CannotAsyncExec,
            ExecTimeout,
            SetNonblocking,
            CannotCancel,
            AsyncConsume,
            Flush,
            AscyncCancel,
            ParsingResult,
            UnknownField
        };

        TString ToLog() const
        {
            return TString{AsStrBuf()};
        }

        TInterfaceError() : type(Unknown)
        {
            dynamic_cast<yexception&>(*this) << Unknown;
        }

        TInterfaceError(Type type, const TString & description) : type(type)
        {
            dynamic_cast<yexception&>(*this) << type << "; description: " << description;
        }


        TInterfaceError(const TInterfaceError & e, const TString & description) : type(e.type)
        {
            dynamic_cast<yexception&>(*this) << type << "; description: " << e.AsStrBuf() << ' ' << description;
        }

        explicit TInterfaceError(Type type) : type(type)
        {
            dynamic_cast<yexception&>(*this) << type;
        }
        Type type;
    };

    template <class T>
    struct TResWithError {
        bool ok;
        T res;
        TInterfaceError error;
        CProf prof;

        TString getProfString() const {
            return TStringBuilder{} << prof;
        }
        TString getErrorString() const {
            return TString{error.AsStrBuf()};
        }

        operator bool() const {
            return ok;
        }

        TResWithError(T res, CProf prof = CProf())
                : ok(true)
                , res(std::move(res))
                , prof(std::move(prof))
        {}
        TResWithError(TInterfaceError error, CProf prof = CProf())
                : ok(false)
                , error(std::move(error))
                , prof(std::move(prof))
        {
        }

        TResWithError<T> &operator=(TResWithError<T>&&) = default;
        TResWithError(TResWithError<T>&&) = default;
    };

    template <>
    struct TResWithError<void> {
        bool ok;
        TInterfaceError error;
        CProf prof;

        TString getProfString() const {
            return TStringBuilder{} << prof;
        }
        TString getErrorString() const {
            return TString{error.AsStrBuf()};
        }

        operator bool() const {
            return ok;
        }

        TResWithError(CProf prof = CProf())
                : ok(true)
                , prof(std::move(prof))
        {
        }

        TResWithError(const TInterfaceError& error, CProf prof = CProf())
                : ok(false)
                , error(error)
                , prof(std::move(prof))
        {
        }

        TResWithError(TResWithError&&) noexcept = default;
    };

    template<class T, class ...Args>
    TResWithError<T> ResWithError(T t, Args... args) {
        return TResWithError<T>(std::move(t), std::forward<Args>(args)...);
    };

    typedef TResWithError<void> TVoidWithError;

    template <class T>
    NStorageStats::TSRSErrType ToTSRSErrType(const TResWithError<T>& err) {
        NStorageStats::TSRSErrType res = NStorageStats::TSRSErrType();

        if (err.ok) {
            res = NStorageStats::SRS_OK;

        } else {
            switch (err.error.type) {
                case TInterfaceError::PoolTimeout:
                    res = NStorageStats::SRS_POOL_TIMEOUT;
                    break;
                case TInterfaceError::ExecTimeout:
                    res = NStorageStats::SRS_RQST_TIMEOUT;
                    break;
                case TInterfaceError::ConnectError:
                case TInterfaceError::ExecError:
                case TInterfaceError::QueryError:
                case TInterfaceError::CannotAsyncExec:
                case TInterfaceError::SetNonblocking:
                case TInterfaceError::CannotCancel:
                case TInterfaceError::AsyncConsume:
                case TInterfaceError::AscyncCancel:
                case TInterfaceError::ParsingResult:
                case TInterfaceError::Unknown:
                    res = NStorageStats::SRS_OTHER_ERROR;
                    break;
                default:
                    break;
            };
        }

        return res;
    }

} /* namespace sql */
