#pragma once

#include <util/generic/list.h>
#include <util/generic/vector.h>
#include <util/generic/string.h>
#include <util/stream/str.h>
#include "Oid.h"
#include "Inet.h"
#include "OidArrayConstructor.h"

namespace sql {
    class Query {
        friend class Work;
        friend class WorkTimed;
        friend class Connection;

    public:
        size_t size() const {
            return params.size();
        }

        void assignQuery(const TString& newQuery);
        TStringStream& getStream() {
            return query;
        };
        const TStringStream& getStream() const {
            return query;
        };
        TString toString() const;

        template <class T> void bind(const T & v) {
            data.push_back(TVector<char>(sizeof(T)));
            inverseBytes(&data.back()[0], &v, sizeof(T));

            bindOid(OidByType<T>::oid);
            bindBack();
        }

        void bind(const char * v);

        template <class T> OidArrayConstructor<T> getBoundOidArray(size_t elementsCount) {
            data.push_back(TVector<char>(OidArrayConstructor<T>::calcBufferSize(elementsCount)));
            bindOid(OidByType<T>::arrayOid);
            bindBack();

            return OidArrayConstructor<T>(&data.back()[0], elementsCount);
        }

        template <class T>
        void bindArray(const T* arr, size_t elementsCount) {
            OidArrayConstructor<T> ac(getBoundOidArray<T>(elementsCount));

            for (size_t i = 0; i < elementsCount; i++) {
                ac.push_back(arr[i]);
            }
        }

        TString ToLog() const {
            TStringStream ss;

            ss << "query:[" << query.Data() << ']';
            ss << "data:[";
            for (const auto &it : data) {
                for (char dIt : it) {
                    ss << int(dIt) << '-';
                }
                ss << ',';
            }
            ss << "] oids:[";

            for (unsigned int oid : oids) {
                ss << oid << ',';
            }
            ss << ']';

            return ss.Str();
        }

        Query(Query && other) noexcept
                : data(std::move(other.data)),
                  query(std::move(other.query)),
                  lengths(std::move(other.lengths)),
                  formats(std::move(other.formats)),
                  oids(std::move(other.oids)) {
            params.clear();
            for(const auto & d : data)
                params.emplace_back(&d[0]);
        }

        explicit Query(const TString& query = "");

    private:
        void assignToBack(const void* v, size_t length);
        void bindOid(Oid oid);
        void bindBack();

        TList<TVector<char>> data;
        TStringStream query;
        TVector<const char*> params;
        TVector<int> lengths;
        TVector<int> formats;
        TVector<Oid> oids;
    };

    template<> void Query::bind<TString>(const TString & v);

} /* namespace sql */
