
#include <contrib/libs/libpq/src/interfaces/libpq/libpq-fe.h>
#include <util/string/cast.h>
#include <util/system/defaults.h>
#include <util/datetime/systime.h>
#include "Result.h"

template<> void TDelete::Destroy<pg_result>(pg_result* res) noexcept {
    PQclear(res);
}

template<> void TDelete::Destroy<pgNotify>(pgNotify* res) noexcept {
    PQfreemem(res);
}

namespace sql {
    template <> size_t TPgTypesTraits<TString>::Size(const TString & v) { return v.size(); };
    template <> TVector<char> TPgTypesTraits<TString>::ToBinary(const TString & v) { return TVector<char>(v.cbegin(), v.cend()); };
    template <> TString TPgTypesTraits<TString>::FromBinary(const void * src, size_t size) { return {(char*)src, size}; };
    template struct TPgTypesTraits<TString>;

    template <> size_t TPgTypesTraits<TStringBuf>::Size(const TStringBuf & v) { return v.size(); };
    template <> TVector<char> TPgTypesTraits<TStringBuf>::ToBinary(const TStringBuf & v) { return TVector<char>(v.cbegin(), v.cend()); };
    template <> TStringBuf TPgTypesTraits<TStringBuf>::FromBinary(const void * src, size_t size) { return {(char*)src, size}; };
    template struct TPgTypesTraits<TStringBuf>;

    static TInstant GetMillenium() {
        static const TInstant millenium = TInstant::ParseIso8601Deprecated("2000-01-01");
        return millenium;
    }

    template <> size_t TPgTypesTraits<TInstant>::Size(TInstant) { return TPgTypesTraits<ui32>::Size({}); };
    template <> TVector<char> TPgTypesTraits<TInstant>::ToBinary(TInstant v) {
        return TPgTypesTraits<ui32>::ToBinary(static_cast<ui32>((v - GetMillenium()).Seconds()));
    };

    template <> TInstant TPgTypesTraits<TInstant>::FromBinary(const void * src, size_t size) {
        const auto fromMilleniumToDate = TDuration::Seconds(TPgTypesTraits<ui32>::FromBinary(src, size) * 24 * 60 * 60); // in PG it is days since 2000-01-01

        return GetMillenium() + fromMilleniumToDate;
    };
    template struct TPgTypesTraits<TInstant>;

    TStringBuf TResult::TCell::GetName() const {
        return PQfname(res, static_cast<int>(i));
    }

    bool TResult::TCell::IsNull() const {
        const auto ret =  PQgetisnull(res, static_cast<int>(line), static_cast<int>(i));
        return ret == 1;
    }

    OID TResult::TCell::GetType() const {
        return static_cast<OID>(PQftype(res, static_cast<int>(i)));
    }

    TResult::TCell::TCell(size_t line, size_t i, const pg_result * res) : line(line), i(i), res(res) {}

    char * TResult::TCell::GetRaw() const {
        return PQgetvalue(res, static_cast<int>(line), static_cast<int>(i));
    }
    size_t TResult::TCell::GetSize() const {
        return static_cast<size_t>(PQgetlength(res, static_cast<int>(line), static_cast<int>(i)));
    }

    bool TResult::TLine::Has(const TString & name) const {
        const auto ret = PQfnumber(res, name.c_str());
        return ret >= 0;
    }

    TResult::TCell TResult::TLine::operator[](const TString & name) const {
        const auto pos = PQfnumber(res, name.c_str());
        if(pos >= 0)
            return {line, (size_t) pos, res};
        ythrow TWithBackTrace<yexception>() << "result hasn't field " << name;
    }

    size_t TResult::AffectedLinesCount() const {
        TStringBuf count(PQcmdTuples(res.Get()));
        return count.empty() ? 0 : FromString<size_t>(count);
    }

    bool TResult::Ok() const {
        return res && (PQresultStatus(res.Get()) == PGRES_TUPLES_OK || PQresultStatus(res.Get()) == PGRES_COMMAND_OK);
    }

    TStringBuf TResult::ErrorMessage() const {
        if (!res)
            return "null result";
        return PQresultErrorMessage(res.Get());
    }

    TResult::TResult(TPgResultHolder _res) :
        res(std::move(_res)),
        columns(static_cast<const size_t>(PQnfields(res.Get()))),
        rows(static_cast<const size_t>(PQntuples(res.Get())))
    {
    }
} /* namespace sql */
