#pragma once

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/bt_exception.h>
#include <util/generic/yexception.h>
#include <util/generic/xrange.h>
#include <util/datetime/base.h>
#include "Inet.h"
#include "Oid.h"

template<> void TDelete::Destroy<struct pg_result>(pg_result* res) noexcept;
template<> void TDelete::Destroy<struct pgNotify>(pgNotify* res) noexcept;

namespace sql {
    using TPgResultHolder = THolder<pg_result>;
    using TPgNotifiesHolder = THolder<pgNotify>;

    template <class T> struct TPgTypesTraits {
        static size_t Size(typename TTypeTraits<T>::TFuncParam) {
            return sizeof(T);
        }
        static TVector<char> ToBinary(typename TTypeTraits<T>::TFuncParam val) {
            TVector<char> res(sizeof(T));
            inverseBytes(&res[0], &val, sizeof(T));
            return res;
        }
        static T FromBinary(const void * src, size_t size) {
            if(size != sizeof(T))
                ythrow TWithBackTrace<yexception>() << "size mismatch " << size << " vs " << sizeof(T);

            T res{};
            inverseBytes(&res, src, sizeof(T));
            return res;
        }
    };

    template <> size_t TPgTypesTraits<TString>::Size(const TString & v);
    template <> TVector<char> TPgTypesTraits<TString>::ToBinary(const TString & v);
    template <> TString TPgTypesTraits<TString>::FromBinary(const void * src, size_t size);

    template <> size_t TPgTypesTraits<TStringBuf>::Size(const TStringBuf & v);
    template <> TVector<char> TPgTypesTraits<TStringBuf>::ToBinary(const TStringBuf & v);
    template <> TStringBuf TPgTypesTraits<TStringBuf>::FromBinary(const void * src, size_t size);

    template <> size_t TPgTypesTraits<TInstant>::Size(TInstant);
    template <> TVector<char> TPgTypesTraits<TInstant>::ToBinary(TInstant v);
    template <> TInstant TPgTypesTraits<TInstant>::FromBinary(const void * src, size_t size);

    class TResult {
    public:

        class TCell{
        public:
            TStringBuf GetName() const;
            bool IsNull() const;
            OID GetType() const;
            template<class T> T As() const {
                return TPgTypesTraits<T>::FromBinary(GetRaw(), GetSize());
            }
            template<class T> TVector<T> AsArrayOf() const {

                const ui32 * src = reinterpret_cast<ui32*>(GetRaw());
                const ui32 * end = src + GetSize() / sizeof(ui32);

                const ui32 arraySize = TPgTypesTraits<ui32>::FromBinary(src + 3, sizeof(ui32));

                TVector<T> array(Reserve(arraySize));

                src +=5;    //  skip header

                while(src < end) {
                    const ui32 size = TPgTypesTraits<ui32>::FromBinary(src++, sizeof(ui32));
                    if(size == ui32(-1)) {
                        array.emplace_back();
                        continue;
                    }
                    array.emplace_back(TPgTypesTraits<T>::FromBinary(src, size));
                    src += size / sizeof(ui32);
                }
                return array;
            }
            TCell(size_t line, size_t i, const pg_result * res);
        private:
            char * GetRaw() const;
            size_t GetSize() const;
        private:
            size_t line, i;
            const pg_result * res;
        };

        class TLine{
        public:
            size_t Size() const {
                return columns;
            }

            bool Has(const TString & name) const;

            TCell operator[](const TString & name) const;

            TCell operator[](size_t i) const {
                if(i > columns)
                    ythrow TWithBackTrace<yexception>() << "column out of range " << columns;

                return {line, i, res};
            }
            TLine(size_t line, size_t columns, const pg_result * res) : line(line), columns(columns), res(res) {}
        private:
            size_t line;
            const size_t columns;
            const pg_result * res;
        };

        size_t AffectedLinesCount() const;

        bool Ok() const;

        TStringBuf ErrorMessage() const;

        size_t Size() const {
            return rows;
        }

        TLine operator[](size_t line) const  {
            if(line > rows)
                ythrow TWithBackTrace<yexception>() << "line out of range " << rows;
            return {line, columns, res.Get()};
        }

        TResult() = default;
        explicit TResult(TPgResultHolder _res);

    private:
        TPgResultHolder res;
        const size_t columns{}, rows{};
    };
} /* namespace sql */
