
#include <util/stream/str.h>
#include <util/generic/yexception.h>
#include <contrib/libs/libpq/src/interfaces/libpq/libpq-fe.h>
#include "connection_holder.h"
#include "Work.h"
#include "Error.h"

template<> void TDelete::Destroy<pg_conn>(pg_conn* conn) noexcept {
    PQfinish(conn);
}

namespace sql {
    TPgConnectionHolder TPgConnectionHolder::PqConnectDb(const TString & connectionString) {
        return TPgConnectionHolder(PQconnectdb(connectionString.c_str()), connectionString);
    }

    TPgConnectionHolder TPgConnectionHolder::PqConnectStart(const TString &connectionString) {
        return TPgConnectionHolder(PQconnectStart(connectionString.c_str()), connectionString);
    }

    TPgResultHolder TPgConnectionHolder::PqGetResult() {
        CheckNotNull();
        return TPgResultHolder(PQgetResult(instance.Get()));
    }

    TPgNotifiesHolder TPgConnectionHolder::PqGetNotifies() {
        CheckNotNull();
        return TPgNotifiesHolder(PQnotifies(instance.Get()));
    }

    int TPgConnectionHolder::PqFlush() {
        CheckNotNull();
        return PQflush(instance.Get());
    }

    int TPgConnectionHolder::PqSocket() const {
        CheckNotNull();
        return PQsocket(instance.Get());
    }

    void TPgConnectionHolder::PqReset() {
        PQreset(instance.Get());
    }

    int TPgConnectionHolder::PqConsumeInput() {
        CheckNotNull();
        return PQconsumeInput(instance.Get());
    }

    int TPgConnectionHolder::PqIsBusy() {
        CheckNotNull();
        return PQisBusy(instance.Get());
    }

    int TPgConnectionHolder::PqIsnonblocking() const {
        CheckNotNull();
        return PQisnonblocking(instance.Get());
    }

    int TPgConnectionHolder::PqSetnonblocking(int nonblocking) {
        CheckNotNull();
        return PQsetnonblocking(instance.Get(), nonblocking);
    }

    int TPgConnectionHolder::PqSendQueryParams(const char *command,
                          int nParams,
                          const Oid *paramTypes,
                          const char *const * paramValues,
                          const int *paramLengths,
                          const int *paramFormats,
                          int resultFormat) {
        CheckNotNull();
        return PQsendQueryParams(instance.Get(), command, nParams, paramTypes, paramValues, paramLengths, paramFormats, resultFormat);
    }

    int TPgConnectionHolder::PqSendQuery(const char *query) {
        CheckNotNull();
        return PQsendQuery(instance.Get(), query);
    }

    TPgResultHolder TPgConnectionHolder::PqExecParams(const char *command,
                                 int nParams,
                                 const Oid *paramTypes,
                                 const char *const * paramValues,
                                 const int *paramLengths,
                                 const int *paramFormats,
                                 int resultFormat) {
        CheckNotNull();
        return TPgResultHolder(PQexecParams(instance.Get(), command, nParams, paramTypes, paramValues, paramLengths, paramFormats, resultFormat));
    }

    TPgResultHolder TPgConnectionHolder::PqExec(const char *query) {
        CheckNotNull();
        return TPgResultHolder(PQexec(instance.Get(), query));
    }

    EPollingStatus TPgConnectionHolder::PqConnectPoll() {
        CheckNotNull();
        switch (PQconnectPoll(instance.Get())) {
            case PGRES_POLLING_OK:      return EPollingStatus::Ok;
            case PGRES_POLLING_FAILED:  return EPollingStatus::Fail;
            case PGRES_POLLING_READING: return EPollingStatus::Read;
            case PGRES_POLLING_WRITING: return EPollingStatus::Write;;
            case PGRES_POLLING_ACTIVE:  return EPollingStatus::InProgress;
        }
    }

    TString TPgConnectionHolder::Info() const {
        TStringStream s;
        s << *this;
        return std::move(s.Str());
    }

    TStringBuf TPgConnectionHolder::Host() const {
        CheckNotNull();
        return PQhost(instance.Get());
    }

    TStringBuf TPgConnectionHolder::Db() const {
        CheckNotNull();
        return PQdb(instance.Get());
    }

    TStringBuf TPgConnectionHolder::ErrorMessage() const {
        CheckNotNull();
        return PQerrorMessage(instance.Get());
    }

    bool TPgConnectionHolder::Ok() const {
        return instance && PQstatus(instance.Get()) != CONNECTION_BAD;
    }

    void TPgConnectionHolder::Reset() {
        instance.Reset();
    }

    bool TPgConnectionHolder::IsReplica() {
        TResult result(PqExec("show transaction_read_only;"));

        if (!result.Ok())
            ythrow TInterfaceError(TInterfaceError::ExecError) << result.ErrorMessage();

        if (result.Size() == 0 || result[0].Size() == 0)
            ythrow TInterfaceError(TInterfaceError::ExecError) << "result is zero size";

        const auto & transaction_read_only = result[0][0].As<TStringBuf>();

        return transaction_read_only == "on";
    }

    IOutputStream &operator << (IOutputStream & stream, const TPgConnectionHolder & holder) {
        if(holder.instance)
            return stream
                    << "host: " << holder.Host() << ';'
                    << "db: " << holder.Db() << ';'
                    << "ok: " << holder.Ok() << ';'
                    << "error: " << holder.ErrorMessage() << ';';
        return stream << "null;";
    }

    void TPgConnectionHolder::CheckNotNull() const {
        if(Y_UNLIKELY(!instance))
            ythrow TWithBackTrace<yexception>() << "null connect";
    }

    TPgConnectionHolder::TPgConnectionHolder(pg_conn * conn, TString connectionString)
            : instance(conn), connectionString(std::move(connectionString)) {}

}
