#pragma once

#include <util/string/type.h>

#include <utility>
#include <util/charset/unidata.h>
#include <util/generic/deque.h>
#include <util/string/vector.h>
#include <mail/so/spamstop/tools/postgreclient/pq/Connection.h>

namespace sql {
    using TPGPool = TPool<sql::Connection>;
    using TConnectionHolder = TPGPool::TItemHolder;
    class PGSettings: public TPGPool::TPoolItemTraits {
    public:
        static TString MakeSafe(TString s) {
            static const TString passTag = "password=";
            size_t passPos = 0;
            while(true) {
                passPos = s.find(passTag, passPos);

                if(passPos == TString::npos)
                    return s;

                passPos += passTag.Size();

                size_t passEnd = passPos + 1;
                while(passEnd < s.Size() && IsAlnum(s[passEnd]))
                    passEnd++;

                s.erase(passPos, passEnd - passPos - 1);
                passPos = passEnd;
            }
        }

        void setExecTimeout(const TDuration& timeout) {
            execTimeout = timeout;
        }

        THolder<sql::Connection> create() const override {
            auto newConnection = MakeHolder<sql::Connection>(connectionString);
            if (execTimeout != TDuration::Zero())
                newConnection->setExecTimeout(execTimeout);

            newConnection->SetConnectionTraits(traits);

            return newConnection;
        }

        TString SafeConnectString() const {
            TVector<TString> safeStrings(Reserve(connectionString.size()));
            for(const auto & s : connectionString)
                safeStrings.emplace_back(MakeSafe(s));

            return JoinStrings(safeStrings, ",");
        }

        void SetConnectionString(const TVector<TString> & cons) {
            connectionString = cons;
            for(const auto & con : connectionString)
                traits->AddAndCheck(con);

            h = std::accumulate(connectionString.cbegin(), connectionString.cend(), size_t{}, [](size_t h, const TString & s) {
                return CombineHashes(h, THash<TString>()(s));
            });
        }

        const TVector<TString> & GetConnectionString() const {
            return connectionString;
        }

        TAtomicSharedPtr<TConnectionsTRaits> GetTraits() {
            return traits;
        }

        TDuration GetTimeout() const {
            return execTimeout;
        }

        PGSettings() = default;

        explicit PGSettings(const TVector<TString> & _connectionString) {
            SetConnectionString(_connectionString);
        }

        ~PGSettings() override = default;

    private:
        TVector<TString> connectionString;
        TDuration execTimeout;
        TAtomicSharedPtr<TConnectionsTRaits> traits = MakeAtomicShared<TConnectionsTRaits>();
        size_t h {};
    };
}
