import json
from time import time
from infra.yasm.yasmapi import GolovanRequest
import typing
import collections

periods = collections.namedtuple('periods', ('sec', 'minute', 'hour', 'day', 'week'))(1, 60, 60 * 60, 60 * 60 * 24, 60 * 60 * 24 * 7)


class TPeriod:
    def __init__(self, start: int, length: int):
        self.start = start
        self.end = start + length

    def __repr__(self):
        return f"{self.start} {self.end}"


class TRequestData:
    def __init__(self, host: str, period: TPeriod, signals: typing.List[str]):
        self.host = host
        self.period = period
        self.signals = signals


SIGNALS_NAMES = (
    "ACNT_1_spam",
    "AETP_NET_ham",
    "APPLICATION_ATTACH_PDF_ham",
    "APPLICATION_ATTACH_ham",
    "ATTACH_ham",
    "ATTACH_malic",
    "ATTACH_spam",
    "BAD_URL_ZONE_ham",
    "BALANCE_BODY_ham",
    "BAN_FAKE_RDNS_ham",
    "BAN_PDF_ATTACH_NAME_ham",
    "BAN_REDIR_G_ham",
    "BROKEN_B64_ham",
    "CLOUD_MAIL_ham",
    "CL_AVIATICKET_ham",
    "CL_AVIATICKET_spam",
    "CL_BILL_ham",
    "CL_BILL_malic",
    "CL_BILL_spam",
    "CL_BOARDINGPASS_ham",
    "CL_BOUNCE_ham",
    "CL_BOUNCE_malic",
    "CL_BOUNCE_spam",
    "CL_CANCEL_ham",
    "CL_CANCEL_spam",
    "CL_DELIVERY_ham",
    "CL_DELIVERY_malic",
    "CL_DELIVERY_spam",
    "CL_DISCOUNT_ham",
    "CL_DISCOUNT_malic",
    "CL_DISCOUNT_spam",
    "CL_EDOC_ham",
    "CL_EDOC_malic",
    "CL_EDOC_spam",
    "CL_ESHOP_ham",
    "CL_ESHOP_malic",
    "CL_ESHOP_spam",
    "CL_ETICKET_ham",
    "CL_ETICKET_malic",
    "CL_ETICKET_spam",
    "CL_FIRSTMAIL_ham",
    "CL_FIRSTMAIL_malic",
    "CL_FIRSTMAIL_spam",
    "CL_GREETING_ham",
    "CL_HOTELBOOK_ham",
    "CL_HOTELBOOK_spam",
    "CL_INVITE_ham",
    "CL_INVITE_spam",
    "CL_MESSAGE_ham",
    "CL_MESSAGE_malic",
    "CL_MESSAGE_spam",
    "CL_NEWS_ham",
    "CL_NEWS_malic",
    "CL_NEWS_spam",
    "CL_NOTIFICATION_ham",
    "CL_NOTIFICATION_malic",
    "CL_NOTIFICATION_spam",
    "CL_PEOPLE_ham",
    "CL_PEOPLE_malic",
    "CL_PEOPLE_spam",
    "CL_PHISHING_ham",
    "CL_PHISHING_malic",
    "CL_PHISHING_spam",
    "CL_REGISTRATION_ham",
    "CL_REGISTRATION_malic",
    "CL_REGISTRATION_spam",
    "CL_REMINDER_ham",
    "CL_REMINDER_malic",
    "CL_REMINDER_spam",
    "CL_SCHEMA_ham",
    "CL_SCHEMA_malic",
    "CL_SCHEMA_spam",
    "CL_SMARTREPLY_ham",
    "CL_SMARTREPLY_malic",
    "CL_SMARTREPLY_spam",
    "CL_SNDR_ADVERT_ham",
    "CL_SNDR_ADVERT_malic",
    "CL_SNDR_ADVERT_spam",
    "CL_SNDR_AVIAETICKET_ham",
    "CL_SNDR_AVIAETICKET_spam",
    "CL_SNDR_BANK_ham",
    "CL_SNDR_BANK_malic",
    "CL_SNDR_BANK_spam",
    "CL_SNDR_COMPANY_ham",
    "CL_SNDR_COMPANY_malic",
    "CL_SNDR_COMPANY_spam",
    "CL_SNDR_COUPON_ham",
    "CL_SNDR_COUPON_malic",
    "CL_SNDR_COUPON_spam",
    "CL_SNDR_DATING_ham",
    "CL_SNDR_DATING_malic",
    "CL_SNDR_DATING_spam",
    "CL_SNDR_DELIVERY_ham",
    "CL_SNDR_DELIVERY_malic",
    "CL_SNDR_DELIVERY_spam",
    "CL_SNDR_ESHOP_ham",
    "CL_SNDR_ESHOP_malic",
    "CL_SNDR_ESHOP_spam",
    "CL_SNDR_EVENT_ham",
    "CL_SNDR_EVENT_malic",
    "CL_SNDR_EVENT_spam",
    "CL_SNDR_FORUM_ham",
    "CL_SNDR_FORUM_malic",
    "CL_SNDR_FORUM_spam",
    "CL_SNDR_GAME_ham",
    "CL_SNDR_GAME_malic",
    "CL_SNDR_GAME_spam",
    "CL_SNDR_INSUR_ham",
    "CL_SNDR_INSUR_malic",
    "CL_SNDR_INSUR_spam",
    "CL_SNDR_JOB_ham",
    "CL_SNDR_JOB_malic",
    "CL_SNDR_JOB_spam",
    "CL_SNDR_MEDIA_ham",
    "CL_SNDR_MEDIA_malic",
    "CL_SNDR_MEDIA_spam",
    "CL_SNDR_MOBILE_ham",
    "CL_SNDR_MOBILE_malic",
    "CL_SNDR_MOBILE_spam",
    "CL_SNDR_NEWS_ham",
    "CL_SNDR_NEWS_malic",
    "CL_SNDR_NEWS_spam",
    "CL_SNDR_PROVIDER_ham",
    "CL_SNDR_PROVIDER_malic",
    "CL_SNDR_PROVIDER_spam",
    "CL_SNDR_REALTY_ham",
    "CL_SNDR_REALTY_malic",
    "CL_SNDR_REALTY_spam",
    "CL_SNDR_SENDER_ham",
    "CL_SNDR_SENDER_malic",
    "CL_SNDR_SENDER_spam",
    "CL_SNDR_SOCIAL_ham",
    "CL_SNDR_SOCIAL_malic",
    "CL_SNDR_SOCIAL_spam",
    "CL_SNDR_STATE_ham",
    "CL_SNDR_STATE_malic",
    "CL_SNDR_STATE_spam",
    "CL_SNDR_TAXI_ham",
    "CL_SNDR_TAXI_spam",
    "CL_SNDR_TECH_ham",
    "CL_SNDR_TECH_malic",
    "CL_SNDR_TECH_spam",
    "CL_SNDR_TRACKER_ham",
    "CL_SNDR_TRACKER_malic",
    "CL_SNDR_TRACKER_spam",
    "CL_SNDR_TRAINING_ham",
    "CL_SNDR_TRAINING_malic",
    "CL_SNDR_TRAINING_spam",
    "CL_SNDR_TRAVEL_ham",
    "CL_SNDR_TRAVEL_malic",
    "CL_SNDR_TRAVEL_spam",
    "CL_SNDR_ZDTICKET_ham",
    "CL_SNDR_ZDTICKET_spam",
    "CL_TAB_NEWS_ham",
    "CL_TAB_NEWS_malic",
    "CL_TAB_NEWS_spam",
    "CL_TAB_NOTIF_ham",
    "CL_TAB_NOTIF_malic",
    "CL_TAB_NOTIF_spam",
    "CL_TAB_PEOPLE_ham",
    "CL_TAB_PEOPLE_malic",
    "CL_TAB_PEOPLE_spam",
    "CL_TAB_SOCIAL_ham",
    "CL_TAB_SOCIAL_malic",
    "CL_TAB_SOCIAL_spam",
    "CL_TICKET_REMIND_ham",
    "CL_TRUST_1_ham",
    "CL_TRUST_1_malic",
    "CL_TRUST_1_spam",
    "CL_TRUST_2_ham",
    "CL_TRUST_2_malic",
    "CL_TRUST_2_spam",
    "CL_TRUST_3_ham",
    "CL_TRUST_3_malic",
    "CL_TRUST_3_spam",
    "CL_TRUST_4_ham",
    "CL_TRUST_4_malic",
    "CL_TRUST_4_spam",
    "CL_TRUST_5_ham",
    "CL_TRUST_5_malic",
    "CL_TRUST_5_spam",
    "CL_TRUST_6_ham",
    "CL_TRUST_6_spam",
    "CL_VIRUS_ham",
    "CL_VIRUS_malic",
    "CL_VIRUS_spam",
    "DL_FBR_ham",
    "DL_FBR_malic",
    "DL_FBR_spam",
    "DL_LIST_PRESENT_ham",
    "DROPBOX_URL_ham",
    "EMAIL_ATTRIBUTION_2_ham",
    "EMAIL_ATTRIBUTION_3_ham",
    "EMAIL_ATTRIBUTION_ham",
    "END_URL_RF_ham",
    "ENVELOPED_2_ham",
    "ENVELOPED_3_ham",
    "EPILOGUE_ham",
    "ETICKET_SUBJ_ham",
    "ETICKET_SUBJ_malic",
    "ETICKET_SUBJ_spam",
    "FILES_MAILRU_TEXT_ham",
    "FONT_WHITE_ham",
    "FORGOTTEN_PASSWORD_ham",
    "FREE_BLOG_ham",
    "FROM_DONOTREPLAY_ham",
    "FROM_TUT_BY_ham",
    "F_INVITE_ham",
    "GOOGLE_DOCS_P_ham",
    "G_INVITE_ham",
    "G_PODELILSYA_ham",
    "HAS_ORGANIZATION_ham",
    "INVITATION_SUBJ_ham",
    "IN_SPRAV_ham",
    "IN_SPRAV_malic",
    "IN_SPRAV_spam",
    "LSA_CAT_BOOST_TAB_NEWS_ham",
    "LSA_CAT_BOOST_TAB_NOTIF_ham",
    "LSA_CAT_BOOST_TAB_PEOPLE_ham",
    "LSA_CAT_BOOST_TAB_SOCIAL_ham",
    "LSA_NN_TAB_NEWS_ham",
    "LSA_NN_TAB_NOTIF_ham",
    "LSA_NN_TAB_PEOPLE_ham",
    "LSA_NN_TAB_SOCIAL_ham",
    "LSA_TAB_NEWS_3_ham",
    "LSA_TAB_NOTIF_3_ham",
    "LSA_TAB_PEOPLE_3_ham",
    "LSA_TAB_SOCIAL_3_ham",
    "LSA_VW_TAB_NEWS_ham",
    "LSA_VW_TAB_NOTIF_ham",
    "LSA_VW_TAB_PEOPLE_ham",
    "LSA_VW_TAB_SOCIAL_ham",
    "MIXED_LET_NUMS_BODY_ham",
    "MIXED_LET_NUMS_SUBJ_ham",
    "NOTIFICATION_SUBJ_ham",
    "NOTIF_FROM_ham",
    "NPRSF_FAIL_ham",
    "NPRSF_FAIL_malic",
    "NPRSF_FAIL_spam",
    "ORIGINAL_MESSAGE_ID_ham",
    "PGP_MESSAGE_ham",
    "PREAMBLE_ham",
    "QUOTE_TWICE_1_ham",
    "RA21_11_ham",
    "REGISTRATION_TRY_1_ham",
    "REGISTRATION_TRY_ham",
    "REG_ATTACK_2_ham",
    "REG_ATTACK_ham",
    "REPLY_WITH_QUOTES_ham",
    "SCHET_ATT_ham",
    "SCHET_SUBJ_ham",
    "SENDER_FAIL_ham",
    "SENDER_FAIL_malic",
    "SENDER_FAIL_spam",
    "SH1_SKIP_ham",
    "SH1_SKIP_malic",
    "SH1_SKIP_spam",
    "SH_27_NEW_ham",
    "SPAM_HASH_ham",
    "SPAM_NET_2_ham",
    "SPAM_NET_ham",
    "SPLIT_WORDS_6_ham",
    "SRCLICK_REDIR_ham",
    "SRCLICK_URL_ham",
    "TRUST_2MAX_ham",
    "TRUST_2MAX_spam",
    "TRUST_NO_ham",
    "TRUST_NO_spam",
    "UAAS_TAB_0_ham",
    "UAAS_TAB_0_malic",
    "UAAS_TAB_0_spam",
    "UAAS_TAB_1_ham",
    "UAAS_TAB_1_malic",
    "UAAS_TAB_1_spam",
    "UAAS_TAB_2_ham",
    "UAAS_TAB_2_malic",
    "UAAS_TAB_2_spam",
    "UAAS_TAB_3_ham",
    "UAAS_TAB_3_malic",
    "UAAS_TAB_3_spam",
    "UAAS_TAB_4_ham",
    "UAAS_TAB_4_malic",
    "UAAS_TAB_4_spam",
    "UAAS_TAB_5_ham",
    "UAAS_TAB_5_malic",
    "UAAS_TAB_5_spam",
    "UAAS_TAB_DEF_ham",
    "UAAS_TAB_DEF_malic",
    "UAAS_TAB_DEF_spam",
    "UAAS_TAB_MULTI_ham",
    "UAAS_TAB_MULTI_malic",
    "UAAS_TAB_MULTI_spam",
    "VK_URL_ham",
    "XSKIPREC_ham",
    "XSKIPREC_malic",
    "XSKIPREC_spam",
    "X_ORIGINAL_MESSAGE_ID_ham",
    "YASENDER_MUA_ham",
    "YASENDER_MUA_malic",
    "YASENDER_MUA_spam",
    "YA_DEVNULL_RP_ham",
    "YA_DEVNULL_RP_spam",
    "YA_DISK_DOCS_ham",
    "YA_DISK_PARAM_2_ham",
    "ZAKAZ_BODY_ham",
    "_EPILOGUE__ham",
    "__ALEN_1000_MAX_ham",
    "__ALEN_1000_MAX_malic",
    "__ALEN_1000_MAX_spam",
    "__APPLICATION_ATTACH_NAME_ARH_ham",
    "__APPLICATION_ATTACH_NAME_DOC_ham",
    "__APPLICATION_ATTACH_NAME_PDF_ham",
    "__APPLICATION_ATTACH_NAME_ZIP_ham",
    "__ATT_CL_ham",
    "__ATT_CL_malic",
    "__ATT_CL_spam",
    "__BAN_SPAMER_ham",
    "__BIG_ATTACHMENTS_ham",
    "__BIG_ATTACHMENTS_malic",
    "__BIG_ATTACHMENTS_spam",
    "__CLOUD_SPAM_ham",
    "__CONFIRM_IN_BODY_ham",
    "__CONTINUOUS_LETTERS_1_ham",
    "__CONTINUOUS_LETTERS_FROM_TO_ham",
    "__CONTINUOUS_LETTERS_SUBJ_ham",
    "__CONTINUOUS_LETTERS_ham",
    "__EMPTY_BODYTEXT_ham",
    "__IMAGE_ATTACH_BMP_ham",
    "__IMAGE_ATTACH_GIF_ham",
    "__IMAGE_ATTACH_JPG_ham",
    "__IMAGE_ATTACH_NAME_ham",
    "__IMAGE_ATTACH_PNG_ham",
    "__IMAGE_ATTACH_ham",
    "__IN_REP_TO_ham",
    "__MAIL_RU_ham",
    "__MESSAGE_READ_ham",
    "__MESS_LEN_0_ham",
    "__MESS_LEN_0_malic",
    "__MESS_LEN_0_spam",
    "__MESS_NO_BODY_ham",
    "__MESS_NO_BODY_malic",
    "__MESS_NO_BODY_spam",
    "__MIXED_LETTERS_FROM_N_ham",
    "__MIXED_LETTERS_IN_WORD_ham",
    "__ORIGINAL_MESSAGE_ham",
    "__PASSWORD_ham",
    "__RAMBLER_RU_ham",
    "__RCVD_LI_ham",
    "__REFERENCES_ham",
    "__REFERENCES_malic",
    "__REFERENCES_spam",
    "__REGISTRATION_IN_BODY_ham",
    "__REGISTRATION_IN_SUBJ_ham",
    "__REP_TO_ham",
    "__SPAM_SRC_ham",
    "__SPLIT_WORDS_2_ham",
    "__SPLIT_WORDS_SUBJ_ham",
    "__SPLIT_WORDS_ham",
    "__SUPPORT_IN_FROM_ham",
    "__TESTING_ham",
    "__UPPERCASE_ham",
    "__YAFRONT_YABACK_ham",
    "__YAFRONT_YABACK_spam",
    "__YAHOO_FR_ham",
    "__YANDEX_CALEND_FROM_ham",
    "__YANDEX_CALEND_FROM_spam",
    "__YA_DISK_ham",
    "__ZAKAZ_SUBJ_ham",
    "___HAS_REFERENCES_ham",
)
PREFIX = "itype=spdaemon;ctype=prod;nanny=spdaemon-in;prj=so:unistat-rule_"
POSTFIX = "_summ"


class TSignal:
    def __init__(self, tags: str, name: str = None):
        if name is None:
            from_index = tags.index(PREFIX)

            if from_index == -1:
                raise Exception(f"cannot find prefix {PREFIX} in {tags}")

            self.tags = tags[:from_index]

            from_index += len(PREFIX)

            to_index = tags.index(POSTFIX, from_index)
            self.name = tags[from_index:to_index]
        else:
            self.tags = tags
            self.name = name

    def __str__(self):
        return PREFIX + self.name + POSTFIX

    def __repr__(self):
        return self.name

    def __hash__(self):
        return hash(self.name)

    def __eq__(self, other: "TSignal"):
        return self.name == other.name


SIGNALS = tuple(f"{PREFIX}{name}{POSTFIX}" for name in SIGNALS_NAMES if name.endswith("ham"))


class TStat:
    def __init__(self, base: int, next_value: int):
        self.base = base
        self.next = next_value

    def diff(self):
        return self.next - self.base

    def is_not_good(self):
        return self.base + self.next >= 100 and self.perc_diff() > 0.5

    def perc_diff(self):
        s = self.base
        if s == 0:
            s = self.next
        if s == 0:
            return 0
        return (self.next - self.base) / s

    def __repr__(self):
        return "%d->%d" % (self.base, self.next)


def perc_to_status(stat: TStat) -> str:
    return "CRIT" if stat.is_not_good() else "OK"


def main():
    host = "ASEARCH"
    delta = 5 * periods.minute

    now = int(time())
    now -= now % delta

    previous_period = TPeriod(now - periods.day - delta, delta)
    current_period = TPeriod(now - delta, delta)

    one_request_limitation = 200

    stats_by_signals: typing.Dict[TSignal, TStat] = {}

    for i in range(0, len(SIGNALS), one_request_limitation):
        str_signals = tuple(str(signal) for signal in SIGNALS[i:i + one_request_limitation])

        try:
            current = list(GolovanRequest(host, delta, current_period.start, current_period.end, str_signals))
            previous = list(GolovanRequest(host, delta, previous_period.start, previous_period.end, str_signals))

            for [str_signal, count_cur], [str_signal_test, count_prev] in zip(current[0][1].items(), previous[0][1].items()):
                if str_signal != str_signal_test:
                    raise Exception(f"{str_signal} vs {str_signal_test}")
                signal = TSignal(str_signal)
                stats_by_signals[signal] = TStat(count_cur or 0, count_prev or 0)
        except Exception as e:
            print(e)
            continue

    enough_great_items = filter(lambda p: p[1].is_not_good(), stats_by_signals.items())
    stats = sorted(enough_great_items, reverse=True, key=lambda p: p[1].perc_diff())

    if stats:
        events = [
            {
                "service": "so_in_top_unstable_rules_daily",
                "description": "\n".join(f"{repr(signal)}:{repr(stat)}" for signal, stat in stats),
                "tags": ["rules_checks"],
                "status": perc_to_status(stats[0][1])
            }
        ]
    else:
        events = [
            {
                "service": "so_in_top_unstable_rules_daily",
                "description": "all rules are stable",
                "tags": ["rules_checks"],
                "status": "OK"
            }
        ]

    data = {
        "events": events
    }

    print(json.dumps(data))
