#pragma once

#include "util/generic/hash.h"
#include <library/cpp/cgiparam/cgiparam.h>
#include <mail/so/spamstop/tools/so-common/shconn.h>
#include <mail/so/spamstop/tools/so-clients/tshinglerenv.h>
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include <mail/so/spamstop/tools/so-common/unnamedsem.h>
#include <list>
#include <vector>
#include "tlogsgroup.h"
#include "tshinglesclasses.h"
#include <mail/so/spamstop/tools/so-common/ttrafficcontrol.h>

#ifndef _win_
#include <sys/types.h>
#include <sys/syscall.h>
#include <fcntl.h>
#include <unistd.h>
#include <mail/so/libs/curl/curl.h>
#include <util/system/thread.h>
#endif

#define MAX_DIRECTS_COUNT 16

//*******************************************************************************************************************************************
//                                                               TRPLQueue
//*******************************************************************************************************************************************

struct TShingleShortInfoRPL {
    ui64 m_att_shingle;
    ui16 m_att_shingle_type;
    TShingleSpamType m_spamtype;
    TPersType m_pers;
    ui16 m_ham;
    ui16 m_dlv;
    ui16 m_spam;
    ui16 m_persham;
    ui16 m_persspam;
    bool m_follow;

    TShingleShortInfoRPL() {
        Clear();
    }

    TShingleShortInfoRPL(ui64 att_shingle, ui16 att_shingle_type, TShingleSpamType spamtype, TPersType pers, ui16 ham, ui16 dlv, ui16 spam, ui16 persham, ui16 persspam, bool follow) {
        m_att_shingle = att_shingle;
        m_att_shingle_type = att_shingle_type;
        m_spamtype = spamtype;
        m_pers = pers;
        m_ham = ham;
        m_dlv = dlv;
        m_spam = spam;
        m_persham = persham;
        m_persspam = persspam;
        m_follow = follow;
    }

    void Clear() {
        m_att_shingle = 0;
        m_att_shingle_type = 0;
        m_spamtype = UNKNOWN;
        m_pers = PERSUNDEF;
        m_ham = 0;
        m_dlv = 0;
        m_spam = 0;
        m_persham = 0;
        m_persspam = 0;
        m_follow = false;
    }

    TString GetFollowInfo() {
        char tbuff[512];

        tbuff[sizeof(tbuff) - 1] = 0x00;
        if (m_att_shingle != 0)
            snprintf(tbuff, sizeof(tbuff) - 1, "h=%d, d=%d, s=%d, ph=%d, ps=%d, attsh=%" PRIx64 "-%d, %s, %s", m_ham, m_dlv, m_spam, m_persham, m_persspam, m_att_shingle, m_att_shingle_type, TShingleSpamTypeToStroka(m_spamtype).c_str(), TPersTypeToStroka(m_pers).c_str());
        else
            snprintf(tbuff, sizeof(tbuff) - 1, "h=%d, d=%d, s=%d, ph=%d, ps=%d", m_ham, m_dlv, m_spam, m_persham, m_persspam);

        return TString(tbuff);
    }
};

typedef THashMap<ui64, TShingleShortInfoRPL> TShingleShortInfoRPLHash;
typedef TShingleShortInfoRPLHash::iterator TShingleShortInfoRPLHashIt;

struct TShingleExtInfoRPL {
    ui64 shingle;
    ui16 type;
    TShingleShortInfoRPL value;

    TShingleExtInfoRPL() {
        Clear();
    }

    TShingleExtInfoRPL(ui64 shingleA, ui16 typeA, ui16 hamA, ui16 dlvA, ui16 spamA, ui16 pershamA, ui16 persspamA, ui64 att_shingleA, ui16 att_shingle_typeA, TShingleSpamType spamtype, TPersType pers, bool followA) {
        shingle = shingleA;
        type = typeA;
        value.m_att_shingle = att_shingleA;
        value.m_att_shingle_type = att_shingle_typeA;
        value.m_spamtype = spamtype;
        value.m_pers = pers;
        value.m_ham = hamA;
        value.m_dlv = dlvA;
        value.m_spam = spamA;
        value.m_persham = pershamA;
        value.m_persspam = persspamA;
        value.m_follow = followA;
    }

    void Clear() {
        shingle = 0;
        type = 0;
        value.Clear();
    }

    TString toLog() {
        char buff[1024];

        buff[sizeof(buff) - 1] = 0x00;
        if (value.m_att_shingle != 0)
            snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "-%d: h=%d, d=%d, s=%d, ph=%d, ps=%d, attsh=%" PRIx64 ", attshtype=%d, %s, %s", shingle, type, value.m_ham, value.m_dlv, value.m_spam, value.m_persham, value.m_persspam, value.m_att_shingle, value.m_att_shingle_type, TShingleSpamTypeToStroka(value.m_spamtype).c_str(), TPersTypeToStroka(value.m_pers).c_str());
        else
            snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "-%d: h=%d, d=%d, s=%d, ph=%d, ps=%d", shingle, type, value.m_ham, value.m_dlv, value.m_spam, value.m_persham, value.m_persspam);

        return TString(buff);
    }
};

typedef std::list<TShingleExtInfoRPL> TShingleExtInfoRPLList;
typedef TShingleExtInfoRPLList::iterator TShingleExtInfoRPLListIt;

class TRPLQueue {
public:
    static const ui16 MAX_COUNT_TYPES = MAX_COUNT_SHINGLES_TYPE;
    static const ui32 MAX_QUEUE_SIZE = 10000;
    static const ui32 MAX_SHINGLES_IN_REQUEST = 200;
    static const ui32 GETDATA_MIN_COUNT = 100;
    static const ui32 GETDATA_TIMEOUT_MS = 100;

public:
    ui32 m_max_queue_size;
    ui32 m_max_shingles_in_request;
    ui32 m_min_shingles_in_request;
    ui32 m_timeout_create_sendblock;

    ui64 m_need_return_data_time_today;
    ui64 m_need_return_data_count_today;
    ui64 m_need_return_data_time_yesterday;
    ui64 m_need_return_data_count_yesterday;

    ui32 inqueue_count;     // считаем, сколько данных у нас сейчас внутри очереди

private:
    TShingleShortInfoRPLHash queuedata[MAX_COUNT_TYPES];
    TShingleExtInfoRPLList queue_list;
    TMutex m_Mutex;
    TUnnamedSemaphore* m_Sema;
    bool m_stop_event;
    int thread_count;
    ui32 m_last_return_data;

    TKAtomic today_in;
    TKAtomic today_up;
    TKAtomic today_out;
    TKAtomic today_lost;
    TKAtomic today_in_rqst;
    TKAtomic today_out_rqst;

    ui64 yesterday_in;
    ui64 yesterday_up;
    ui64 yesterday_out;
    ui64 yesterday_in_rqst;
    ui64 yesterday_out_rqst;
    ui64 yesterday_lost;
    TMutex MidnightMutex;

    void SendEvent();
    void WaitEvent();
    void UnBlockAllEvent(int thread_count);

public:
    TRPLQueue();
    ~TRPLQueue();

    void Init(int thread_countA, ui32 max_queue_sizeA, ui32 max_shingles_in_requestA, ui32 min_shingles_in_requestA, ui32 timeout_create_sendblockA);

    bool AddRecord(TShingleExtInfoRPLList& shinglelist);
    bool GetRecord(TShingleExtInfoRPLList& shinglelist);
    void Midnight();
    void Shutdown();

    void SendEventFunc() {
        SendEvent();
    }

    ui64 GetTodayIN() {
        return today_in.Value();
    }
    ui64 GetTodayUP() {
        return today_up.Value();
    }
    ui64 GetTodayOUT() {
        return today_out.Value();
    }
    ui64 GetTodayINRqst() {
        return today_in_rqst.Value();
    }
    ui64 GetTodayOUTRqst() {
        return today_out_rqst.Value();
    }
    ui64 GetTodayLost() {
        return today_lost.Value();
    }
    ui32 GetTodayCount();
    ui64 GetYesterdayIN() {
        return yesterday_in;
    }
    ui64 GetYesterdayUP() {
        return yesterday_up;
    }
    ui64 GetYesterdayOUT() {
        return yesterday_out;
    }
    ui64 GetYesterdayINRqst() {
        return yesterday_in_rqst;
    }
    ui64 GetYesterdayOUTRqst() {
        return yesterday_out_rqst;
    }
    ui64 GetYesterdayLost() {
        return yesterday_lost;
    }
};

//*******************************************************************************************************************************************
//                                                             TRPLStatDataItem
//*******************************************************************************************************************************************

struct TRPLStatDataItem {
    // по клиентам
    ui32 all_mail_count;
    ui32 excludehosts_mail_count;
    ui32 connect_ok;
    ui32 connect_error;
    ui32 connect_timeout;
    ui32 request_200_count;
    ui32 request_4xx_count;
    ui32 request_5xx_count;
    ui32 request_other_count;
    ui32 request_error;
    ui32 request_timeout;
    ui32 request_buffer_overflow;
    //�� �������
    ui64 queue_in;
    ui64 queue_up;
    ui32 queue_count;
    ui64 queue_lost;
    ui64 queue_out;
    ui64 queue_in_rqst;
    ui64 queue_out_rqst;

    TRPLStatDataItem() {
        Clear();
    }

    void IncAllMailCount() {
        all_mail_count = IncMax32(all_mail_count, 1);
    }
    void IncExcludeHostsMailCount() {
        excludehosts_mail_count = IncMax32(excludehosts_mail_count, 1);
    }
    void IncConnectOK() {
        connect_ok = IncMax32(connect_ok, 1);
    }
    void IncConnectError() {
        connect_error = IncMax32(connect_error, 1);
    }
    void IncConnectTimeout() {
        connect_timeout = IncMax32(connect_timeout, 1);
    }
    void IncRequest200Count() {
        request_200_count = IncMax32(request_200_count, 1);
    }
    void IncRequest4xxCount() {
        request_4xx_count = IncMax32(request_4xx_count, 1);
    }
    void IncRequest5xxCount() {
        request_5xx_count = IncMax32(request_5xx_count, 1);
    }
    void IncRequestOtherCount() {
        request_other_count = IncMax32(request_other_count, 1);
    }
    void IncRequestError() {
        request_error = IncMax32(request_error, 1);
    }
    void IncRequestTimeout() {
        request_timeout = IncMax32(request_timeout, 1);
    }
    void IncRequsetBufferOverflow() {
        request_buffer_overflow = IncMax32(request_buffer_overflow, 1);
    }

    void Clear() {
        all_mail_count = 0;
        excludehosts_mail_count = 0;
        connect_ok = 0;
        connect_error = 0;
        connect_timeout = 0;
        request_200_count = 0;
        request_4xx_count = 0;
        request_5xx_count = 0;
        request_other_count = 0;
        request_error = 0;
        request_timeout = 0;
        request_buffer_overflow = 0;
        queue_in = 0;
        queue_up = 0;
        queue_count = 0;
        queue_lost = 0;
        queue_out = 0;
        queue_in_rqst = 0;
        queue_out_rqst = 0;
    }

    TRPLStatDataItem& operator+=(const TRPLStatDataItem& obj) {
        this->all_mail_count = IncMax32(this->all_mail_count, obj.all_mail_count);
        this->excludehosts_mail_count = IncMax32(this->excludehosts_mail_count, obj.excludehosts_mail_count);
        this->connect_ok = IncMax32(this->connect_ok, obj.connect_ok);
        this->connect_error = IncMax32(this->connect_error, obj.connect_error);
        this->connect_timeout = IncMax32(this->connect_timeout, obj.connect_timeout);
        this->request_200_count = IncMax32(this->request_200_count, obj.request_200_count);
        this->request_4xx_count = IncMax32(this->request_4xx_count, obj.request_4xx_count);
        this->request_5xx_count = IncMax32(this->request_5xx_count, obj.request_5xx_count);
        this->request_other_count = IncMax32(this->request_other_count, obj.request_other_count);
        this->request_error = IncMax32(this->request_error, obj.request_error);
        this->request_timeout = IncMax32(this->request_timeout, obj.request_timeout);
        this->request_buffer_overflow = IncMax32(this->request_buffer_overflow, obj.request_buffer_overflow);

        this->queue_in = IncMax64(this->queue_in, obj.queue_in);
        this->queue_up = IncMax64(this->queue_up, obj.queue_up);
        this->queue_count = IncMax32(this->queue_count, obj.queue_count);
        this->queue_lost = IncMax64(this->queue_lost, obj.queue_lost);
        this->queue_out = IncMax64(this->queue_out, obj.queue_out);
        this->queue_in_rqst = IncMax64(this->queue_in_rqst, obj.queue_in_rqst);
        this->queue_out_rqst = IncMax64(this->queue_out_rqst, obj.queue_out_rqst);

        return *this;
    }

    TRPLStatDataItem& operator-=(const TRPLStatDataItem& obj) {
        this->all_mail_count = Minus32(this->all_mail_count, obj.all_mail_count);
        this->excludehosts_mail_count = Minus32(this->excludehosts_mail_count, obj.excludehosts_mail_count);
        this->connect_ok = Minus32(this->connect_ok, obj.connect_ok);
        this->connect_error = Minus32(this->connect_error, obj.connect_error);
        this->connect_timeout = Minus32(this->connect_timeout, obj.connect_timeout);
        this->request_200_count = Minus32(this->request_200_count, obj.request_200_count);
        this->request_4xx_count = Minus32(this->request_4xx_count, obj.request_4xx_count);
        this->request_5xx_count = Minus32(this->request_5xx_count, obj.request_5xx_count);
        this->request_other_count = Minus32(this->request_other_count, obj.request_other_count);
        this->request_error = Minus32(this->request_error, obj.request_error);
        this->request_timeout = Minus32(this->request_timeout, obj.request_timeout);
        this->request_buffer_overflow = Minus32(this->request_buffer_overflow, obj.request_buffer_overflow);

        this->queue_in = Minus64(this->queue_in, obj.queue_in);
        this->queue_up = Minus64(this->queue_up, obj.queue_up);
        this->queue_count = Minus32(this->queue_count, obj.queue_count);
        this->queue_lost = Minus64(this->queue_lost, obj.queue_lost);
        this->queue_out = Minus64(this->queue_out, obj.queue_out);
        this->queue_in_rqst = Minus64(this->queue_in_rqst, obj.queue_in_rqst);
        this->queue_out_rqst = Minus64(this->queue_out_rqst, obj.queue_out_rqst);

        return *this;
    }
};

struct TRPLStatData {
    ui16 threads;
    bool enable_client;
    bool old;
    TRPLStatDataItem today;
    TRPLStatDataItem yesterday;
    TString host;
    int port;
    TString remote_srvid;
    bool itself;
    int elapsedtime;
    ui32 max_queue_size;
    ui32 max_shingles_in_request;
    ui32 min_shingles_in_request;
    ui32 timeout_create_sendblock;
    ui64 need_return_data_time_today;
    ui64 need_return_data_count_today;
    ui64 need_return_data_time_yesterday;
    ui64 need_return_data_count_yesterday;
    ui32 inqueue_count;

    TRPLStatData() {
        Clear();
    }

    void Midnight() {
        yesterday = today;
        today.Clear();
    }
    TRPLStatData& operator+=(const TRPLStatData& obj) {
        this->today += obj.today;
        this->yesterday += obj.yesterday;

        return *this;
    }

    TRPLStatData& operator-=(const TRPLStatData& obj) {
        this->today -= obj.today;
        this->yesterday -= obj.yesterday;

        return *this;
    }

    void Clear() {
        threads = 0;
        enable_client = false;
        old = false;
        host = "";
        port = 0;
        remote_srvid = "";
        itself = false;
        elapsedtime = 0;
        max_queue_size = 0;
        max_shingles_in_request = 0;
        min_shingles_in_request = 0;
        timeout_create_sendblock = 0;
        need_return_data_time_today = 0;
        need_return_data_count_today = 0;
        need_return_data_time_yesterday = 0;
        need_return_data_count_yesterday = 0;
        inqueue_count = 0;
        today.Clear();
        yesterday.Clear();
    }
};

struct TRPLStatDataSummary {
    TRPLStatData data[MAX_DIRECTS_COUNT];
    int max_count;
};

struct TKWStrokaRPL {
    TString host;
    TString url;
    TString action;

    TKWStrokaRPL() {
        host = "";
        url = "";
        action = "";
    }
};

//*******************************************************************************************************************************************
//                                                             TItselfClass
//*******************************************************************************************************************************************

class TItselfClass {
private:
    static const ui32 REFRASH_PERIOD_SEC = 60;

private:
    time_t m_last_set_itself;
    bool m_itself;
    TString m_remote_srv_id;
    TMutex m_Mutex;

public:
    TItselfClass() {
        m_last_set_itself = 0;
        m_itself = false;
        m_remote_srv_id = "";
    }

    ~TItselfClass() {
    }

    void SetItSelf(const TString& srv_id) {
        m_Mutex.Acquire();

        m_itself = true;
        m_last_set_itself = time(NULL);
        m_remote_srv_id = srv_id;

        m_Mutex.Release();
    }

    void SetSrvID(const TString& srv_id) {
        m_Mutex.Acquire();

        m_remote_srv_id = srv_id;

        m_Mutex.Release();
    }

    TString GetRemoteServerID() {
        return m_remote_srv_id;
    }

    bool ItSelfStat() {
        bool res = false;

        res = m_itself;

        return res;
    }

    bool ItSelf() {
        bool res = false;

        if (m_itself) {
            ui32 current_time = time(NULL);
            if (current_time < m_last_set_itself) {
                m_Mutex.Acquire();

                m_itself = false;
                m_last_set_itself = 0;

                m_Mutex.Release();

                res = false;

            } else if ((current_time - m_last_set_itself) > REFRASH_PERIOD_SEC) {
                m_Mutex.Acquire();

                m_itself = false;

                m_Mutex.Release();

                res = false;

            } else
                res = true;
        }

        return res;
    }

    ui32 GetElapsedTime() {
        ui32 res = 0;
        ui32 current_time = time(NULL);

        if (current_time > m_last_set_itself)
            res = current_time - m_last_set_itself;

        return res;
    }
};

//*******************************************************************************************************************************************
//                                                             TRPLPoolThreads
//*******************************************************************************************************************************************

class TRPLPoolThreads {
private:
    static const ui32 RCV_BUFFER_SIZE = 65535;

    int dirnumb;
    bool old;   // осуществлять репликацию по старому протоколу
    NCurl::TCurl curl;
    char* rcv_buffer;
    TLogsGroup* LogsGroup;
    void* m_GeneralObject;
    TString m_server_id;
    TMutex m_HostDataMutex;
    TString m_host;
    TString m_url;
    TDuration m_ConnectTimeOut = TDuration::MilliSeconds(100), m_GetTimeOut = TDuration::MilliSeconds(200);
    TString m_action;
    ui16 m_port;
    TString m_fullhost;
    TRPLStatData m_statdata;
    TItselfClass* itself_obj;

    TKWStrokaRPL GetHost(const TString& host);
    TString ModifyResponce(const TString& s);
    TString GetRequestNew(TShingleExtInfoRPLList& shinglelist);
    TString GetRequestOld(TShingleExtInfoRPLList& shinglelist);
    void PrintFollowInfo(TShingleExtInfoRPLList& shinglelist, const TString& text);

public:
    TRPLPoolThreads();
    ~TRPLPoolThreads();

    void Init(int dirnumbA, bool oldA, const TString& server_id, void* GeneralObjectA, TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, TLogsGroup* LogsGroupA, TItselfClass* itself_objA);
    void SetHostData(TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec);
    void GetRequestToRemoteServer(TShingleExtInfoRPLList& shinglelist, TPrintBuff& PBUFF);
    TString GetRequest(TShingleExtInfoRPLList& shinglelist);
    TRPLStatData GetStat();
    TString GetExampleRequest();
    TString GetShortURL();
    void Midnight();
    bool NoDefinedHostOrPort();
};

//*******************************************************************************************************************************************
//                                                             TRPLPoolDirect
//*******************************************************************************************************************************************

typedef std::vector<TRPLPoolThreads*> TRPLPoolThreadsList;
typedef TRPLPoolThreadsList::iterator TRPLPoolThreadsListIt;

typedef std::vector<TThread*> TRPLThreadList;
typedef TRPLThreadList::iterator TRPLThreadListIt;

class TRPLPoolDirect {
private:
    static const ui32 MAX_THREAD_COUNT = 32;
    static const ui32 MAX_QUEUE_SIZE = 1000;

private:
    int dirnumb;
    TLogsGroup* LogsGroup;
    int thread_count;
    bool enable;    // работает или нет
    bool old;       // осуществлять репликацию по старому протоколу
    TString m_server_id;
    void* m_GeneralObject;
    TItselfClass itself_obj;

    TString m_host;
    ui16 m_port;
    ui32 m_connecttimeout;
    ui32 m_requesttimeout;

    int m_real_thread_count;
    bool m_stop_threads;
    TMutex m_MutexThreads;

    TRPLPoolThreadsList itemlist;
    TRPLThreadList threadlist;
    TRPLQueue queue;

public:
    int IncrementThreadCount();     // инкрементировать кол-во работающих потоков
    void DecrementThreadCount();    // декрементировать кол-во работающих потоков
    void ResetStopThreads();        // сбросить сигнал остановки потоков
    void SetStopThreads();          // послать сигнал на остановку работы потоков
    bool IsShouldStopThreads();     // true - нужно завершить работу в потоке
    bool IsStopThreads();           // true - все потоки остановились (счетчик обнулился)
    int ReadThreadsCount();         // реальное число запущенных в работу потоков
    void WriteThreadIDToLog(int thread_index);
    void PrintFollowInfo(TShingleExtInfoRPLList& shinglelist, const TString& text);

    bool ObrabRequest(int index, TPrintBuff& PBUFF);

public:
    TRPLPoolDirect();
    ~TRPLPoolDirect();

    bool InitBeforeFork(int dirnumbA, const TString& server_id, TLogsGroup* LogsGroupA, bool enableA, int thread_countA, const TString& hostA, int portA, int connecttimeoutA, int requesttimeoutA, bool oldA, ui32 max_queue_sizeA, ui32 max_shingles_in_requestA, ui32 min_shingles_in_requestA, ui32 timeout_create_sendblockA);
    bool InitAfterFork(void* GeneralObjectA);
    void Midnight();
    void Shutdown();
    void EventTick();
    TRPLStatData GetStat();
    TString GetExampleRequest();
    TString GetShortURL();

    void AddRequest(TShingleExtInfoRPLList& shinglelist);
};

//*******************************************************************************************************************************************
//                                                             TRPLMain
//*******************************************************************************************************************************************

class TRPLMain {
private:
    TRPLPoolDirect* directs_data[MAX_DIRECTS_COUNT];
    TLogsGroup* LogsGroup;

public:
    TRPLMain();
    ~TRPLMain();

    bool InitBeforeFork(const TString& server_id, TKConfig* configA, TLogsGroup* LogsGroupA);
    bool InitAfterFork(void* GeneralObjectA);
    void Midnight();
    void Shutdown();
    void EventTick();
    TRPLStatDataSummary GetStat();
    TString GetExampleRequest(int type);
    void AddRequest(TShingleExtInfoRPLList& shinglelist);
    void AddRequest(TShingleExtInfoRPL shingle_data, const TString& debugstr);
};

//*******************************************************************************************************************************************
