#include <util/folder/filelist.h>
#include <util/folder/path.h>
#include <util/string/reverse.h>
#include "tshinglestorage.h"

TString TCleanupTypeToStroka(TCleanupType type) {
    TString res = "";

    switch (type) {
        case CT_NEW:
            res = "CT_NEW";
            break;
        case CT_DEL:
            res = "CT_DEL";
            break;
        default:
            res = "unknown";
    };

    return res;
}

//********************************************************************************************************************************
//                                                TTimeDistributedWriteClass
//********************************************************************************************************************************

TTimeDistributedWriteClass::TTimeDistributedWriteClass() {
    m_part_count = 0;
    m_period_sec = 0;
    for (int i = 0; i < MAX_COUNT_SHINGLES_TYPE; i++)
        m_lastwritebuffer[i] = 0;
}

TTimeDistributedWriteClass::~TTimeDistributedWriteClass() {
}

void TTimeDistributedWriteClass::Init(ui32 part_count, ui32 period_sec) {
    m_part_count = part_count;
    m_period_sec = period_sec;

    if (m_part_count > MAX_COUNT_SHINGLES_TYPE)
        m_part_count = MAX_COUNT_SHINGLES_TYPE;

    if ((m_part_count > 0) && (m_period_sec > 0)) {
        ui32 currenttime = time(NULL);
        for (int i = (m_part_count - 1); i >= 0; i--)
            m_lastwritebuffer[i] = period_sec + currenttime - (m_part_count - i - 1);

        m_lastwrite = period_sec + currenttime;
    }
}

// каждый истекший период m_period_sec мы ищем минимальное время в массиве (самый старый writedump) и возвращаем его индекс
int TTimeDistributedWriteClass::NeedWriteDump() {
    int res = -1;
    ui32 currenttime = time(NULL);
    ui32 min_value = 0;
    int index_min_value = -1;

    if ((m_part_count > 0) && (m_period_sec > 0) && ((currenttime - m_lastwrite >= m_period_sec))) {
        m_Mutex.Acquire();

        index_min_value = 0;
        min_value = m_lastwritebuffer[index_min_value];
        for (size_t i = 0; i < m_part_count; i++) {
            if (m_lastwritebuffer[i] < min_value) {
                min_value = m_lastwritebuffer[i];
                index_min_value = i;
            }
        }
        if ((index_min_value >= 0) && (index_min_value < m_part_count)) {
            m_lastwritebuffer[index_min_value] = currenttime;
            res = index_min_value;
            m_lastwrite = currenttime;
        }

        m_Mutex.Release();
    }

    return res;
}

//********************************************************************************************************************************
//                                                TDeleteObjectMain
//********************************************************************************************************************************

TDeleteObjectMain::TDeleteObjectMain() {
    LogsGroup = NULL;
}

TDeleteObjectMain::~TDeleteObjectMain() {
    DeleteAllData();
}

void TDeleteObjectMain::Init(TLogsGroup* LogsGroupA) {
    LogsGroup = LogsGroupA;
}

void TDeleteObjectMain::Add(TDeleteStruct ds) {
    m_Mutex.Acquire();

    datalist.push_back(ds);

    m_Mutex.Release();
}

bool TDeleteObjectMain::DeleteAllData() {
    bool res = true;
    TDeleteStructListIt it;
    TDeleteStruct ds;
    ui32 today_record_count = 0;
    ui32 yest_record_count = 0;
    ui32 heavy_record_count = 0;
    ui32 today_time = 0;
    ui32 yest_time = 0;
    ui32 heavy_time = 0;
    ui32 full_time = 0;

    it = datalist.begin();
    while (it != datalist.end()) {
        ds = (*it);
        m_Mutex.Acquire();

        datalist.erase(it++);

        m_Mutex.Release();

        full_time = CShingleTime::GetMs();

        today_time = CShingleTime::GetMs();
        if (ds.ordinary_today != NULL) {
            today_record_count = ds.ordinary_today->size();
            delete ds.ordinary_today;
            ds.ordinary_today = NULL;
        }
        today_time = CShingleTime::GetMs() - today_time;

        yest_time = CShingleTime::GetMs();
        if (ds.ordinary_yesterday != NULL) {
            yest_record_count = ds.ordinary_yesterday->size();
            delete ds.ordinary_yesterday;
            ds.ordinary_yesterday = NULL;
        }
        yest_time = CShingleTime::GetMs() - yest_time;

        heavy_time = CShingleTime::GetMs();
        if (ds.heavy_hash != NULL) {
            heavy_record_count = ds.heavy_hash->size();
            delete ds.heavy_hash;
            ds.heavy_hash = NULL;
        }
        heavy_time = CShingleTime::GetMs() - heavy_time;

        full_time = CShingleTime::GetMs() - full_time;

        if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
            LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "CLEARHASH(t=%d,p=%d,op='%s') to %u msec (TH: %u to %u msec, YH: %u to %u msec; HH: %u to %u msec)", ds.type, ds.part, ds.op_ident.c_str(), full_time, today_record_count, today_time, yest_record_count, yest_time, heavy_record_count, heavy_time);
    }

    return res;
}

//********************************************************************************************************************************
//                                         TShingleStorageItem
//********************************************************************************************************************************

TShingleStorageItem::TShingleStorageItem() {
    m_shutdown = false;

    m_part = -1;
    m_type = -1;
    RPLPool = NULL;
    m_koef_div = TShingleStat::YESTERDAY_DIV_KOEF;

    m_heavy_treshold = HEAVY_TRESHOLD_DEFAULT;
    m_max_size_ordinary_hash = MAX_SIZE_PART_ORDINARYHASH_DEFAULT;
    m_save_treshold = SAVE_TRESHOLD_DEFAULT;
    m_cleanup_treshold = CLEANUP_TRESHOLD_DEFAULT;
    m_rpldiff_value = RPLDIFF_VALUE_DEFAULT;
    m_dumpfilename = "";
    m_cleanup_type = CT_NEW;
    m_shutdown_save_treshold = SHUTDOWN_SAVE_TRESHOLD_DEFAULT;
    m_today_cleanup_count = 0;

    LogsGroup = NULL;
    delobj = NULL;
    m_heavy_hash = NULL;
    m_ordinary_today = NULL;
    m_ordinary_yesterday = NULL;

    m_busy_ordinary = false;
    m_atomic_ord.Swap(0);

    m_last_cleanup = 0;
    m_last_midnight = 0;
    m_last_save = 0;
    m_last_cleanup_during = 0;
    m_last_midnight_during = 0;
    m_last_save_during = 0;

    m_need_cleanup = false;
    m_need_writedump = false;
    m_need_midnight = false;

    m_tick_busy = 0;
    m_busytype = TBT_UNKNOWN;

    m_followhash = NULL;

    m_lock_type = TD_NONE;
#ifdef SOSERV_DEBUG
    m_prev_timing.Clear();
#endif
}

TShingleStorageItem::~TShingleStorageItem() {
    if (m_ordinary_yesterday != NULL) {
        delete m_ordinary_yesterday;
        m_ordinary_yesterday = NULL;
    }
    if (m_ordinary_today != NULL) {
        delete m_ordinary_today;
        m_ordinary_today = NULL;
    }
    if (m_heavy_hash != NULL) {
        delete m_heavy_hash;
        m_heavy_hash = NULL;
    }
}

bool TShingleStorageItem::Init(int part, int type, const TString& dumpfilename, TLogsGroup* LogsGroupA, TDeleteObjectMain* delobjA, ui32 rpldiff_value, ui32 heavy_treshold, ui32 max_size_ohash, ui32 save_treshold, ui32 cleanup_treshold, TRPLMain* RPLPoolA, TFollowShingleHash* followhashA, ui32 koef_div, TBusyMonStat* busy_monstat, TCleanupType cleanup_type, ui32 shutdown_save_treshold) {
    bool res = true;

    m_part = part;
    m_type = type;
    RPLPool = RPLPoolA;
    m_koef_div = koef_div;
    m_busy_monstat = busy_monstat;
    busycacheobj.Init(busy_monstat);

    LogsGroup = LogsGroupA;
    delobj = delobjA;
    m_rpldiff_value = rpldiff_value;
    m_heavy_treshold = heavy_treshold;
    m_max_size_ordinary_hash = max_size_ohash;
    m_save_treshold = save_treshold;
    m_cleanup_treshold = cleanup_treshold;
    m_dumpfilename = dumpfilename;
    m_cleanup_type = cleanup_type;
    m_shutdown_save_treshold = shutdown_save_treshold;

    m_heavy_hash = new TShingleDataTEHash();
    m_ordinary_today = new TShingleDataSingleHash();
    m_ordinary_yesterday = new TShingleDataSingleHash();

    m_followhash = followhashA;

    return res;
}

void TShingleStorageItem::LockOrdinary(const TSOSERVDBG_LOCKTYPE lock_type) {
    m_atomic_ord.Increment();
    m_MutexOrdinary.Acquire();
    m_lock_type = lock_type;
}

void TShingleStorageItem::UnLockOrdinary() {
    m_lock_type = TD_NONE;
    m_MutexOrdinary.Release();
    m_atomic_ord.Decrement();
}

ui32 TShingleStorageItem::SetBusyOrdinary(bool busyA, ui32& during, TBusyType busytype) {
    ui32 res = 0;
    m_MutexOrdinaryBusy.Acquire();

    m_busy_ordinary = busyA;
    if (m_busy_ordinary) {
        m_tick_busy = CShingleTime::GetMs();
        during = 0;
        m_busytype = busytype;
    } else {
        res = CShingleTime::GetMs() - m_tick_busy;
        during = res;
        m_tick_busy = 0;
        m_busytype = TBT_UNKNOWN;
    }

    m_MutexOrdinaryBusy.Release();

    return res;
}

bool TShingleStorageItem::GetBusyOrdinary() {
    return m_busy_ordinary;
}

void TShingleStorageItem::LockHeavy() {
    m_MutexHeavy.Acquire();
}

void TShingleStorageItem::UnLockHeavy() {
    m_MutexHeavy.Release();
}

TOrdinaryPointers TShingleStorageItem::FindShingleInOrdinaryHash(ui64 shingle, ui32& tick_today, ui32& tick_yesterday) {
    TOrdinaryPointers res;
    TShingleDataSingle sds_tmp;

    tick_today = 0;
    tick_yesterday = 0;
    if (m_ordinary_today != NULL) {
        tick_today = CShingleTime::GetMs();
        res.t_it = m_ordinary_today->find(shingle);
        tick_today = CShingleTime::GetMs() - tick_today;
        if (res.t_it != m_ordinary_today->end()) {
            sds_tmp = (*res.t_it).second;
            res.exists_today = true;
            if (!(*res.t_it).second.DontHaveYesterdayData()) {
                if (m_ordinary_yesterday != NULL) {
                    tick_yesterday = CShingleTime::GetMs();
                    res.y_it = m_ordinary_yesterday->find(shingle);
                    tick_yesterday = CShingleTime::GetMs() - tick_yesterday;
                    if (res.y_it != m_ordinary_yesterday->end()) {
                        (*res.t_it).second.SetHaveYesterdayData();
                        res.exists_yesterday = true;
                    } else {
                        (*res.t_it).second.SetDontHaveYesterdayData();
                    }
                }
            }
        }
    }
    if (!res.exists_today && !res.exists_yesterday) {
        if (m_ordinary_yesterday != NULL) {
            tick_yesterday = CShingleTime::GetMs();
            res.y_it = m_ordinary_yesterday->find(shingle);
            tick_yesterday = CShingleTime::GetMs() - tick_yesterday;
            if (res.y_it != m_ordinary_yesterday->end())
                res.exists_yesterday = true;
        }
    }

    return res;
}

TShingleStat TShingleStorageItem::Get(const TString& abonent_ip, ui64 shingle, bool is_second) {
    TShingleStat res;
    TShingleDataTEHashIt it;
    TShingleDataSingle* t_data = NULL;
    TShingleDataSingle* y_data = NULL;
    TOrdinaryPointers ord_point;
    ui32 pr = 0;
    ui32 tick_today = 0;
    ui32 tick_yesterday = 0;
    ui32 tick_findfsh = 0;

#ifdef SOSERV_DEBUG
    res.debuginfo.SetSummary_start();
#endif

    res.CalcAgeKoef(m_last_midnight, m_koef_div);

    if (GetBusyOrdinary()) // хэш занят, отвечаем из кэша тяжелых шинглов
    {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetLockHeavy_start();
#endif
        LockHeavy();
#ifdef SOSERV_DEBUG
        res.debuginfo.SetLockHeavy_stop();
#endif

        if (m_heavy_hash != NULL) {
#ifdef SOSERV_DEBUG
            res.debuginfo.SetFindHeavy_start();
#endif
            it = m_heavy_hash->find(shingle);
#ifdef SOSERV_DEBUG
            res.debuginfo.SetFindHeavy_stop();
#endif
            if (it != m_heavy_hash->end()) {
                res.SetShingleData((*it).second, true, pr);
#ifdef SOSERV_DEBUG
                res.debuginfo.SetBusyAddStatHeavy_start();
#endif
                busycacheobj.AddBusygetStatFindHeavy();
#ifdef SOSERV_DEBUG
                res.debuginfo.SetBusyAddStatHeavy_start();
#endif
            }
        }

        UnLockHeavy();

        res.SetAction(ABUSY);

#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyAddStat_start();
#endif
        busycacheobj.AddBusygetStat();
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyAddStat_stop();
#endif

    } else {
// ищем в паре хэшей heavy (today/yesterday)
#ifdef SOSERV_DEBUG
        res.debuginfo.SetLockType(m_lock_type);
        res.debuginfo.SetLock_start();
#endif
        LockOrdinary(TD_GET);
#ifdef SOSERV_DEBUG
        res.debuginfo.SetLock_stop();
        res.debuginfo.AddPrevRqstTiming(m_prev_timing);
#endif

        // if ( (m_type == 2) || (m_type == 26) )
        //    Sleep(60000);

        ord_point = FindShingleInOrdinaryHash(shingle, tick_today, tick_yesterday);
#ifdef SOSERV_DEBUG
        res.debuginfo.SetFindToday(tick_today);
        res.debuginfo.SetFindYesterday(tick_yesterday);
#endif
        if (ord_point.exists_today)
            t_data = &(*(ord_point.t_it)).second;

        if (ord_point.exists_yesterday)
            y_data = &(*(ord_point.y_it)).second;

        if (ord_point.exists_today || ord_point.exists_yesterday) {
            if (t_data != NULL)
                pr = t_data->GetPr();

            res.SetShingleData(TShingleDataTE(t_data, y_data), false, pr);
            res.SetAction(AEXISTS);

        } else {
            res.SetAction(ANOTEXISTS);
        }

#ifdef SOSERV_DEBUG
        m_prev_timing.SetFindToday(res.debuginfo.GetFindToday());
        m_prev_timing.SetFindYesterday(res.debuginfo.GetFindYesterday());
#endif

        UnLockOrdinary();
    }

    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh)) {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetWriteLogFollowShingle_start();
#endif
        if (is_second)
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d GETs     RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, res.GetFollowInfo().c_str());
        else
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d GET      RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, res.GetFollowInfo().c_str());
#ifdef SOSERV_DEBUG
        res.debuginfo.SetWriteLogFollowShingle_stop();
#endif
    }

#ifdef SOSERV_DEBUG
    res.debuginfo.SetFindFollowShingle(tick_findfsh);
    res.debuginfo.SetSummary_stop();
#endif

    return res;
}

TShingleStat TShingleStorageItem::PutA(const TString& abonent_ip, ui64 shingle, ui16 count, TShingleSpamType spamtype, ui64 att_shingle, int att_shingle_type, TPersType pers, ui32 heavytreshold, const TString& debugstr, bool no_replic) {
    TShingleStat res;
    TShingleDataTEHashIt it;
    TShingleDataSingle* t_data = NULL;
    TShingleDataSingle* y_data = NULL;
    TOrdinaryPointers ord_point;
    std::pair<TShingleDataSingleHashIt, bool> insres;
    bool need_rpl = false;
    TShingleExtInfoRPL rpldata = TShingleExtInfoRPL();
    bool follow_shingle = false;
    bool exist_today = false;
    bool exist_yesterday = false;
    ui32 pr = 0;
    ui32 tick_today = 0;
    ui32 tick_yesterday = 0;
    ui32 tick_findfsh = 0;
    ui32 tick_findfsh1 = 0;

#ifdef SOSERV_DEBUG
    res.debuginfo.SetSummary_start();
#endif

    res.CalcAgeKoef(m_last_midnight, m_koef_div);

    follow_shingle = ExistsFollowShingle(shingle, tick_findfsh);
    if ((att_shingle != 0) && (!follow_shingle))
        follow_shingle = ExistsFollowShingle(att_shingle, tick_findfsh1);
#ifdef SOSERV_DEBUG
    res.debuginfo.SetFindFollowShingle(tick_findfsh + tick_findfsh1);
#endif

// ищем в паре хэшей heavy (today/yesterday)
#ifdef SOSERV_DEBUG
    res.debuginfo.SetLockType(m_lock_type);
    res.debuginfo.SetLock_start();
#endif
    LockOrdinary(TD_PUTA);
#ifdef SOSERV_DEBUG
    res.debuginfo.SetLock_stop();
    res.debuginfo.AddPrevRqstTiming(m_prev_timing);
#endif

    ord_point = FindShingleInOrdinaryHash(shingle, tick_today, tick_yesterday);
#ifdef SOSERV_DEBUG
    res.debuginfo.SetFindToday(tick_today);
    res.debuginfo.SetFindYesterday(tick_yesterday);
#endif
    if (ord_point.exists_today) {
        exist_today = true;
        t_data = &(*(ord_point.t_it)).second;
        if (t_data != NULL)
            t_data->Add(spamtype, count, pers);

    } else {
        // создаем today шингл
        exist_today = false;
        if (m_ordinary_today != NULL) {
#ifdef SOSERV_DEBUG
            res.debuginfo.SetCreateShingle_start();
#endif
            insres = m_ordinary_today->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(spamtype, count, pers)));
#ifdef SOSERV_DEBUG
            res.debuginfo.SetCreateShingle_stop();
#endif
            ord_point.t_it = insres.first;
            if (ord_point.t_it != m_ordinary_today->end()) {
                ord_point.exists_today = true;
                t_data = &(*(ord_point.t_it)).second;
            }

            if (m_ordinary_today->size() >= m_max_size_ordinary_hash)
                SetCleanupEvent();
        }
    }
    if (ord_point.exists_yesterday) {
        exist_yesterday = true;
        y_data = &(*(ord_point.y_it)).second;
    }

    // если превышен порог репликации, передаем данные по шинглу (только для шинглов без att_shingle,
    // для них репликация обрабатывается выше уровнем)
    if ((!no_replic) && (RPLPool != NULL) && (ord_point.exists_today) && (t_data != NULL)) {
        ui32 summdiffvalue = 0;

        summdiffvalue = t_data->DiffHam() + t_data->DiffSpam() + t_data->DiffMalic() /* + t_data->DiffPersHam() + t_data->DiffPersSpam()*/;
        if ((summdiffvalue >= m_rpldiff_value) || (att_shingle != 0)) {
            rpldata = TShingleExtInfoRPL(shingle, m_type, t_data->DiffHam(), t_data->DiffMalic(), t_data->DiffSpam(), t_data->DiffPersHam(), t_data->DiffPersSpam(), att_shingle, att_shingle_type, spamtype, pers, follow_shingle);
            need_rpl = true;
            t_data->ClearDiffs();
        }
    }

    // формируем ответ
    if (ord_point.exists_today || ord_point.exists_yesterday) {
        if (t_data != NULL)
            pr = t_data->GetPr();

        res.SetShingleData(TShingleDataTE(t_data, y_data), false, pr);
        res.SetAction(AEXISTS);

        // если шингл тяжелый, копируем его (или обновляем данные) в хэш тяжелых шинглов
        if (res.CalcHeavyStatus(heavytreshold)) {
#ifdef SOSERV_DEBUG
            res.debuginfo.SetLockHeavy_start();
#endif
            LockHeavy();
#ifdef SOSERV_DEBUG
            res.debuginfo.SetLockHeavy_stop();
#endif

            if (m_heavy_hash != NULL) {
#ifdef SOSERV_DEBUG
                res.debuginfo.SetFindHeavy_start();
#endif
                it = m_heavy_hash->find(shingle);
#ifdef SOSERV_DEBUG
                res.debuginfo.SetFindHeavy_stop();
#endif
                if (it != m_heavy_hash->end()) {
                    (*it).second = res.GetShingleData();

                } else {
#ifdef SOSERV_DEBUG
                    res.debuginfo.SetAddHeavy_start();
#endif
                    m_heavy_hash->insert(TShingleDataTEHash::value_type(shingle, res.GetShingleData()));
#ifdef SOSERV_DEBUG
                    res.debuginfo.SetAddHeavy_stop();
#endif
                }
            }

            UnLockHeavy();
        }
    }

    res.SetExistToday(exist_today);
    res.SetExistYesterday(exist_yesterday);

#ifdef SOSERV_DEBUG
    m_prev_timing.SetFindToday(res.debuginfo.GetFindToday());
    m_prev_timing.SetFindYesterday(res.debuginfo.GetFindYesterday());
    m_prev_timing.SetCreateShingle(res.debuginfo.GetCreateShingle());
    m_prev_timing.SetLockHeavy(res.debuginfo.GetLockHeavy());
    m_prev_timing.SetFindHeavy(res.debuginfo.GetFindHeavy());
    m_prev_timing.SetAddHeavy(res.debuginfo.GetAddHeavy());
#endif

    UnLockOrdinary();

    if (need_rpl)   // требуется репликация шингла
    {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetAddToRplPool_start();
#endif
        RPLPool->AddRequest(rpldata, debugstr);
#ifdef SOSERV_DEBUG
        res.debuginfo.SetAddToRplPool_stop();
#endif
        m_replic_shingles_count.AddRequest();

        // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
        if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(rpldata.shingle, tick_findfsh)) {
#ifdef SOSERV_DEBUG
            res.debuginfo.SetWriteLogFollowShingleRpl_start();
#endif
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d ADDRPL   [%s]", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), rpldata.shingle, rpldata.type, m_part, rpldata.value.GetFollowInfo().c_str());
#ifdef SOSERV_DEBUG
            res.debuginfo.SetWriteLogFollowShingleRpl_stop();
#endif
        }
#ifdef SOSERV_DEBUG
        res.debuginfo.SetFindFollowShingleRpl(tick_findfsh);
#endif
    }

#ifdef SOSERV_DEBUG
    res.debuginfo.SetSummary_stop();
#endif

    return res;
}

TShingleStat TShingleStorageItem::PutMA(ui64 shingle, ui16 ham, ui16 dlv, ui16 spam, ui16 persham, ui16 persspam, ui64 att_shingle, int att_shingle_type, ui32 heavytreshold) {
    TShingleStat res;
    TShingleDataTEHashIt it;
    TShingleDataSingle* t_data = NULL;
    TShingleDataSingle* y_data = NULL;
    TOrdinaryPointers ord_point;
    std::pair<TShingleDataSingleHashIt, bool> insres;
    bool exist_today = false;
    bool exist_yesterday = false;
    ui32 pr = 0;
    ui32 tick_today = 0;
    ui32 tick_yesterday = 0;

#ifdef SOSERV_DEBUG
    res.debuginfo.SetSummary_start();
#endif

    res.CalcAgeKoef(m_last_midnight, m_koef_div);

// ищем в паре хэшей heavy (today/yesterday)
#ifdef SOSERV_DEBUG
    res.debuginfo.SetLockType(m_lock_type);
    res.debuginfo.SetLock_start();
#endif
    LockOrdinary(TD_PUTMA);
#ifdef SOSERV_DEBUG
    res.debuginfo.SetLock_stop();
    res.debuginfo.AddPrevRqstTiming(m_prev_timing);
#endif

    ord_point = FindShingleInOrdinaryHash(shingle, tick_today, tick_yesterday);
#ifdef SOSERV_DEBUG
    res.debuginfo.SetFindToday(tick_today);
    res.debuginfo.SetFindYesterday(tick_yesterday);
#endif
    if (ord_point.exists_today) {
        exist_today = true;
        t_data = &(*(ord_point.t_it)).second;
        if (t_data != NULL)
            t_data->AddWODiff(ham, dlv, spam, persham, persspam);
    } else {
        // создаем today шингл
        exist_today = false;
        if (m_ordinary_today != NULL) {
#ifdef SOSERV_DEBUG
            res.debuginfo.SetCreateShingle_start();
#endif
            insres = m_ordinary_today->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(ham, dlv, spam, persham, persspam, 0)));
#ifdef SOSERV_DEBUG
            res.debuginfo.SetCreateShingle_stop();
#endif
            ord_point.t_it = insres.first;
            if (ord_point.t_it != m_ordinary_today->end()) {
                ord_point.exists_today = true;
                t_data = &(*(ord_point.t_it)).second;
            }

            if (m_ordinary_today->size() >= m_max_size_ordinary_hash)
                SetCleanupEvent();
        }
    }
    if (ord_point.exists_yesterday) {
        exist_yesterday = true;
        y_data = &(*(ord_point.y_it)).second;
    }

    if (ord_point.exists_today || ord_point.exists_yesterday) {
        if (t_data != NULL)
            pr = t_data->GetPr();

        res.SetShingleData(TShingleDataTE(t_data, y_data), false, pr);
        res.SetAction(AEXISTS);

        if (res.CalcHeavyStatus(heavytreshold)) // шингл тяжелый, копируем его (или обновляем данные) в хэш тяжелых шинглов
        {
#ifdef SOSERV_DEBUG
            res.debuginfo.SetLockHeavy_start();
#endif
            LockHeavy();
#ifdef SOSERV_DEBUG
            res.debuginfo.SetLockHeavy_stop();
#endif

            if (m_heavy_hash != NULL) {
#ifdef SOSERV_DEBUG
                res.debuginfo.SetFindHeavy_start();
#endif
                it = m_heavy_hash->find(shingle);
#ifdef SOSERV_DEBUG
                res.debuginfo.SetFindHeavy_stop();
#endif
                if (it != m_heavy_hash->end()) {
                    (*it).second = res.GetShingleData();

                } else {
#ifdef SOSERV_DEBUG
                    res.debuginfo.SetAddHeavy_start();
#endif
                    m_heavy_hash->insert(TShingleDataTEHash::value_type(shingle, res.GetShingleData()));
#ifdef SOSERV_DEBUG
                    res.debuginfo.SetAddHeavy_stop();
#endif
                }
            }

            UnLockHeavy();
        }
    }

    res.SetExistToday(exist_today);
    res.SetExistYesterday(exist_yesterday);

#ifdef SOSERV_DEBUG
    m_prev_timing.SetFindToday(res.debuginfo.GetFindToday());
    m_prev_timing.SetFindYesterday(res.debuginfo.GetFindYesterday());
    m_prev_timing.SetCreateShingle(res.debuginfo.GetCreateShingle());
    m_prev_timing.SetLockHeavy(res.debuginfo.GetLockHeavy());
    m_prev_timing.SetFindHeavy(res.debuginfo.GetFindHeavy());
    m_prev_timing.SetAddHeavy(res.debuginfo.GetAddHeavy());
#endif

    UnLockOrdinary();

#ifdef SOSERV_DEBUG
    res.debuginfo.SetSummary_stop();
#endif

    return res;
}

TShingleStat TShingleStorageItem::SetMA(ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, ui64 att_shingle, int att_shingle_type, ui32 heavytreshold) {
    TShingleStat res;
    TShingleDataTEHashIt it;
    TShingleDataSingle* t_data = NULL;
    TShingleDataSingle* y_data = NULL;
    TOrdinaryPointers ord_point;
    std::pair<TShingleDataSingleHashIt, bool> insres;
    ui32 summ_data_t = 0;
    ui32 summ_data_y = 0;
    bool no_add_empty_data = false;
    ui32 pr = 0;
    ui32 tick_today = 0;
    ui32 tick_yesterday = 0;

    res.CalcAgeKoef(m_last_midnight, m_koef_div);

    // ищем в паре хэшей heavy (today/yesterday)
    LockOrdinary(TD_SETMA);

    ord_point = FindShingleInOrdinaryHash(shingle, tick_today, tick_yesterday);

    if (!ord_point.exists_today && !ord_point.exists_yesterday) // если в хэшах нет данный и приходящие данные тоже пусты, то не добавляем в хэш ничего
    {
        summ_data_t = t_ham + t_dlv + t_spam + t_persham + t_persspam;
        summ_data_y = y_ham + y_dlv + y_spam + y_persham + y_persspam;
        if ((summ_data_t == 0) && (t_weight == 0) && (summ_data_y == 0) && (y_weight == 0))
            no_add_empty_data = true;
    }

    if (!no_add_empty_data) {
        if (ord_point.exists_today) {
            t_data = &(*(ord_point.t_it)).second;
            if (t_data != NULL)
                t_data->Set(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight);
        } else {
            // создаем today шингл
            if (m_ordinary_today != NULL) {
                insres = m_ordinary_today->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight)));
                ord_point.t_it = insres.first;
                if (ord_point.t_it != m_ordinary_today->end()) {
                    ord_point.exists_today = true;
                    t_data = &(*(ord_point.t_it)).second;
                }

                if (m_ordinary_today->size() >= m_max_size_ordinary_hash)
                    SetCleanupEvent();
            }
        }
        if (ord_point.exists_yesterday) {
            y_data = &(*(ord_point.y_it)).second;
            if (y_data != NULL)
                y_data->Set(y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight);
        } else {
            // создаем yesterday шингл
            if (m_ordinary_today != NULL) {
                insres = m_ordinary_yesterday->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight)));
                ord_point.y_it = insres.first;
                if (ord_point.y_it != m_ordinary_yesterday->end()) {
                    ord_point.exists_yesterday = true;
                    y_data = &(*(ord_point.y_it)).second;
                }
            }
        }

        if (ord_point.exists_today || ord_point.exists_yesterday) {
            if (t_data != NULL)
                pr = t_data->GetPr();

            res.SetShingleData(TShingleDataTE(t_data, y_data), false, pr);
            res.SetAction(AEXISTS);

            if (res.CalcHeavyStatus(heavytreshold)) // шингл тяжелый, копируем его (или обновляем данные) в хэш тяжелых шинглов
            {
                LockHeavy();

                if (m_heavy_hash != NULL) {
                    it = m_heavy_hash->find(shingle);
                    if (it != m_heavy_hash->end()) {
                        (*it).second = res.GetShingleData();
                    } else
                        m_heavy_hash->insert(TShingleDataTEHash::value_type(shingle, res.GetShingleData()));
                }

                UnLockHeavy();
            }
        }
    }

    UnLockOrdinary();

    return res;
}

int TShingleStorageItem::SetWeightA(ui64 shingle, float weight) {
    int res = 0;

    // ищем в паре хэшей heavy (today/yesterday)
    LockOrdinary(TD_SETWEIGHTA);

    if (m_ordinary_today != NULL) {
        TShingleDataSingleHashIt it;

        it = m_ordinary_today->find(shingle);
        if (it != m_ordinary_today->end()) {
            (*it).second.SetWeight(weight);
            res = 2;

        } else {
            m_ordinary_today->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(weight)));
            res = 3;
        }
    }

    UnLockOrdinary();

    return res;
}

bool TShingleStorageItem::ExistsFollowShingle(ui64 shingle, ui32& tick) {
    bool res = false;

    tick = CShingleTime::GetMs();
    if (m_followhash != NULL) {
        if (m_followhash->find(0) != m_followhash->end())
            res = true;

        if ((!res) && (m_followhash->find(shingle) != m_followhash->end()))
            res = true;
    }
    tick = CShingleTime::GetMs() - tick;

    return res;
}

void TShingleStorageItem::WriteExitStep(const TString& text) {
    if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
        LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "EXITEVENT: %s", text.c_str());
        LogsGroup->GetServerLog()->FFlush();
    }
}

void TShingleStorageItem::ApplyBusyCacheEvent() {
    if (!busycacheobj.IsEmpty()) {
        TBusyCacheItem value;
        TShingleStat res;
        int resi = 0;
        bool shutdownev = false;
        ui32 lost_count = 0;
        ui32 tick_findfsh = 0;

        while (busycacheobj.Get(value)) {
            if (GetShutdownPriznak()) {
                lost_count = busycacheobj.GetStat().m_count;
                shutdownev = true;
                break;
            }

            switch (value.m_type) {
                case BC_PUT:
                    res = PutA("applebusycache", value.m_shingle, value.m_t_ham, (TShingleSpamType)value.m_t_dlv, value.m_att_shingle, value.m_att_shingle_type, (TPersType)value.m_t_spam, m_heavy_treshold, "", value.m_no_replic);

                    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(value.m_shingle, tick_findfsh))
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s applebusycache %016" PRIx64 "-%03d-%02d PUT(ab)  [cnt=%d, st=%s, atts=%016" PRIx64 "-%02d, pers=%s, norepl=%s] RES=\"%s\"", LogsGroup->GetServerID().c_str(), value.m_shingle, m_type, m_part, value.m_t_ham, TShingleSpamTypeToStroka((TShingleSpamType)value.m_t_dlv).c_str(), value.m_att_shingle, value.m_att_shingle_type, TPersTypeToStroka((TPersType)value.m_t_spam).c_str(), BoolToStroka2(value.m_no_replic).c_str(), res.GetFollowInfo().c_str());
                    break;
                case BC_PUTM:
                    res = PutMA(value.m_shingle, value.m_t_ham, value.m_t_dlv, value.m_t_spam, value.m_t_persham, value.m_t_persspam, value.m_att_shingle, value.m_att_shingle_type, m_heavy_treshold);

                    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(value.m_shingle, tick_findfsh))
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s applebusycache %016" PRIx64 "-%03d-%02d PUTM(ab) [h=%d, d=%d, s=%d, ph=%d, ps=%d, atts=%016" PRIx64 "-%02d] RES=\"%s\"", LogsGroup->GetServerID().c_str(), value.m_shingle, m_type, m_part, value.m_t_ham, value.m_t_dlv, value.m_t_spam, value.m_t_persham, value.m_t_persspam, value.m_att_shingle, value.m_att_shingle_type, res.GetFollowInfo().c_str());
                    break;
                case BC_SETM:
                    res = SetMA(value.m_shingle, value.m_t_ham, value.m_t_dlv, value.m_t_spam, value.m_t_persham, value.m_t_persspam, value.m_t_weight, value.m_y_ham, value.m_y_dlv, value.m_y_spam, value.m_y_persham, value.m_y_persspam, value.m_y_weight, value.m_att_shingle, value.m_att_shingle_type, m_heavy_treshold);

                    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(value.m_shingle, tick_findfsh))
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s applebusycache %016" PRIx64 "-%03d-%02d SETM(ab) [th=%d, td=%d, ts=%d, tph=%d, tps=%d, tw=%f, yh=%d, yd=%d, ys=%d, yph=%d, yps=%d, yw=%f, atts=%016" PRIx64 "-%02d] RES=\"%s\"", LogsGroup->GetServerID().c_str(), value.m_shingle, m_type, m_part, value.m_t_ham, value.m_t_dlv, value.m_t_spam, value.m_t_persham, value.m_t_persspam, value.m_t_weight, value.m_y_ham, value.m_y_dlv, value.m_y_spam, value.m_y_persham, value.m_y_persspam, value.m_y_weight, value.m_att_shingle, value.m_att_shingle_type, res.GetFollowInfo().c_str());
                    break;
                case BC_SETWEIGHT:
                    resi = SetWeightA(value.m_shingle, value.m_t_weight);

                    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(value.m_shingle, tick_findfsh))
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s applybusycache %016" PRIx64 "-%03d-%02d SETW(ab) [w=%f] RES=\"%d\"", LogsGroup->GetServerID().c_str(), value.m_shingle, m_type, m_part, value.m_t_weight, resi);
                    break;
            };
        }

        if (shutdownev)
            WriteExitStep("busyhash break: t=" + IntToStroka3(m_type) + ", p=" + IntToStroka2(m_part) + ", lost=" + IntToStroka(lost_count) + " record");
    }
}

TShingleStat TShingleStorageItem::Put(const TString& abonent_ip, ui64 shingle, ui16 count, TShingleSpamType spamtype, ui64 att_shingle, int att_shingle_type, TPersType pers, const TString& debugstr, bool no_replic, bool is_putm) {
    TShingleStat res;
    ui32 tick_findfsh = 0;

    if (GetBusyOrdinary())  // хэш занят, задание в очередь для выполнения, когда освободится
    {
        res.SetAction(ABUSY);

#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyAddStat_start();
#endif
        busycacheobj.Add(TBusyCacheItem(shingle, count, spamtype, att_shingle, att_shingle_type, pers, no_replic));
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyAddStat_stop();
#endif

    } else {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyApply_start();
#endif
        ApplyBusyCacheEvent();
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyApply_stop();
#endif
        res = PutA(abonent_ip, shingle, count, spamtype, att_shingle, att_shingle_type, pers, m_heavy_treshold, debugstr, no_replic);
    }

    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh)) {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetWriteLogFollowShingle2_start();
#endif
        if (is_putm)
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d PUTm     [cnt=%d, st=%s, atts=0x%" PRIx64 "-%02d, pers=%s, norepl=%s] RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, count, TShingleSpamTypeToStroka(spamtype).c_str(), att_shingle, att_shingle_type, TPersTypeToStroka(pers).c_str(), BoolToStroka2(no_replic).c_str(), res.GetFollowInfo().c_str());
        else
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d PUT      [cnt=%d, st=%s, atts=0x%" PRIx64 "-%02d, pers=%s, norepl=%s] RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, count, TShingleSpamTypeToStroka(spamtype).c_str(), att_shingle, att_shingle_type, TPersTypeToStroka(pers).c_str(), BoolToStroka2(no_replic).c_str(), res.GetFollowInfo().c_str());
#ifdef SOSERV_DEBUG
        res.debuginfo.SetWriteLogFollowShingle2_stop();
#endif
    }
#ifdef SOSERV_DEBUG
    res.debuginfo.SetFindFollowShingle2(tick_findfsh);
#endif

    return res;
}

TShingleStat TShingleStorageItem::PutM(const TString& abonent_ip, ui64 shingle, ui16 ham, ui16 dlv, ui16 spam, ui16 persham, ui16 persspam, ui64 att_shingle, int att_shingle_type) {
    TShingleStat res;
    ui32 tick_findfsh = 0;

    if (GetBusyOrdinary())  // хэш занят, задание в очередь для выполнения, когда освободится
    {
        res.SetAction(ABUSY);

#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyAddStat_start();
#endif
        busycacheobj.Add(TBusyCacheItem(shingle, ham, dlv, spam, persham, persspam, att_shingle, att_shingle_type));
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyAddStat_stop();
#endif

    } else {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyApply_start();
#endif
        ApplyBusyCacheEvent();
#ifdef SOSERV_DEBUG
        res.debuginfo.SetBusyApply_stop();
#endif
        res = PutMA(shingle, ham, dlv, spam, persham, persspam, att_shingle, att_shingle_type, m_heavy_treshold);
    }

    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh)) {
#ifdef SOSERV_DEBUG
        res.debuginfo.SetWriteLogFollowShingle2_start();
#endif
        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d PUTM     [h=%d, d=%d, s=%d, ph=%d, ps=%d, atts=0x%" PRIx64 "-%02d] RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, ham, dlv, spam, persham, persspam, att_shingle, att_shingle_type, res.GetFollowInfo().c_str());
#ifdef SOSERV_DEBUG
        res.debuginfo.SetWriteLogFollowShingle2_stop();
#endif
    }
#ifdef SOSERV_DEBUG
    res.debuginfo.SetFindFollowShingle2(tick_findfsh);
#endif

    return res;
}

TShingleStat TShingleStorageItem::SetM(const TString& abonent_ip, ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, ui64 att_shingle, int att_shingle_type) {
    TShingleStat res;
    ui32 tick_findfsh = 0;

    if (GetBusyOrdinary())  // хэш занят, задание в очередь для выполнения, когда освободится
    {
        res.SetAction(ABUSY);

        busycacheobj.Add(TBusyCacheItem(shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, att_shingle, att_shingle_type));

    } else {
        ApplyBusyCacheEvent();
        res = SetMA(shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, att_shingle, att_shingle_type, m_heavy_treshold);
    }

    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d SETM     [th=%d, td=%d, ts=%d, tph=%d, tps=%d, tw=%f, yh=%d, yd=%d, ys=%d, yph=%d, yps=%d, yw=%f, atts=0x%" PRIx64 "-%02d] RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, att_shingle, att_shingle_type, res.GetFollowInfo().c_str());

    return res;
}

int TShingleStorageItem::SetWeight(const TString& abonent_ip, ui64 shingle, float weight) {
    int res = 0;
    ui32 tick_findfsh = 0;

    if (GetBusyOrdinary())  // хэш занят, отвечаем из кэша тяжелых шинглов
    {
        res = 1;
        busycacheobj.Add(TBusyCacheItem(shingle, weight));

    } else {
        ApplyBusyCacheEvent();
        res = SetWeightA(shingle, weight);
    }

    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d SETW     [w=%f] RES=\"%d\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, weight, res);

    return res;
}

bool TShingleStorageItem::ClearShingleA(const TString& abonent_ip, ui64 shingle, bool& busy) {
    bool res = false;
    TShingleDataTEHashIt it;
    TShingleDataSingle* t_data = NULL;
    TShingleDataSingle* y_data = NULL;
    TOrdinaryPointers ord_point;
    ui32 tick_today = 0;
    ui32 tick_yesterday = 0;
    ui32 tick_findfsh = 0;

    if (GetBusyOrdinary())  // хэш занят, отвечаем из кэша тяжелых шинглов
    {
        busy = true;

    } else {
        busy = false;
        // ищем в паре хэшей heavy (today/yesterday)
        LockOrdinary(TD_CLEARSHINGLEA);

        ord_point = FindShingleInOrdinaryHash(shingle, tick_today, tick_yesterday);
        if (ord_point.exists_today) {
            t_data = &(*(ord_point.t_it)).second;
            t_data->Clear();
        }

        if (ord_point.exists_yesterday) {
            y_data = &(*(ord_point.y_it)).second;
            y_data->Clear();
        }

        res = true;

        UnLockOrdinary();

        // ищем и удаляем в хэше тяжелых шинглов
        LockHeavy();

        if (m_heavy_hash != NULL) {
            it = m_heavy_hash->find(shingle);
            if (it != m_heavy_hash->end())
                m_heavy_hash->erase(it);
        }

        UnLockHeavy();
    }

    // печатаем отладочную информацию по шинглам их хэша отслеживающих шинглов
    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh)) {
        TString restext = "";

        restext = BoolToStroka2(res);
        if (busy)
            restext = restext + ", BUSY";

        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d-%02d CLR      RES=\"%s\"", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), shingle, m_type, m_part, restext.c_str());
    }

    return res;
}

TSHTypePartStat TShingleStorageItem::GetStat() {
    TSHTypePartStat res;
    ui32 currenttime = time(NULL);

    m_MutexOrdinary.Acquire();

    if (m_ordinary_today != NULL)
        res.today_count = m_ordinary_today->size();

    if (m_ordinary_yesterday != NULL)
        res.yesterday_count = m_ordinary_yesterday->size();

    m_MutexOrdinary.Release();

    m_MutexHeavy.Acquire();

    if (m_heavy_hash != NULL)
        res.heavy_count = m_heavy_hash->size();

    m_MutexHeavy.Release();

    // защита для отображения очень старых клинапов
    if ((m_last_cleanup > 0) && ((currenttime - m_last_cleanup) > (3 * 86400)))
    {
        res.last_cleanup = 0;
        res.last_cleanup_during = 0;

    } else {
        res.last_cleanup = m_last_cleanup;
        res.last_cleanup_during = m_last_cleanup_during;
    }

    // защита для отображения очень старых миднайтов (по идее срабатывать не должна, т.к. миднайт каждые сутки)
    if ((m_last_midnight > 0) && ((currenttime - m_last_midnight) > (7 * 86400)))
    {
        res.last_midnight = 0;
        res.last_midnight_during = 0;

    } else {
        res.last_midnight = m_last_midnight;
        res.last_midnight_during = m_last_midnight_during;
    }

    // защита для отображения очень старых записей дампа на диск (по идее срабатывать не должна,
    // т.к. запись дампа на диск происходит несколько раз в сутки)
    if ((m_last_save > 0) && ((currenttime - m_last_save) > (7 * 86400)))
    {
        res.last_save = 0;
        res.last_save_during = 0;

    } else {
        res.last_save = m_last_save;
        res.last_save_during = m_last_save_during;
    }
    res.rpl_today_shingles_count = m_replic_shingles_count.GetTodayRequest();
    res.rpl_yesterday_shingles_count = m_replic_shingles_count.GetYesterdayRequest();
    res.today_cleanup_count = m_today_cleanup_count;
    res.busycachestat = busycacheobj.GetStat();
    if (m_busy_ordinary)
        res.busytype = m_busytype;
    else
        res.busytype = TBT_UNKNOWN;

    return res;
}

void TShingleStorageItem::ShedulerAction() {
    ui32 varv = 0;

    if (m_need_cleanup) {
        SetBusyOrdinary(true, varv, TBT_CLEANUP);
        usleep(10000);  // даем 10 мсек, чтобы очереди на мьютексах могли разгрестись
        Cleanup();
        SetBusyOrdinary(false, m_last_cleanup_during, TBT_UNKNOWN);

    } else if (m_need_midnight) {
        SetBusyOrdinary(true, varv, TBT_MIDNIGHT);
        usleep(10000);  // даем 10 мсек, чтобы очереди на мьютексах могли разгрестись
        Midnight();
        SetBusyOrdinary(false, m_last_midnight_during, TBT_UNKNOWN);

    } else if (m_need_writedump) {
        SetBusyOrdinary(true, varv, TBT_WRITEDUMP);
        usleep(10000);  // даем 10 мсек, чтобы очереди на мьютексах могли разгрестись
        PeriodWriteDump();
        SetBusyOrdinary(false, m_last_save_during, TBT_UNKNOWN);
    }
}

void TShingleStorageItem::SetWriteDumpEvent() {
    m_event_mutex.Acquire();
    m_need_writedump = true;
    m_event_mutex.Release();
}

void TShingleStorageItem::SetMidnightEvent() {
    m_event_mutex.Acquire();
    m_need_midnight = true;
    m_event_mutex.Release();
}

void TShingleStorageItem::SetCleanupEvent() {
    m_event_mutex.Acquire();
    m_need_cleanup = true;
    m_event_mutex.Release();
}

ui32 TShingleStorageItem::ShutdownWriteDump() {
    return WriteDumpA(m_shutdown_save_treshold);
}

ui32 TShingleStorageItem::PeriodWriteDump() {
    return WriteDumpA(m_save_treshold);
}

ui32 TShingleStorageItem::WriteDumpA(ui32 treshold) {
    ui32 res = 0;
    ui32 all_today_count = 0;
    ui32 all_yesterday_count = 0;
    ui32 all_heavy_count = 0;
    ui32 save_today_count = 0;
    ui32 save_yesterday_count = 0;
    ui32 save_heavy_count = 0;
    ui32 onlywrite = 0;
    ui32 onlywrite2 = 0;
    ui64 shingle = 0;
    ui32 tick_findfsh = 0;

    res = CShingleTime::GetMs();
    if (!m_dumpfilename.empty()) {
        FILE* handle = NULL;

        handle = fopen(m_dumpfilename.c_str(), "ab");
        if (handle != NULL) {
            LockOrdinary(TD_WRITEDUMPA);
            onlywrite = CShingleTime::GetMs();
            // last_midnight
            fprintf(handle, "M FFFFFFFFFFFFFFFF %u %d\n", (ui32)m_last_midnight, m_part);
            // m_last_cleanup
            fprintf(handle, "C FFFFFFFFFFFFFFFF %u %d\n", (ui32)m_last_cleanup, m_part);
            // today data
            if (m_ordinary_today != NULL) {
                TShingleDataSingleHashIt it;
                ui32 savetreshcalc = 0;
                char buff[256];

                all_today_count = m_ordinary_today->size();
                it = m_ordinary_today->begin();
                while (it != m_ordinary_today->end()) {
                    savetreshcalc = (*it).second.Ham() + (*it).second.Malic() + (*it).second.Spam() + (*it).second.PersHam() + (*it).second.PersSpam();

                    if ((savetreshcalc >= treshold) || ((*it).second.Weight() != 0)) {
                        shingle = (*it).first;
                        buff[sizeof(buff) - 1] = 0x00;
                        snprintf(buff, sizeof(buff) - 1, "T %016" PRIx64 " %d %d %d %d %d %5.2f\n", (*it).first, (*it).second.Ham(), (*it).second.Malic(), (*it).second.Spam(), (*it).second.PersHam(), (*it).second.PersSpam(), (*it).second.Weight());
                        fprintf(handle, "%s", buff);
                        save_today_count++;

                        if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
                            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %016" PRIx64 "-%03d-%02d WRITEDMP t%s", LogsGroup->GetServerID().c_str(), shingle, m_type, m_part, (*it).second.GetFollowText().c_str());
                    }

                    ++it;
                }
            }
            // yesterday data
            if (m_ordinary_yesterday != NULL) {
                TShingleDataSingleHashIt it;
                ui32 savetreshcalc = 0;
                char buff[256];

                all_yesterday_count = m_ordinary_yesterday->size();
                it = m_ordinary_yesterday->begin();
                while (it != m_ordinary_yesterday->end()) {
                    savetreshcalc = (*it).second.Ham() + (*it).second.Malic() + (*it).second.Spam() + (*it).second.PersHam() + (*it).second.PersSpam();

                    if ((savetreshcalc >= treshold) || ((*it).second.Weight() != 0)) {
                        shingle = (*it).first;
                        buff[sizeof(buff) - 1] = 0x00;
                        snprintf(buff, sizeof(buff) - 1, "Y %016" PRIx64 " %d %d %d %d %d %5.2f\n", (*it).first, (*it).second.Ham(), (*it).second.Malic(), (*it).second.Spam(), (*it).second.PersHam(), (*it).second.PersSpam(), (*it).second.Weight());
                        fprintf(handle, "%s", buff);
                        save_yesterday_count++;

                        if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
                            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %016" PRIx64 "-%03d-%02d WRITEDMP y%s", LogsGroup->GetServerID().c_str(), shingle, m_type, m_part, (*it).second.GetFollowText().c_str());
                    }

                    ++it;
                }
            }

            onlywrite = CShingleTime::GetMs() - onlywrite;
            UnLockOrdinary();

            // heavy data - потенциально опасно!!! - опасность убрал, поставил мьютексы
            LockHeavy();
            onlywrite2 = CShingleTime::GetMs();
            if (m_heavy_hash != NULL) {
                TShingleDataTEHashIt it;
                ui32 savetreshcalc = 0;
                char buff[256];

                all_heavy_count = m_heavy_hash->size();
                it = m_heavy_hash->begin();
                while (it != m_heavy_hash->end()) {
                    savetreshcalc = (*it).second.Ham(TODAY) + (*it).second.Malic(TODAY) + (*it).second.Spam(TODAY) + (*it).second.PersHam(TODAY) + (*it).second.PersSpam(TODAY);

                    if ((savetreshcalc >= treshold) || ((*it).second.Weight(TODAY) != 0) || ((*it).second.Weight(YESTERDAY) != 0)) {
                        buff[sizeof(buff) - 1] = 0x00;
                        snprintf(buff, sizeof(buff) - 1, "H %016" PRIx64 " %d %d %d %d %d %5.2f, %d %d %d %d %d %5.2f\n", (*it).first, (*it).second.Ham(TODAY), (*it).second.Malic(TODAY), (*it).second.Spam(TODAY), (*it).second.PersHam(TODAY), (*it).second.PersSpam(TODAY), (*it).second.Weight(TODAY), (*it).second.Ham(YESTERDAY), (*it).second.Malic(YESTERDAY), (*it).second.Spam(YESTERDAY), (*it).second.PersHam(YESTERDAY), (*it).second.PersSpam(YESTERDAY), (*it).second.Weight(YESTERDAY));
                        fprintf(handle, "%s", buff);
                        save_heavy_count++;
                    }

                    ++it;
                }
            }
            onlywrite2 = CShingleTime::GetMs() - onlywrite2;
            UnLockHeavy();

            fclose(handle);
        }
    }
    res = CShingleTime::GetMs() - res;

    m_event_mutex.Acquire();
    m_need_writedump = false;
    m_event_mutex.Release();

    onlywrite += onlywrite2;
    if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
        LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "WRITEDUMP(t=%03d,p=%02d,trsh=%u) to % 5u(% 5u) msec T=%u(%u) Y=%u(%u) H=%u(%u)", m_type, m_part, treshold, res, onlywrite, save_today_count, all_today_count, save_yesterday_count, all_yesterday_count, save_heavy_count, all_heavy_count);

    m_last_save = time(NULL);

    return res;
}

ui32 TShingleStorageItem::Cleanup() {
    ui32 res = 0;
    ui32 all_today_count = 0;
    ui32 del_today_count = 0;
    ui32 remain_today_count = 0;
    ui32 onlycleanup = 0;
    ui64 shingle = 0;
    bool shutdownbreak = false;
    TShingleDataSingleHash* new_hash = NULL;
    TShingleDataSingleHash* temp_hash = NULL;
    ui32 tick_findfsh = 0;
    TDeleteStruct deldata;

    res = CShingleTime::GetMs();

    // if ( (m_type == 2) && (m_part == 2) )
    //    sleep(60);

    LockOrdinary(TD_CLEANUP);
    onlycleanup = CShingleTime::GetMs();
    if (m_ordinary_today != NULL) {
        TShingleDataSingleHashIt it;
        ui32 savetreshcalc = 0;

        all_today_count = m_ordinary_today->size();
        if (m_cleanup_type == CT_DEL)   // удаляем элементы их хэша
        {
            it = m_ordinary_today->begin();
            while (it != m_ordinary_today->end()) {
                if (GetShutdownPriznak()) {
                    shutdownbreak = true;
                    break;
                }

                savetreshcalc = (*it).second.Ham() + (*it).second.Malic() + (*it).second.Spam() + (*it).second.PersHam() + (*it).second.PersSpam();
                if ((savetreshcalc < m_cleanup_treshold) && ((*it).second.Weight() == 0)) {
                    shingle = (*it).first;
                    m_ordinary_today->erase(it++);
                    del_today_count++;

                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %016" PRIx64 "-%03d-%02d DELETE (cleanup)", LogsGroup->GetServerID().c_str(), shingle, m_type, m_part);

                } else {
                    ++it;
                }
            }

        } else {
            new_hash = new TShingleDataSingleHash();
            if (new_hash != NULL) {
                it = m_ordinary_today->begin();
                while (it != m_ordinary_today->end()) {
                    if (GetShutdownPriznak()) {
                        shutdownbreak = true;
                        break;
                    }

                    savetreshcalc = (*it).second.Ham() + (*it).second.Malic() + (*it).second.Spam() + (*it).second.PersHam() + (*it).second.PersSpam();
                    if ((savetreshcalc >= m_cleanup_treshold) || ((*it).second.Weight() != 0)) {
                        shingle = (*it).first;
                        new_hash->insert(TShingleDataSingleHash::value_type(shingle, (*it).second));
                        remain_today_count++;

                        if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
                            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %016" PRIx64 "-%03d-%02d REMAIN (cleanup)", LogsGroup->GetServerID().c_str(), shingle, m_type, m_part);
                    }

                    ++it;
                }

                del_today_count = all_today_count - remain_today_count;
                temp_hash = m_ordinary_today;
                m_ordinary_today = new_hash;

                // удаляем старый хэш сегодня, если прочистка проводилась путем создания нового хэша
                if (temp_hash != NULL) {
                    deldata.part = m_part;
                    deldata.type = m_type;
                    deldata.op_ident = "cleanup";
                    deldata.ordinary_today = temp_hash;
                    deldata.ordinary_yesterday = NULL;
                    deldata.heavy_hash = NULL;

                    if (delobj != NULL)
                        delobj->Add(deldata);
                }
            }
        }
    }

    m_last_cleanup = time(NULL);
    m_today_cleanup_count = IncMax32(m_today_cleanup_count, 1);

    onlycleanup = CShingleTime::GetMs() - onlycleanup;
    UnLockOrdinary();

    res = CShingleTime::GetMs() - res;

    m_event_mutex.Acquire();
    m_need_cleanup = false;
    m_event_mutex.Release();

    if (shutdownbreak) {
        WriteExitStep("cleanup break: t=" + IntToStroka3(m_type) + ", p=" + IntToStroka2(m_part));

        if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
            LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "CLEANUP(t=%03d,p=%02d) break(shutdown): % 5u(% 5u) msec, del=%u all=%u", m_type, m_part, res, onlycleanup, del_today_count, all_today_count);

    } else {
        if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
            LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "CLEANUP(t=%03d,p=%02d) to % 5u(% 5u) msec, del=%u all=%u", m_type, m_part, res, onlycleanup, del_today_count, all_today_count);
    }

    return res;
}

ui32 TShingleStorageItem::Midnight() {
    ui32 res = 0;
    ui32 onlymidnight = 0;
    TDeleteStruct deldata;

    res = CShingleTime::GetMs();

    LockOrdinary(TD_MIDNIGHT);
    onlymidnight = CShingleTime::GetMs();

    m_today_cleanup_count = 0;

    deldata.part = m_part;
    deldata.type = m_type;
    deldata.op_ident = "midnight";
    deldata.ordinary_today = NULL;
    deldata.ordinary_yesterday = m_ordinary_yesterday;
    m_ordinary_yesterday = m_ordinary_today;
    m_ordinary_today = new TShingleDataSingleHash();
    deldata.heavy_hash = m_heavy_hash;
    m_heavy_hash = new TShingleDataTEHash();

    if (delobj != NULL)
        delobj->Add(deldata);

    m_last_midnight = time(NULL);

    onlymidnight = CShingleTime::GetMs() - onlymidnight;

    UnLockOrdinary();

    res = CShingleTime::GetMs() - res;

    m_event_mutex.Acquire();
    m_need_midnight = false;
    m_event_mutex.Release();

    busycacheobj.Midnight();
    m_replic_shingles_count.Midnight();

    if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
        LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "MIDNIGHT(t=%03d,p=%02d) to % 5u(% 5u) msec", m_type, m_part, res, onlymidnight);

    return res;
}

void TShingleStorageItem::Shutdown() {
    m_shutdown_mutex.Acquire();

    m_shutdown = true;

    m_shutdown_mutex.Release();
}

void TShingleStorageItem::SetLastMidnightTime(ui32 value) {
    LockOrdinary(TD_SETLASTMIDNIGHTTIME);

    m_last_midnight = value;

    UnLockOrdinary();
}

void TShingleStorageItem::SetLastCleanupTime(ui32 value) {
    LockOrdinary(TD_SETLASTCLEANUPTIME);

    m_last_cleanup = value;

    UnLockOrdinary();
}

void TShingleStorageItem::AddTodayDumpItem(ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, bool& exists) {
    TShingleDataSingleHashIt it;
    ui32 tick_findfsh = 0;

    if (m_ordinary_today != NULL) {
        it = m_ordinary_today->find(shingle);
        if (it != m_ordinary_today->end()) {
            (*it).second = TShingleDataSingle(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight);
            exists = true;

        } else {
            m_ordinary_today->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight)));
            exists = false;
        }

        if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %016" PRIx64 "-%03d-%02d READDUMP t%s", LogsGroup->GetServerID().c_str(), shingle, m_type, m_part, TShingleDataSingle(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight).GetFollowText().c_str());
    }
}

void TShingleStorageItem::AddYesterdayDumpItem(ui64 shingle, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, bool& exists) {
    TShingleDataSingleHashIt it;
    ui32 tick_findfsh = 0;

    if (m_ordinary_yesterday != NULL) {
        it = m_ordinary_yesterday->find(shingle);
        if (it != m_ordinary_yesterday->end()) {
            (*it).second = TShingleDataSingle(y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight);
            exists = true;

        } else {
            m_ordinary_yesterday->insert(TShingleDataSingleHash::value_type(shingle, TShingleDataSingle(y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight)));
            exists = false;
        }

        if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(shingle, tick_findfsh))
            LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %016" PRIx64 "-%03d-%02d READDUMP y%s", LogsGroup->GetServerID().c_str(), shingle, m_type, m_part, TShingleDataSingle(y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight).GetFollowText().c_str());
    }
}

void TShingleStorageItem::AddHeavyDumpItem(ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, bool& exists) {
    TShingleDataTEHashIt it;

    if (m_heavy_hash != NULL) {
        it = m_heavy_hash->find(shingle);
        if (it != m_heavy_hash->end()) {
            (*it).second = TShingleDataTE(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight);
            exists = true;

        } else {
            m_heavy_hash->insert(TShingleDataTEHash::value_type(shingle, TShingleDataTE(t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight)));
            exists = false;
        }
    }
}

//********************************************************************************************************************************
//                                                 TBusyCache
//********************************************************************************************************************************

TBusyCache::TBusyCache() {
    m_busy_monstat = NULL;
}

TBusyCache::~TBusyCache() {
}

void TBusyCache::Init(TBusyMonStat* busy_monstat) {
    m_busy_monstat = busy_monstat;
}

void TBusyCache::Add(const TBusyCacheItem& value) {
    switch (value.m_type) {
        case BC_PUT:
            BusyCacheMutex.Acquire();

            if (busycache.size() < MAX_SIZE_BUSYCACHE) {
                busycache.push_back(value);

                statdata.m_today_put++;
            } else
                statdata.m_today_lost_put++;

            if (m_busy_monstat != NULL)
                m_busy_monstat->m_put.Increment();

            BusyCacheMutex.Release();
            break;
        case BC_PUTM:
            BusyCacheMutex.Acquire();

            if (busycache.size() < MAX_SIZE_BUSYCACHE) {
                busycache.push_back(value);

                statdata.m_today_putm++;
            } else
                statdata.m_today_lost_putm++;

            if (m_busy_monstat != NULL)
                m_busy_monstat->m_putm.Increment();

            BusyCacheMutex.Release();
            break;
        case BC_SETM:
            BusyCacheMutex.Acquire();

            if (busycache.size() < MAX_SIZE_BUSYCACHE) {
                busycache.push_back(value);

                statdata.m_today_setm++;
            } else
                statdata.m_today_lost_setm++;

            if (m_busy_monstat != NULL)
                m_busy_monstat->m_setm.Increment();

            BusyCacheMutex.Release();
            break;
        case BC_SETWEIGHT:
            BusyCacheMutex.Acquire();

            if (busycache.size() < MAX_SIZE_BUSYCACHE) {
                busycache.push_back(value);

                statdata.m_today_setweight++;
            } else
                statdata.m_today_lost_setweight++;

            if (m_busy_monstat != NULL)
                m_busy_monstat->m_setweight.Increment();

            BusyCacheMutex.Release();
            break;
        case BC_UNKNOWN:
            break;
    };
}

bool TBusyCache::Get(TBusyCacheItem& value) {
    bool res = false;
    TBusyCacheItemListIt it;

    if (busycache.size() > 0) {
        BusyCacheMutex.Acquire();

        it = busycache.begin();
        if (it != busycache.end()) {
            value = (*it);
            busycache.erase(it);
            res = true;
        }

        BusyCacheMutex.Release();
    }

    return res;
}

void TBusyCache::Midnight() {
    BusyCacheMutex.Acquire();
    statdata.Midnight();
    BusyCacheMutex.Release();
}

TBusyCacheStat TBusyCache::GetStat() {
    TBusyCacheStat res;

    res = statdata;
    res.m_count = busycache.size();

    return res;
}

bool TBusyCache::IsEmpty() {
    bool res = true;

    if (busycache.size() > 0)
        res = false;

    return res;
}

void TBusyCache::AddBusygetStat() {
    BusyCacheMutex.Acquire();

    statdata.m_today_get++;
    if (m_busy_monstat != NULL)
        m_busy_monstat->m_get.Increment();

    BusyCacheMutex.Release();
}

void TBusyCache::AddBusygetStatFindHeavy() {
    BusyCacheMutex.Acquire();

    statdata.m_today_get_heavy++;
    if (m_busy_monstat != NULL)
        m_busy_monstat->m_get_heavy.Increment();

    BusyCacheMutex.Release();
}

//********************************************************************************************************************************
//                                   TShingleStorageV
//********************************************************************************************************************************

TShingleStorageV::TShingleStorageV() {
    RPLPool = NULL;
    m_type = -1;
    LogsGroup = NULL;
    delobj = NULL;
    m_partcount = PART_COUNT_MAX;
    m_heavy_treshold = TShingleStorageItem::HEAVY_TRESHOLD_DEFAULT;
    m_max_size_ohash = TShingleStorageItem::MAX_SIZE_PART_ORDINARYHASH_DEFAULT;
    m_save_treshold = TShingleStorageItem::SAVE_TRESHOLD_DEFAULT;
    m_cleanup_treshold = TShingleStorageItem::CLEANUP_TRESHOLD_DEFAULT;
    m_rpldiff_value = TShingleStorageItem::RPLDIFF_VALUE_DEFAULT;
    m_dumpfilename = "";
    m_loaddump_at_startup = LOAD_AT_STARTUP_DEFAULT;
    res_loaddump = "";
    m_midnight_koef_div = 0;
    m_cleanup_type = CT_NEW;
    m_shutdown_save_treshold = TShingleStorageItem::SHUTDOWN_SAVE_TRESHOLD_DEFAULT;
}

TShingleStorageV::~TShingleStorageV() {
    TShingleStorageItemVectorIt it;
    ui32 tick = 0;
    int part = 0;

    WriteExitStep("start write dump TShingleStorageV-" + IntToStroka3(m_type));

    tick = CShingleTime::GetMs();
    if (m_loaddump_at_startup)
        WriteDumpToClose();
    tick = CShingleTime::GetMs() - tick;

    WriteExitStep("write dump compleate TShingleStorageV-" + IntToStroka3(m_type) + " to " + IntToStroka(tick) + " msec");

    it = m_data.begin();
    while (it != m_data.end()) {
        WriteExitStep("start delete part TShingleStorageV-" + IntToStroka3(m_type) + "-" + IntToStroka2(part));
        tick = CShingleTime::GetMs();
        if ((*it) != NULL) {
            delete (*it);
            (*it) = NULL;
        }
        tick = CShingleTime::GetMs() - tick;
        WriteExitStep("delete part compleate TShingleStorageV-" + IntToStroka3(m_type) + "-" + IntToStroka2(part) + " to " + IntToStroka(tick) + " msec");

        part++;

        ++it;
    }

    WriteExitStep("start delete attend_hash TShingleStorageV-" + IntToStroka3(m_type));
}

void TShingleStorageV::WriteExitStep(const TString& text) {
    if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
        LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "EXITEVENT: %s", text.c_str());
        LogsGroup->GetServerLog()->FFlush();
    }
}

bool TShingleStorageV::Init(int type, ui8 partcountA, TLogsGroup* LogsGroupA, TDeleteObjectMain* delobjA, ui32 rpldiff_value, ui32 heavy_treshold, ui32 max_size_ohash, ui32 save_treshold, ui32 cleanup_treshold, const TString& dumpfilename, bool loaddump_at_startup, TRPLMain* RPLPoolA, ui32 koef_div, TBusyMonStat* busy_monstat, TCleanupType cleanup_type, ui32 shutdown_save_treshold) {
    bool res = true;
    TShingleStorageItem* ssi = NULL;

    if (partcountA == 0)
        m_partcount = 1;
    else if (partcountA > PART_COUNT_MAX)
        m_partcount = PART_COUNT_MAX;
    else
        m_partcount = partcountA;

    RPLPool = RPLPoolA;
    m_type = type;
    LogsGroup = LogsGroupA;
    delobj = delobjA;
    m_rpldiff_value = rpldiff_value;
    m_heavy_treshold = heavy_treshold;
    m_max_size_ohash = max_size_ohash / m_partcount;
    m_save_treshold = save_treshold;
    m_cleanup_treshold = cleanup_treshold;
    m_dumpfilename = dumpfilename;
    m_loaddump_at_startup = loaddump_at_startup;
    m_midnight_koef_div = koef_div;
    m_cleanup_type = cleanup_type;
    m_shutdown_save_treshold = shutdown_save_treshold;

    for (int i = 0; i < m_partcount; i++) {
        ssi = new TShingleStorageItem();
        if (ssi != NULL) {
            if (!ssi->Init(i, m_type, m_dumpfilename, LogsGroup, delobj, m_rpldiff_value, m_heavy_treshold, m_max_size_ohash, m_save_treshold, m_cleanup_treshold, RPLPool, &m_followhash, koef_div, busy_monstat, m_cleanup_type, m_shutdown_save_treshold)) {
                delete ssi;
                ssi = NULL;

                if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                    LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "Error initialize part=%d for type=%d", i, m_type);

                res = false;
                break;
            } else {
                m_data.push_back(ssi);
            }
        } else {
            res = false;
            break;
        }
    }

    /*if (loaddump_at_startup)
   {
      if (!ReadDump())
      {
         res = false;
         if ( (LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL) )
            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "Error read dump for type=%d (%s)", m_type, m_dumpfilename.c_str());
      }
   }*/

    return res;
}

bool TShingleStorageV::ReadDumpStorage(bool loaddump_at_startup, ui32& lastmidnighttime) {
    bool res = true;

    if (loaddump_at_startup) {
        if (!ReadDump(lastmidnighttime)) {
            res = false;
            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "Error read dump for type=%d (%s)", m_type, m_dumpfilename.c_str());
        }
    }

    return res;
}

void TShingleStorageV::CorrectLastMidnightTime(ui32 value) {
    for (int i = 0; i < m_partcount; i++) {
        if (m_data[i] != NULL)
            m_data[i]->SetLastMidnightTime(value);
    }
}

ui8 TShingleStorageV::GetPartNumber(ui64 shingle) {
    return shingle % m_partcount;
}

TShingleStat TShingleStorageV::Put(const TString& abonent_ip, ui64 shingle, ui16 count, TShingleSpamType spamtype, ui64 att_shingle, int att_shingle_type, TPersType pers, const TString& debugstr, bool no_replic, bool is_putm) {
    TShingleStat res;
    ui8 partnumber = GetPartNumber(shingle);

    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL) {
            res = m_data[partnumber]->Put(abonent_ip, shingle, count, spamtype, att_shingle, att_shingle_type, pers, debugstr, no_replic, is_putm);
        }
    }

    return res;
}

TShingleStat TShingleStorageV::PutM(const TString& abonent_ip, ui64 shingle, ui16 ham, ui16 dlv, ui16 spam, ui16 persham, ui16 persspam, ui64 att_shingle, int att_shingle_type) {
    TShingleStat res;
    ui8 partnumber = GetPartNumber(shingle);

    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL) {
            res = m_data[partnumber]->PutM(abonent_ip, shingle, ham, dlv, spam, persham, persspam, att_shingle, att_shingle_type);
        }
    }

    return res;
}

TShingleStat TShingleStorageV::SetM(const TString& abonent_ip, ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, ui64 att_shingle, int att_shingle_type) {
    TShingleStat res;
    ui8 partnumber = GetPartNumber(shingle);

    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL) {
            res = m_data[partnumber]->SetM(abonent_ip, shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, att_shingle, att_shingle_type);
        }
    }

    return res;
}

TShingleStat TShingleStorageV::Get(const TString& abonent_ip, ui64 shingle, bool is_second) {
    TShingleStat res;
    ui8 partnumber = GetPartNumber(shingle);

    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL) {
            res = m_data[partnumber]->Get(abonent_ip, shingle, is_second);
        } else {
            res.SetAction(ABADPARTHASH);
        }
    } else {
        res.SetAction(AEXCEEDPART);
    }

    return res;
}

bool TShingleStorageV::ClearShingleA(const TString& abonent_ip, ui64 shingle, bool& busy) {
    bool res = false;
    ui8 partnumber = GetPartNumber(shingle);

    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL)
            res = m_data[partnumber]->ClearShingleA(abonent_ip, shingle, busy);
    }

    return res;
}

int TShingleStorageV::SetWeight(const TString& abonent_ip, ui64 shingle, float weight) {
    int res = 0;
    ui8 partnumber = GetPartNumber(shingle);

    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL) {
            res = m_data[partnumber]->SetWeight(abonent_ip, shingle, weight);
        }
    }

    return res;
}

TSHTypeStat TShingleStorageV::GetStat() {
    TSHTypeStat res;

    for (int i = 0; i < m_partcount; i++) {
        if (m_data[i] != NULL)
            res.partdata[i] = m_data[i]->GetStat();
    }
    res.real_part_count = m_partcount;
    res.rpl_diff_value = m_rpldiff_value;
    res.heavy_treshold = m_heavy_treshold;
    res.max_size_ohash = m_max_size_ohash;
    res.save_treshold = m_save_treshold;
    res.cleanup_treshold = m_cleanup_treshold;
    res.dumpfilename = m_dumpfilename;
    res.load_dump_at_startup = m_loaddump_at_startup;
    res.res_loaddump = res_loaddump;
    res.midnight_koef_div = m_midnight_koef_div;
    res.cleanup_type = m_cleanup_type;
    res.shutdown_save_treshold = m_shutdown_save_treshold;

    return res;
}

void TShingleStorageV::SetWriteDumpEvent() {
    if (!m_dumpfilename.empty()) {
        TString oldfilename = "";
        FILE* handle = NULL;

        oldfilename = m_dumpfilename + ".0";
        remove(oldfilename.c_str());
        rename(m_dumpfilename.c_str(), oldfilename.c_str());
        handle = fopen(m_dumpfilename.c_str(), "wb");
        if (handle != NULL)
            fclose(handle);

        for (int i = 0; i < m_partcount; i++) {
            if (m_data[i] != NULL)
                m_data[i]->SetWriteDumpEvent();
        }
    }
}

void TShingleStorageV::WriteDumpToClose() {
    if (!m_dumpfilename.empty()) {
        TString oldfilename = "";
        FILE* handle = NULL;
        ui32 tick = CShingleTime::GetMs();

        oldfilename = m_dumpfilename + ".0";
        remove(oldfilename.c_str());
        rename(m_dumpfilename.c_str(), oldfilename.c_str());
        handle = fopen(m_dumpfilename.c_str(), "wb");
        if (handle != NULL)
            fclose(handle);

        for (int i = 0; i < m_partcount; i++) {
            if (m_data[i] != NULL)
                m_data[i]->ShutdownWriteDump();
        }

        // if ( (m_attendant_hash) && (m_attend_hash != NULL) )
        //    m_attend_hash->WriteDump();

        tick = CShingleTime::GetMs() - tick;

        printf("WRITEDUMP(type=%03d) to %u msec - OK\n", m_type, tick);
    }
}

void TShingleStorageV::SetCleanupEvent() {
    for (int i = 0; i < m_partcount; i++) {
        if (m_data[i] != NULL)
            m_data[i]->SetCleanupEvent();
    }
}

void TShingleStorageV::SetMidnightEvent() {
    for (int i = 0; i < m_partcount; i++) {
        if (m_data[i] != NULL)
            m_data[i]->SetMidnightEvent();
    }
}

void TShingleStorageV::ShedulerAction() {
    for (int i = 0; i < m_partcount; i++) {
        if (m_data[i] != NULL)
            m_data[i]->ShedulerAction();
    }
}

void TShingleStorageV::AddTodayDumpItem(ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, bool& exists) {
    ui8 partnumber = GetPartNumber(shingle);

    exists = false;
    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL)
            m_data[partnumber]->AddTodayDumpItem(shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, exists);
    }
}

void TShingleStorageV::AddYesterdayDumpItem(ui64 shingle, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, bool& exists) {
    ui8 partnumber = GetPartNumber(shingle);

    exists = false;
    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL)
            m_data[partnumber]->AddYesterdayDumpItem(shingle, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, exists);
    }
}

void TShingleStorageV::AddHeavyDumpItem(ui64 shingle, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, bool& exists) {
    ui8 partnumber = GetPartNumber(shingle);

    exists = false;
    if ((partnumber >= 0) && (partnumber < m_partcount)) {
        if (m_data[partnumber] != NULL)
            m_data[partnumber]->AddHeavyDumpItem(shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, exists);
    }
}

void TShingleStorageV::Shutdown() {
    TShingleStorageItemVectorIt it;

    it = m_data.begin();
    while (it != m_data.end()) {
        if ((*it) != NULL)
            (*it)->Shutdown();

        ++it;
    }
}

bool TShingleStorageV::ReadDump(ui32& lastmidnighttime) {
    bool res = true;
    FILE* handle = NULL;
    char buff[512];
    char buff_out[512];
    ui32 today_count = 0;
    ui32 yesterday_count = 0;
    ui32 heavy_count = 0;
    ui32 today_attend_count = 0;
    ui32 yesterday_attend_count = 0;
    ui32 today_exists_count = 0;
    ui32 yesterday_exists_count = 0;
    ui32 heavy_exists_count = 0;
    ui32 today_attend_exists_count = 0;
    ui32 yesterday_attend_exists_count = 0;
    ui32 error_count = 0;
    char tsymb = ' ';
    ui64 shingle = 0;
    ui32 vt1 = 0;
    int vt2 = 0, vt3 = 0, vt4 = 0, vt5 = 0;
    float ft = 0;
    int vy1 = 0, vy2 = 0, vy3 = 0, vy4 = 0, vy5 = 0;
    float fy = 0;
    int field_count = 0;
    ui32 tick = 0;
    int readerror = 0;
    bool exists = false;
    ui16 t_ham = 0, t_dlv = 0, t_spam = 0, t_persham = 0, t_persspam = 0;
    float t_weight = 0;
    ui16 y_ham = 0, y_dlv = 0, y_spam = 0, y_persham = 0, y_persspam = 0;
    float y_weight = 0;

    tick = CShingleTime::GetMs();
    if (!m_dumpfilename.empty()) {
        handle = fopen(m_dumpfilename.c_str(), "rb");
        if (handle != NULL) {
            // "T %016" PRIx64 " %d %d %d %d %d %5.2f\n"
            // "Y %016" PRIx64 " %d %d %d %d %d %5.2f\n"
            // "H %016" PRIx64 " %d %d %d %d %d %5.2f, %d %d %d %d %d %5.2f\n"
            // "A %016" PRIx64 " %u\n"
            // "B %016" PRIx64 " %u\n"
            while (fgets(buff, sizeof(buff), handle) != NULL) {
                field_count = sscanf(buff, "%c %" PRIx64 " %u %d %d %d %d %f, %d %d %d %d %d %f", &tsymb, &shingle, &vt1, &vt2, &vt3, &vt4, &vt5, &ft, &vy1, &vy2, &vy3, &vy4, &vy5, &fy);
                switch (field_count) {
                    case 3:
                        if (tsymb == 'A') {
                            // if (m_attendant_hash && (m_attend_hash != NULL) )
                            //    m_attend_hash->AddTodayDumpItem(shingle, vt1, exists);

                            if (exists)
                                today_attend_exists_count++;
                            today_attend_count++;

                        } else if (tsymb == 'B') {
                            // if (m_attendant_hash && (m_attend_hash != NULL) )
                            //    m_attend_hash->AddYesterdayDumpItem(shingle, vt1, exists);

                            if (exists)
                                yesterday_attend_exists_count++;
                            yesterday_attend_count++;

                        } else
                            error_count++;
                        break;
                    case 4:
                        if (tsymb == 'M') // last midnight
                        {
                            if (vt1 > lastmidnighttime)
                                lastmidnighttime = vt1;

                            for (int i = 0; i < m_partcount; i++) {
                                if (m_data[i] != NULL)
                                    m_data[i]->SetLastMidnightTime(vt1);
                            }

                        } else if (tsymb == 'C') // last cleanup
                        {
                            if ((vt2 >= 0) && (vt2 < m_partcount)) {
                                if (m_data[vt2] != NULL)
                                    m_data[vt2]->SetLastCleanupTime(vt1);
                            }

                        } else
                            error_count++;
                        break;
                    case 8:
                        if (tsymb == 'T') {
                            t_ham = vt1;
                            t_dlv = vt2;
                            t_spam = vt3;
                            t_persham = vt4;
                            t_persspam = vt5;
                            t_weight = ft;
                            AddTodayDumpItem(shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, exists);
                            if (exists)
                                today_exists_count++;
                            today_count++;

                        } else if (tsymb == 'Y') {
                            y_ham = vt1;
                            y_dlv = vt2;
                            y_spam = vt3;
                            y_persham = vt4;
                            y_persspam = vt5;
                            y_weight = ft;
                            AddYesterdayDumpItem(shingle, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, exists);
                            if (exists)
                                yesterday_exists_count++;
                            yesterday_count++;

                        } else
                            error_count++;

                        break;
                    case 14:
                        if (tsymb == 'H') {
                            t_ham = vt1;
                            t_dlv = vt2;
                            t_spam = vt3;
                            t_persham = vt4;
                            t_persspam = vt5;
                            t_weight = ft;
                            y_ham = vy1;
                            y_dlv = vy2;
                            y_spam = vy3;
                            y_persham = vy4;
                            y_persspam = vy5;
                            y_weight = fy;

                            AddHeavyDumpItem(shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, exists);
                            if (exists)
                                heavy_exists_count++;
                            heavy_count++;

                        } else
                            error_count++;
                        break;
                    default:
                        error_count++;
                };
            }

            tick = CShingleTime::GetMs() - tick;
            if (feof(handle) != 0) {
                snprintf(buff, sizeof(buff) - 1, "to % 5u msec: today=%u (exists=%u), yesterday=%u (exists=%u), heavy=%u (exists=%u), error=%u", tick, today_count, today_exists_count, yesterday_count, yesterday_exists_count, heavy_count, heavy_exists_count, error_count);
                res_loaddump = TString(buff);
                if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                    LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "LOADDUMP(t=%03d) %s", m_type, buff);

                snprintf(buff_out, sizeof(buff_out) - 1, "to % 5u msec: t=%u, y=%u, h=%u, err=%u", tick, today_count, yesterday_count, heavy_count, error_count);
                printf("LOADDUMP(t=%03d) %s\n", m_type, buff_out);
            } else {
                readerror = ferror(handle);
                snprintf(buff, sizeof(buff) - 1, "to % 5u msec: read error=%d [today=%u (exists=%u), yesterday=%u (exists=%u), heavy=%u (exists=%u), error=%u]", tick, readerror, today_count, today_exists_count, yesterday_count, yesterday_exists_count, heavy_count, heavy_exists_count, error_count);
                res_loaddump = TString(buff);
                if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                    LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KERROR, "LOADDUMP(t=%03d) %s", m_type, buff);

                snprintf(buff_out, sizeof(buff_out) - 1, "to % 5u msec: read error=%d [t=%u, y=%u, h=%u, err=%u]", tick, readerror, today_count, yesterday_count, heavy_count, error_count);
                printf("LOADDUMP(t=%03d) %s\n", m_type, buff_out);
            }

            fclose(handle);

        } else {
            snprintf(buff, sizeof(buff) - 1, "error open file %s", m_dumpfilename.c_str());
            res_loaddump = TString(buff);
            if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KERROR, "LOADDUMP(t=%03d) %s", m_type, buff);

            snprintf(buff_out, sizeof(buff_out) - 1, "error open file %s", m_dumpfilename.c_str());
            printf("LOADDUMP(t=%03d) %s\n", m_type, buff_out);
        }
    } else {
        snprintf(buff, sizeof(buff) - 1, "no defined dump filename");
        res_loaddump = TString(buff);
        if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
            LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KERROR, "LOADDUMP(t=%03d) %s", m_type, buff);

        snprintf(buff_out, sizeof(buff_out) - 1, "no defined dump filename");
        printf("LOADDUMP(t=%03d) %s\n", m_type, buff_out);
        res = false;
    }

    return res;
}

bool TShingleStorageV::ExistsFollowShingle(ui64 shingle, ui32& tick) {
    bool res = false;
    TFollowShingleHashIt it;

    tick = CShingleTime::GetMs();

    m_followmutex.Acquire();

    // если есть элемент с shingle == 0, то для всего типа пишем статистику
    it = m_followhash.find(0);
    if (it != m_followhash.end()) {
        (*it).second = IncMax64((*it).second, 1);
        res = true;
    }
    if (!res) {
        it = m_followhash.find(shingle);
        if (it != m_followhash.end()) {
            (*it).second = IncMax64((*it).second, 1);
            res = true;
        }
    }

    m_followmutex.Release();

    tick = CShingleTime::GetMs() - tick;

    return res;
}

bool TShingleStorageV::AddFollowShingle(ui64 shingle) {
    bool res = false;
    TFollowShingleHashIt it;

    m_followmutex.Acquire();

    it = m_followhash.find(shingle);
    if (it == m_followhash.end()) {
        m_followhash[shingle] = 0;
        res = true;

    } else
        res = true;

    m_followmutex.Release();

    return res;
}

bool TShingleStorageV::DeleteFollowShingle(ui64 shingle) {
    bool res = false;
    TFollowShingleHashIt it;

    m_followmutex.Acquire();

    it = m_followhash.find(shingle);
    if (it != m_followhash.end()) {
        m_followhash.erase(shingle);
        res = true;
    }

    m_followmutex.Release();

    return res;
}

void TShingleStorageV::GetListFollowShingles(TFollowShinglesList& list) {
    TFollowShingleHashIt it;

    m_followmutex.Acquire();

    it = m_followhash.begin();
    while (it != m_followhash.end()) {
        list.push_back(TFollowShingles((*it).first, m_type, (*it).second));

        ++it;
    }

    m_followmutex.Release();
}

//********************************************************************************************************************************
//                                        TShingleStorageMain
//********************************************************************************************************************************

TShingleStorageMain::TShingleStorageMain() {
    m_server_id = "";
    LogsGroup = NULL;
    config = NULL;
    mode = MUNKNOWN;
    PoolProxy = NULL;
    RPLPool = NULL;
    SSObj = NULL;
    weights_dir_name = "";
    weights_filename = "";
    weights_type = 6;
    periodwritedumpsec = 3600;
    periodwritedumpsec_slow = 20;
    writedumptype = 0;
    m_koef_div = TShingleStat::YESTERDAY_DIV_KOEF;

    m_stop = false;
    m_SleepTime = 1; // Seconds
    m_StopSheduler = true;
    m_Sheduler = NULL;
    m_StartSheduler = false;

    m_longdelayms = 200;

    m_followsh_filename = "";

    for (int i = 0; i < MAX_COUNT_TYPES; i++)
        typesdata[i] = NULL;
}

TShingleStorageMain::~TShingleStorageMain() {
    ui32 tick = 0;

    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL) {
            WriteExitStep("start delete TShingleStorageV-" + IntToStroka3(i));

            tick = CShingleTime::GetMs();

            delete typesdata[i];
            typesdata[i] = NULL;

            tick = CShingleTime::GetMs() - tick;

            WriteExitStep("compleate delete TShingleStorageV-" + IntToStroka3(i) + " to " + IntToStroka(tick) + " msec");

            if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "DELTYPEOBG(t=%03d) to % 5u msec", i, tick);
        }
    }
}

void TShingleStorageMain::WriteExitStep(const TString& text) {
    if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
        LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "EXITEVENT: %s", text.c_str());
        LogsGroup->GetServerLog()->FFlush();
    }
}

bool TShingleStorageMain::InitBeforeFork(const TString& server_id, TLogsGroup* LogsGroupA, TKConfig* configA, TShinglerSrvMode modeA, TProxyPool* PoolProxyA, TRPLMain* RPLPoolA, TSimpleStorage* SSObjA) {
    bool res = true;
    TString dumpprefix = "";
    bool loaddump_at_startup = TShingleStorageV::LOAD_AT_STARTUP_DEFAULT;
    ui32 heavy_treshold = TShingleStorageItem::HEAVY_TRESHOLD_DEFAULT;
    ui32 max_size_ordinary_hash = TShingleStorageItem::MAX_SIZE_PART_ORDINARYHASH_DEFAULT;
    ui32 save_treshold = TShingleStorageItem::SAVE_TRESHOLD_DEFAULT;
    ui32 shutdown_save_treshold = TShingleStorageItem::SHUTDOWN_SAVE_TRESHOLD_DEFAULT;
    ui32 cleanup_treshold = TShingleStorageItem::CLEANUP_TRESHOLD_DEFAULT;
    ui32 rpldiff_value = TShingleStorageItem::RPLDIFF_VALUE_DEFAULT;
    ui16 partcount = TShingleStorageV::PART_COUNT_MAX;
    TCleanupType cleanup_type = CT_NEW;
    ui32 heavy_treshold_default = TShingleStorageItem::HEAVY_TRESHOLD_DEFAULT;
    ui32 max_size_ordinary_hash_default = TShingleStorageItem::MAX_SIZE_PART_ORDINARYHASH_DEFAULT;
    ui32 save_treshold_default = TShingleStorageItem::SAVE_TRESHOLD_DEFAULT;
    ui32 shutdown_save_treshold_default = TShingleStorageItem::SHUTDOWN_SAVE_TRESHOLD_DEFAULT;
    ui32 cleanup_treshold_default = TShingleStorageItem::CLEANUP_TRESHOLD_DEFAULT;
    ui32 rpldiff_value_default = TShingleStorageItem::RPLDIFF_VALUE_DEFAULT;
    ui16 partcount_default = TShingleStorageV::PART_COUNT_MAX;
    TCleanupType cleanup_type_default = CT_NEW;
    ui32 longdelay_ms = 200;

    m_server_id = server_id;
    PoolProxy = PoolProxyA;
    RPLPool = RPLPoolA;
    SSObj = SSObjA;
    LogsGroup = LogsGroupA;
    config = configA;
    mode = modeA;
    if (config != NULL) {
        dumpprefix = config->ReadStroka("shingle", "dump_prefix", "");
        loaddump_at_startup = config->ReadBool("shingle", "load_dump_at_startup", TShingleStorageV::LOAD_AT_STARTUP_DEFAULT);
        weights_dir_name = config->ReadStroka("shingle", "weights_dir", "");
        weights_filename = config->ReadStroka("shingle", "weights_filename", "");
        weights_type = config->ReadInteger("shingle", "weights_type", 6);
        writedumptype = config->ReadInteger("shingle", "writedump_type", 0);
        periodwritedumpsec = config->ReadInteger("shingle", "periodwritedumpsec", 3600);
        periodwritedumpsec_slow = config->ReadInteger("shingle", "periodwritedumpsec_slow", 20);
        m_followsh_filename = config->ReadStroka("shingle", "followsh_filename", "");
        m_koef_div = config->ReadInteger("shingle", "koef_div", TShingleStat::YESTERDAY_DIV_KOEF);
        longdelay_ms = config->ReadInteger("shingle", "longdelay_ms", 200);
    }

    SetLongDelayMs(longdelay_ms);

    wd_obg.Init(MAX_COUNT_TYPES, periodwritedumpsec_slow);

    oldrequest_all.Init(10000, 20000, true);
    newrequest_all.Init(10000, 20000, true);
    request_rpl.Init(10000, 20000, true);
    request_prx.Init(10000, 20000, true);

    if ((config != NULL) && (config->SectionExists("typeshingle_default"))) {
        heavy_treshold_default = config->ReadInteger("typeshingle_default", "heavy_treshold", TShingleStorageItem::HEAVY_TRESHOLD_DEFAULT);
        max_size_ordinary_hash_default = config->ReadInteger("typeshingle_default", "max_size_hash", TShingleStorageItem::MAX_SIZE_PART_ORDINARYHASH_DEFAULT);
        save_treshold_default = config->ReadInteger("typeshingle_default", "save_treshold", TShingleStorageItem::SAVE_TRESHOLD_DEFAULT);
        cleanup_treshold_default = config->ReadInteger("typeshingle_default", "cleanup_treshold", TShingleStorageItem::CLEANUP_TRESHOLD_DEFAULT);
        rpldiff_value_default = config->ReadInteger("typeshingle_default", "rpl_diff_value", TShingleStorageItem::RPLDIFF_VALUE_DEFAULT);
        partcount_default = config->ReadInteger("typeshingle_default", "part_count", TShingleStorageV::PART_COUNT_MAX);
        cleanup_type_default = (TCleanupType)config->ReadInteger("typeshingle_default", "cleanup_type", CT_NEW);
        shutdown_save_treshold_default = config->ReadInteger("typeshingle_default", "shutdown_save_treshold", TShingleStorageItem::SHUTDOWN_SAVE_TRESHOLD_DEFAULT);
    }

    delobj.Init(LogsGroup);

    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (config != NULL) {
            heavy_treshold = config->ReadInteger("typeshingle_" + IntToStroka3(i), "heavy_treshold", heavy_treshold_default);
            max_size_ordinary_hash = config->ReadInteger("typeshingle_" + IntToStroka3(i), "max_size_hash", max_size_ordinary_hash_default);
            save_treshold = config->ReadInteger("typeshingle_" + IntToStroka3(i), "save_treshold", save_treshold_default);
            cleanup_treshold = config->ReadInteger("typeshingle_" + IntToStroka3(i), "cleanup_treshold", cleanup_treshold_default);
            rpldiff_value = config->ReadInteger("typeshingle_" + IntToStroka3(i), "rpl_diff_value", rpldiff_value_default);
            partcount = config->ReadInteger("typeshingle_" + IntToStroka3(i), "part_count", partcount_default);
            cleanup_type = (TCleanupType)config->ReadInteger("typeshingle_" + IntToStroka3(i), "cleanup_type", cleanup_type_default);
            shutdown_save_treshold = config->ReadInteger("typeshingle_" + IntToStroka3(i), "shutdown_save_treshold", shutdown_save_treshold_default);
        }

        typesdata[i] = new TShingleStorageV();
        if (typesdata[i] != NULL) {
            if (!(typesdata[i]->Init(i, partcount, LogsGroup, &delobj, rpldiff_value, heavy_treshold, max_size_ordinary_hash, save_treshold, cleanup_treshold, dumpprefix + IntToStroka3(i) + ".dmp", loaddump_at_startup, RPLPool, m_koef_div, &m_busy_monstat, cleanup_type, shutdown_save_treshold))) {
                if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                    LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "Error initialize type %d", i);

                res = false;
                break;
            }

        } else {
            res = false;
            break;
        }
    }

    // читаем следящие шинглы
    ReadFollowShingles(m_followsh_filename);

    // читаем дампы
    ui32 lastmidnighttime = 0;

    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            typesdata[i]->ReadDumpStorage(loaddump_at_startup, lastmidnighttime);
    }

    // устанавливаем всем одинаковое время последнего миднайта
    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            typesdata[i]->CorrectLastMidnightTime(lastmidnighttime);
    }
    printf("LOADDUMP: Set common last_midnight_time=%u (%s)\n", lastmidnighttime, TimeToStr(lastmidnighttime).c_str());
    if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL)) {
        LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "LOADDUMP: Set common last_midnight_time=%u (%s)", lastmidnighttime, TimeToStr(lastmidnighttime).c_str());
        LogsGroup->GetShinglesLog()->FFlush();
    }

    return res;
}

bool TShingleStorageMain::InitAfterFork() {
    bool res = true;

    StartSheduler();

    m_actionmutex.Acquire();
    m_last_load_weight = time(NULL);
    m_last_writedumpevent = time(NULL);
    m_actionmutex.Release();

    return res;
}

TShingleStat TShingleStorageMain::Put(const TString& abonent_ip, ui64 shingle, ui16 type, ui16 count, TShingleSpamType spamtype, ui64 att_shingle, int att_shingle_type, TPersType pers, const TString& debugstr) {
    TShingleStat res;
    TShingleStat res_att;
    ui32 tick_fs = 0;
    ui32 writelog_tick_fs = 0;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL) {
            res = typesdata[type]->Put(abonent_ip, shingle, count, spamtype, att_shingle, att_shingle_type, pers, debugstr, false, false);

            if ((att_shingle != 0) && (att_shingle_type >= 0) && (att_shingle_type < MAX_COUNT_TYPES)) {
                if (res.IsFirstRecord()) {
                    if (typesdata[att_shingle_type] != NULL) {
                        res_att = typesdata[att_shingle_type]->Put(abonent_ip, att_shingle, 1, spamtype, att_shingle, att_shingle_type, pers, debugstr, true, false);
                        res.SetUniqShingleData(res_att);
                    }

                } else {
                    // печатаем отладочную информацию по шинглам из хэша отслеживающих шинглов
                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(att_shingle, att_shingle_type, tick_fs)) {
                        writelog_tick_fs = CShingleTime::GetMs();
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d    PUTs     no add, exist %016" PRIx64 "-%03d", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), att_shingle, att_shingle_type, shingle, type);
                        writelog_tick_fs = CShingleTime::GetMs() - writelog_tick_fs;
                    }

                    if (typesdata[att_shingle_type] != NULL) {
                        res_att = typesdata[att_shingle_type]->Get(abonent_ip, att_shingle, true);
#ifdef SOSERV_DEBUG
                        res_att.debuginfo.SetFindFollowShingle2(tick_fs);
                        res_att.debuginfo.SetWriteLogFollowShingle2(writelog_tick_fs);
#endif
                        res.SetUniqShingleData(res_att);
                    }
                }
            }
        }
    }
    res.SetShingleType(TShingleStat::TSHSHINGLE, shingle, type, att_shingle, att_shingle_type);

    return res;
}

TShingleStat TShingleStorageMain::PutM(const TString& abonent_ip, ui64 shingle, ui16 type, ui16 ham, ui16 dlv, ui16 spam, ui16 persham, ui16 persspam, ui64 att_shingle, int att_shingle_type, TShingleSpamType att_spamtype, TPersType att_pers) {
    TShingleStat res;
    TShingleStat res_att;
    TString debugstr = "";
    ui32 tick_fs = 0;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL) {
            res = typesdata[type]->PutM(abonent_ip, shingle, ham, dlv, spam, persham, persspam, att_shingle, att_shingle_type);
            if ((att_shingle != 0) && (att_shingle_type >= 0) && (att_shingle_type < MAX_COUNT_TYPES)) {
                if (res.IsFirstRecord()) {
                    if (typesdata[att_shingle_type] != NULL) {
                        res_att = typesdata[att_shingle_type]->Put(abonent_ip, att_shingle, 1, att_spamtype, att_shingle, att_shingle_type, att_pers, debugstr, true, true);
                        res.SetUniqShingleData(res_att);
                    }
                } else {
                    // печатаем отладочную информацию по шинглам из хэша отслеживающих шинглов
                    if ((LogsGroup != NULL) && (LogsGroup->GetFollowShinglesLog() != NULL) && ExistsFollowShingle(att_shingle, att_shingle_type, tick_fs))
                        LogsGroup->GetFollowShinglesLog()->WriteMessageAndData("%s %s %016" PRIx64 "-%03d    PUTms    no add, exist %016" PRIx64 "-%03d", LogsGroup->GetServerID().c_str(), abonent_ip.c_str(), att_shingle, att_shingle_type, shingle, type);
                }
            }
        }
    }
    res.SetShingleType(TShingleStat::TSHSHINGLE, shingle, type, att_shingle, att_shingle_type);

    return res;
}

TShingleStat TShingleStorageMain::SetM(const TString& abonent_ip, ui64 shingle, ui16 type, ui16 t_ham, ui16 t_dlv, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_dlv, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight, ui64 att_shingle, int att_shingle_type) {
    TShingleStat res;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            res = typesdata[type]->SetM(abonent_ip, shingle, t_ham, t_dlv, t_spam, t_persham, t_persspam, t_weight, y_ham, y_dlv, y_spam, y_persham, y_persspam, y_weight, att_shingle, att_shingle_type);
    }
    res.SetShingleType(TShingleStat::TSHSHINGLE, shingle, type, att_shingle, att_shingle_type);

    return res;
}

TShingleStat TShingleStorageMain::Get(const TString& abonent_ip, ui64 shingle, ui16 type, ui64 att_shingle, int att_shingle_type) {
    TShingleStat res;
    TShingleStat res_att;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL) {
            res = typesdata[type]->Get(abonent_ip, shingle, false);
            if ((att_shingle != 0) && (att_shingle_type >= 0) && (att_shingle_type < MAX_COUNT_TYPES)) {
                res_att = typesdata[att_shingle_type]->Get(abonent_ip, att_shingle, true);
                res.SetUniqShingleData(res_att);
            }
        } else {
            res.SetAction(ABADTYPEHASH);
        }
    } else {
        res.SetAction(AEXCEEDTYPE);
    }
    res.SetShingleType(TShingleStat::TSHSHINGLE, shingle, type, att_shingle, att_shingle_type);

    return res;
}

bool TShingleStorageMain::ClearShingleA(const TString& abonent_ip, ui64 shingle, ui16 type, bool& busy) {
    bool res = false;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            res = typesdata[type]->ClearShingleA(abonent_ip, shingle, busy);
    }

    return res;
}

int TShingleStorageMain::SetWeight(const TString& abonent_ip, ui64 shingle, ui16 type, float weight) {
    int res = 0;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            res = typesdata[type]->SetWeight(abonent_ip, shingle, weight);
    }

    return res;
}

TShingleStat TShingleStorageMain::GetStorage(ui64 shingle) {
    TShingleStat res;
    TString st_value;

    res = TShingleStat(shingle);
    if ((SSObj != NULL) && (SSObj->Get(shingle, st_value)))
        res.SetStorageData(st_value);

    return res;
}

TShingleStat TShingleStorageMain::AddStorage(ui64 shingle, const TString& stdata) {
    TShingleStat res;

    res = TShingleStat(shingle);
    if ((SSObj != NULL) && (SSObj->Add(shingle, stdata)))
        res.SetStorageData(stdata);

    return res;
}

/*bool TShingleStorageMain::ParseRequestParamsBase(const TString &request, TReqstParams &paramlist, char record_delimiter, char keyvalue_delimiter)
{
  bool         res      = true;
  const char   *p_start = NULL;
  const char   *p_end   = NULL;
  const char   *p_amp   = NULL;
  const char   *p_equal = NULL;
  TString       key      = "";
  TString       value    = "";
  TReqstParams::iterator it;

  if (!request.empty())
  {
     p_start = request.c_str();
     p_end   = request.c_str() + request.length();
     while (true)
     {
         if (p_start == p_end)
            break;
         key = "";
         value = "";
         p_amp = strchr(p_start, record_delimiter);
         if (p_amp == NULL)
            p_amp = p_end;
         if (p_amp != NULL)
         {
            if (p_amp == p_start)
            {
               res = false;
               break;
            } else
            {
               if (keyvalue_delimiter == 0x00) //only key
               {
                  if (p_amp > p_start)
                  {
                     key = Stroka(p_start, p_amp - p_start);

                     it = paramlist.find(key);
                     if (it == paramlist.end())
                     {
                        paramlist[key] = "";

                     } else
                     {
                        // повторные запросы игнорируем
                        //res = false;
                        //break;
                     }
                  } else
                  {
                     res = false;
                     break;
                  }
               } else                          //key record_delimiter value
               {
                  p_equal = strchr(p_start, keyvalue_delimiter);
                  if ( (p_equal != NULL) && (p_equal < p_amp) )
                  {
                     if ((p_equal - p_start) > 0)
                     {
                        key = Stroka(p_start, p_equal - p_start);
                        if ((p_amp - p_equal - 1) > 0)
                           value = Stroka(p_equal + 1, p_amp - p_equal - 1);

                        it = paramlist.find(key);
                        if (it == paramlist.end())
                        {
                           paramlist[key] = value;

                        } else
                        {
                           // повторные запросы игнорируем
                           //res = false;
                           //break;
                        }

                     } else
                     {
                        res = false;
                        break;
                     }
                  } else
                  {
                     res = false;
                     break;
                  }
               }
            }
            if (p_amp < p_end)
               p_start = p_amp + 1;
            else
               break;
         }
     }
  } else
     res = false;

  return res;
}*/

bool TShingleStorageMain::ParseRequestParamsBaseList(const TString& request, TReqstParamsList& paramlist, char record_delimiter, char keyvalue_delimiter) {
    bool res = true;
    const char* p_start = NULL;
    const char* p_end = NULL;
    const char* p_amp = NULL;
    const char* p_equal = NULL;
    TString key = "";
    TString value = "";

    if (!request.empty()) {
        p_start = request.c_str();
        p_end = request.c_str() + request.length();
        while (true) {
            if (p_start == p_end)
                break;
            key = "";
            value = "";
            p_amp = strchr(p_start, record_delimiter);
            if (p_amp == NULL)
                p_amp = p_end;
            if (p_amp != NULL) {
                if (p_amp == p_start) {
                    res = false;
                    break;
                } else {
                    if (keyvalue_delimiter == 0x00) // only key
                    {
                        if (p_amp > p_start) {
                            key = TString(p_start, p_amp - p_start);
                            paramlist.push_back(TReqstParamsStruct(key, ""));

                        } else {
                            res = false;
                            break;
                        }
                    } else // key record_delimiter value
                    {
                        p_equal = strchr(p_start, keyvalue_delimiter);
                        if ((p_equal != NULL) && (p_equal < p_amp)) {
                            if ((p_equal - p_start) > 0) {
                                key = TString(p_start, p_equal - p_start);
                                if ((p_amp - p_equal - 1) > 0)
                                    value = TString(p_equal + 1, p_amp - p_equal - 1);

                                paramlist.push_back(TReqstParamsStruct(key, value));

                            } else {
                                res = false;
                                break;
                            }
                        } else {
                            res = false;
                            break;
                        }
                    }
                }
                if (p_amp < p_end)
                    p_start = p_amp + 1;
                else
                    break;
            }
        }
    } else
        res = false;

    return res;
}

const char* GetMinAddress2(const char* start, const char* ident, const char* pmin) {
    const char* res = NULL;
    const char* p = NULL;

    p = strstr(start, ident);
    if (p != NULL) {
        if (pmin == NULL)
            pmin = p;
        else if (p < pmin)
            pmin = p;
    }
    res = pmin;

    return res;
}

bool TShingleStorageMain::ParseRequestA(const TString& NumbRequest, const char* s, TWStroka& result, bool old, const TString& abonent_ip, int thread_index, TProtokolType prtype) {
    bool res = false;
    const char* remote_server_ident = "srvid=";
    const char* shingle_put_ident = "shput=";
    const char* shingle_get_ident = "shget=";
    const char* storage_get_ident = "stget=";
    const char* shingle_rpl_ident = "shrpl=";
    const char* shingle_prxget_ident = "shprxget=";
    const char* shingle_prxput_ident = "shprxput=";
    TString remote_server_str = "";
    TString shingle_put_str = "";
    TString shingle_get_str = "";
    TString storage_get_str = "";
    TString shingle_rpl_str = "";
    TString shingle_prxget_str = "";
    TString shingle_prxput_str = "";
    const char* p = NULL;
    const char* pbegin = s;
    const char* pend = NULL;
    const char* pmin = NULL;
    ui32 strsize = 0;
    TShingleStorageMain::TWStroka resitem;
    ui32 parse_tick = 0;

    result.Clear();
    if (s != NULL) {
        pend = s + strlen(s);

        parse_tick = CShingleTime::GetMs();
        //shingle_put_str
        p = strstr(pbegin, shingle_put_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_put_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_put_ident) + 1;
            if (strsize > 0)
                shingle_put_str = TString(p + strlen(shingle_put_ident) - 1, strsize);
        }
        //shingle_get_str
        p = strstr(pbegin, shingle_get_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_get_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_get_ident) + 1;
            if (strsize > 0)
                shingle_get_str = TString(p + strlen(shingle_get_ident) - 1, strsize);
        }
        //storage_get_str
        p = strstr(pbegin, storage_get_ident);
        if (p != NULL) {
            p = p + 1;
            storage_get_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(storage_get_ident) + 1;
            if (strsize > 0)
                storage_get_str = TString(p + strlen(storage_get_ident) - 1, strsize);
        }
        //shingle_rpl_str
        p = strstr(pbegin, shingle_rpl_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_rpl_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_rpl_ident) + 1;
            if (strsize > 0)
                shingle_rpl_str = TString(p + strlen(shingle_rpl_ident) - 1, strsize);
        }
        //shingle_prxget_str
        p = strstr(pbegin, shingle_prxget_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_prxget_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_prxget_ident) + 1;
            if (strsize > 0)
                shingle_prxget_str = TString(p + strlen(shingle_prxget_ident) - 1, strsize);
        }
        //shingle_prxput_str
        p = strstr(pbegin, shingle_prxput_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_prxput_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_prxput_ident) + 1;
            if (strsize > 0)
                shingle_prxput_str = TString(p + strlen(shingle_prxput_ident) - 1, strsize);
        }
        //remote_server_str
        p = strstr(pbegin, remote_server_ident);
        if (p != NULL) {
            p = p + 1;
            remote_server_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, remote_server_ident, pmin);
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            pmin = GetMinAddress2(p, shingle_rpl_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxget_ident, pmin);
            pmin = GetMinAddress2(p, shingle_prxput_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(remote_server_ident) + 1;
            if (strsize > 0)
                remote_server_str = TString(p + strlen(remote_server_ident) - 1, strsize);
            if ((!remote_server_str.empty()) && (remote_server_str[remote_server_str.length() - 1] == '&'))
                remote_server_str = remote_server_str.substr(0, remote_server_str.length() - 1);
        }

        parse_tick = CShingleTime::GetMs() - parse_tick;
        result.m_debug = result.m_debug + "PARSE:" + IntToStroka(parse_tick) + ";";

        if (!shingle_put_str.empty()) {
            res = true;
            resitem = ParseShPutRequest(NumbRequest, shingle_put_ident, shingle_put_str, old, abonent_ip, thread_index, prtype);
            result.m_answer = result.m_answer + resitem.m_answer;
            result.m_debug = result.m_debug + resitem.m_debug;
        }

        if (!shingle_get_str.empty()) {
            res = true;
            resitem = ParseShGetRequest(NumbRequest, shingle_get_ident, shingle_get_str, old, abonent_ip, thread_index, prtype);
            result.m_answer = result.m_answer + resitem.m_answer;
            result.m_debug = result.m_debug + resitem.m_debug;
        }

        if (!storage_get_str.empty()) {
            res = true;
            resitem = ParseStGetRequest(NumbRequest, storage_get_ident, storage_get_str, old, abonent_ip, thread_index, prtype);
            result.m_answer = result.m_answer + resitem.m_answer;
            result.m_debug = result.m_debug + resitem.m_debug;
        }

        if (!shingle_rpl_str.empty()) {
            res = true;
            if ((!m_server_id.empty()) && (m_server_id == remote_server_str)) {
                result.m_answer = result.m_answer + "<SRVID:" + m_server_id + ">";

            } else {
                resitem = ParseShRplRequest(NumbRequest, abonent_ip, shingle_rpl_ident, shingle_rpl_str, remote_server_str);
                result.m_answer = result.m_answer + resitem.m_answer;
                result.m_debug = result.m_debug + resitem.m_debug;
            }
        }

        if (!shingle_prxget_str.empty()) {
            res = true;
            resitem = ParseShPrxGetRequest(NumbRequest, abonent_ip, shingle_prxget_ident, shingle_prxget_str);
            result.m_answer = result.m_answer + resitem.m_answer;
            result.m_debug = result.m_debug + resitem.m_debug;
        }

        if (!shingle_prxput_str.empty()) {
            res = true;
            resitem = ParseShPrxPutRequest(NumbRequest, abonent_ip, shingle_prxput_ident, shingle_prxput_str);
            result.m_answer = result.m_answer + resitem.m_answer;
            result.m_debug = result.m_debug + resitem.m_debug;
        }
    }

    return res;
}

TShingleStorageMain::TWStroka TShingleStorageMain::ParseShGetRequest(const TString& NumbRequest, const char* ident, const TString& request, bool old, const TString& abonent_ip, int thread_index, TProtokolType prtype) {
    // shingle-type=uniq_shingle-uniq_shingle_type&shingle-type=uniq_shingle-uniq_shingle_type&
    TWStroka res;
    TReqstParamsList paramlist;
    TReqstParamsList::iterator it;
    TString key = "";
    TString value = "";
    ui64 shingle_t = 0;
    ui32 shtype_t = 0;
    ui64 shingle = 0;
    ui16 shtype = 0;
    TShingleStat shstat;
    TShinglePutDataList shinglelist;
    bool is_proxy_mode = false;
    ui64 uniq_shingle = 0;
    ui32 uniq_shingle_type = 0;

    if ((mode == MPROXY) && (PoolProxy != NULL))
        is_proxy_mode = true;

    res.Clear();
    if (ParseRequestParamsBaseList(request, paramlist, '&', '=')) {
        it = paramlist.begin();
        while (it != paramlist.end()) {
            key = (*it).first;
            value = (*it).second;
            if (!key.empty()) {
                shingle_t = 0;
                shtype_t = 0;

                if (sscanf(key.c_str(), "%" PRIx64 "-%u", &shingle_t, &shtype_t) == 2) {
                    switch (prtype) {
                        case VP_OLD:
                            shingles_shget_old.AddRequest();
                            break;
                        case VP_NEW:
                            shingles_shget_new.AddRequest();
                            break;
                    };

                    if (!value.empty()) {
                        if (sscanf(value.c_str(), "%" PRIx64 "-%u", &uniq_shingle, &uniq_shingle_type) != 2) {
                            uniq_shingle = 0;
                            uniq_shingle_type = 0;
                        }
                    }

                    shingle = shingle_t;
                    shtype = shtype_t;

                    if (is_proxy_mode)
                        shinglelist.push_back(TShinglePutData(false, shingle, shtype, 0, UNKNOWN, PERSUNDEF, 0));

                    shstat = Get(abonent_ip, shingle, shtype, uniq_shingle, uniq_shingle_type);

                    if (old)
                        res.m_answer = res.m_answer + shstat.GetOldAnswer();
                    else
                        res.m_answer = res.m_answer + shstat.GetFullAnswer();

                    res.m_debug = res.m_debug + shstat.GetDebugInfo();
                }
            }

            ++it;
        }
    }

    if ((mode == MPROXY) && (PoolProxy != NULL))
        PoolProxy->AddRequest(shinglelist, abonent_ip, thread_index);

    return res;
}

TShingleStorageMain::TWStroka TShingleStorageMain::ParseShPutRequest(const TString& NumbRequest, const char* ident, const TString& request, bool old, const TString& abonent_ip, int thread_index, TProtokolType prtype) {
    // shingle-type=count-spamtype-pers,attendant_shingle&shingle-type=count-spamtype-pers,attendant_s
    TWStroka res;
    TReqstParamsList paramlist;
    TReqstParamsList::iterator it;
    TString key = "";
    TString value = "";
    ui64 shingle_t = 0;
    ui32 shtype_t = 0;
    ui64 att_shingle_t = 0;
    int att_shingle_type_t = 0;
    ui32 count_t = 0;
    ui32 spamtype_t = 0;
    ui32 pers_t = 0;
    ui64 shingle = 0;
    ui16 shtype = 0;
    ui64 att_shingle = 0;
    int att_shingle_type = 0;
    ui16 count = 0;
    TShingleSpamType spamtype = UNKNOWN;
    TPersType pers = PERSUNDEF;
    int parse_param_count = 0;
    TShingleStat shstat;
    TShinglePutDataList shinglelist;
    bool is_proxy_mode = false;

    if ((mode == MPROXY) && (PoolProxy != NULL))
        is_proxy_mode = true;

    res.Clear();
    if (ParseRequestParamsBaseList(request, paramlist, '&', '=')) {
        it = paramlist.begin();
        while (it != paramlist.end()) {
            key = (*it).first;
            value = (*it).second;
            if (!key.empty() && !value.empty()) {
                shingle_t = 0;
                shtype_t = 0;
                att_shingle_t = 0;
                att_shingle_type_t = 0;
                count_t = 0;
                spamtype_t = 0;

                if (sscanf(key.c_str(), "%" PRIx64 "-%u", &shingle_t, &shtype_t) == 2) {
                    parse_param_count = sscanf(value.c_str(), "%u-%u-%u,%" PRIx64 "-%d", &count_t, &spamtype_t, &pers_t, &att_shingle_t, &att_shingle_type_t);

                    shingle = shingle_t;
                    shtype = shtype_t;
                    count = count_t;
                    spamtype = (TShingleSpamType)spamtype_t;
                    pers = (TPersType)pers_t;
                    att_shingle = att_shingle_t;
                    att_shingle_type = att_shingle_type_t;

                    if ((parse_param_count >= 3) && (parse_param_count <= 5)) {
                        switch (prtype) {
                            case VP_OLD:
                                shingles_shput_old.AddRequest();
                                break;
                            case VP_NEW:
                                shingles_shput_new.AddRequest();
                                break;
                        };

                        if (is_proxy_mode)
                            shinglelist.push_back(TShinglePutData(true, shingle, shtype, count, spamtype, pers, att_shingle));

                        shstat = Put(abonent_ip, shingle, shtype, count, spamtype, att_shingle, att_shingle_type, pers, NumbRequest);

                        if (old)
                            res.m_answer = res.m_answer + shstat.GetOldAnswer();
                        else
                            res.m_answer = res.m_answer + shstat.GetFullAnswer();

                        res.m_debug = res.m_debug + shstat.GetDebugInfo();
                    }
                }
            }

            ++it;
        }
    }

    if ((mode == MPROXY) && (PoolProxy != NULL))
        PoolProxy->AddRequest(shinglelist, abonent_ip, thread_index);

    return res;
}

TShingleStorageMain::TWStroka TShingleStorageMain::ParseStGetRequest(const TString& NumbRequest, const char* ident, const TString& request, bool old, const TString& abonent_ip, int thread_index, TProtokolType prtype) {
    // shingle=&shingle=&
    TWStroka res;
    TReqstParamsList paramlist;
    TReqstParamsList::iterator it;
    TString key = "";
    TString value = "";
    TString st_value = "";
    ui64 shingle_t = 0;
    TShingleStat shstat;
    TShinglePutDataList shinglelist;
    bool is_proxy_mode = false;

    if ((mode == MPROXY) && (PoolProxy != NULL))
        is_proxy_mode = true;

    res.Clear();
    if (ParseRequestParamsBaseList(request, paramlist, '&', '=')) {
        it = paramlist.begin();
        while (it != paramlist.end()) {
            key = (*it).first;
            value = (*it).second;
            if (!key.empty()) {
                shingle_t = 0;
                st_value = "";

                if (sscanf(key.c_str(), "%" PRIx64 "", &shingle_t) == 1) {
                    switch (prtype) {
                        case VP_OLD:
                            shingles_stget_old.AddRequest();
                            break;
                        case VP_NEW:
                            shingles_stget_new.AddRequest();
                            break;
                    };

                    if (is_proxy_mode)
                        shinglelist.push_back(TShinglePutData(shingle_t));

                    shstat = GetStorage(shingle_t);

                    if (old)
                        res.m_answer = res.m_answer + shstat.GetOldAnswer();
                    else
                        res.m_answer = res.m_answer + shstat.GetFullAnswer();

                    res.m_debug = res.m_debug + shstat.GetDebugInfo();
                }
            }

            ++it;
        }
    }

    if ((mode == MPROXY) && (PoolProxy != NULL))
        PoolProxy->AddRequest(shinglelist, abonent_ip, thread_index);

    return res;
}

TShingleStorageMain::TWStroka TShingleStorageMain::ParseShPrxGetRequest(const TString& NumbRequest, const TString& abonent_ip, const char* ident, const TString& request) {
    // shingle-type=&shingle-type=&
    TWStroka res;
    TReqstParamsList paramlist;
    TReqstParamsList::iterator it;
    TString key = "";
    TString value = "";
    ui64 shingle_t = 0;
    ui32 shtype_t = 0;
    ui64 shingle = 0;
    ui16 shtype = 0;
    TShingleStat shstat;
    ui64 uniq_shingle = 0;
    ui32 uniq_shingle_type = 0;

    res.Clear();
    if (ParseRequestParamsBaseList(request, paramlist, '&', '=')) {
        it = paramlist.begin();
        while (it != paramlist.end()) {
            key = (*it).first;
            value = (*it).second;
            if (!key.empty()) {
                shingle_t = 0;
                shtype_t = 0;

                if (sscanf(key.c_str(), "%" PRIx64 "-%u", &shingle_t, &shtype_t) == 2) {
                    shingles_prxget.AddRequest();

                    shingle = shingle_t;
                    shtype = shtype_t;

                    if (shtype == 0xFFFF) {
                        shstat = GetStorage(shingle);

                    } else {
                        if (!value.empty()) {
                            if (sscanf(value.c_str(), "%" PRIx64 "-%u", &uniq_shingle, &uniq_shingle_type) != 2) {
                                uniq_shingle = 0;
                                uniq_shingle_type = 0;
                            }
                        }

                        shstat = Get(abonent_ip, shingle, shtype, uniq_shingle, uniq_shingle_type);
                    }

                    res.m_answer = res.m_answer + shstat.GetFullAnswer();
                    res.m_debug = res.m_debug + shstat.GetDebugInfo();
                }
            }

            ++it;
        }
    }

    return res;
}

TShingleStorageMain::TWStroka TShingleStorageMain::ParseShPrxPutRequest(const TString& NumbRequest, const TString& abonent_ip, const char* ident, const TString& request) {
    // shingle-type=count-spamtype-pers,attendant_shingle&shingle-type=count-spamtype-pers,attendant_shingle&
    TWStroka res;
    TReqstParamsList paramlist;
    TReqstParamsList::iterator it;
    TString key = "";
    TString value = "";
    ui64 shingle_t = 0;
    ui32 shtype_t = 0;
    ui64 att_shingle_t = 0;
    int att_shingle_type_t = 0;
    ui32 count_t = 0;
    ui32 spamtype_t = 0;
    ui32 pers_t = 0;
    ui64 shingle = 0;
    ui16 shtype = 0;
    ui64 att_shingle = 0;
    int att_shingle_type = 0;
    ui16 count = 0;
    TShingleSpamType spamtype = UNKNOWN;
    TPersType pers = PERSUNDEF;
    int parse_param_count = 0;
    TShingleStat shstat;

    res.Clear();
    if (ParseRequestParamsBaseList(request, paramlist, '&', '=')) {
        it = paramlist.begin();
        while (it != paramlist.end()) {
            key = (*it).first;
            value = (*it).second;
            if (!key.empty() && !value.empty()) {
                shingle_t = 0;
                shtype_t = 0;
                att_shingle_t = 0;
                att_shingle_type_t = 0;
                count_t = 0;
                spamtype_t = 0;

                if (sscanf(key.c_str(), "%" PRIx64 "-%u", &shingle_t, &shtype_t) == 2) {
                    parse_param_count = sscanf(value.c_str(), "%u-%u-%u,%" PRIx64 "-%d", &count_t, &spamtype_t, &pers_t, &att_shingle_t, &att_shingle_type_t);

                    shingle = shingle_t;
                    shtype = shtype_t;
                    count = count_t;
                    spamtype = (TShingleSpamType)spamtype_t;
                    pers = (TPersType)pers_t;
                    att_shingle = att_shingle_t;
                    att_shingle_type = att_shingle_type_t;

                    if ((parse_param_count >= 3) && (parse_param_count <= 5)) {
                        shingles_prxput.AddRequest();

                        shstat = Put(abonent_ip, shingle, shtype, count, spamtype, att_shingle, att_shingle_type, pers, NumbRequest);

                        res.m_answer = res.m_answer + shstat.GetFullAnswer();
                        res.m_debug = res.m_debug + shstat.GetDebugInfo();
                    }
                }
            }

            ++it;
        }
    }

    return res;
}

TShingleStorageMain::TWStroka TShingleStorageMain::ParseShRplRequest(const TString& NumbRequest, const TString& abonent_ip, const char* ident, const TString& request, const TString& remote_srv_ident) {
    // shingle-type=hamcount-spamcount-dlvcount-pershamcount-persspamcount,attendant_shingle-attendant_shingle_type-spamtype,pers&shingle-type=hamcount-spamcount-dlvcount-pershamcount-persspamcount,attendant_shingle-attendant_shingle_type-spamtype-pers&
    TWStroka res;
    TReqstParamsList paramlist;
    TReqstParamsList::iterator it;
    TString key = "";
    TString value = "";
    ui64 shingle_t = 0;
    ui32 shtype_t = 0;
    ui64 att_shingle_t = 0;
    ui32 att_shingle_type_t = 0;
    ui32 att_spamtype_t = 0;
    ui32 att_pers_t = 0;
    ui32 hamcount_t = 0;
    ui32 spamcount_t = 0;
    ui32 dlvcount_t = 0;
    ui32 pershamcount_t = 0;
    ui32 persspamcount_t = 0;
    ui64 shingle = 0;
    ui16 shtype = 0;
    ui64 att_shingle = 0;
    ui32 att_shingle_type = 0;
    ui8 att_spamtype = 0;
    ui8 att_pers = 0;
    ui16 hamcount = 0;
    ui16 spamcount = 0;
    ui16 dlvcount = 0;
    ui16 pershamcount = 0;
    ui16 persspamcount = 0;
    int parse_param_count = 0;
    TShingleStat shstat;

    res.Clear();
    if (ParseRequestParamsBaseList(request, paramlist, '&', '=')) {
        it = paramlist.begin();
        while (it != paramlist.end()) {
            key = (*it).first;
            value = (*it).second;
            if (!key.empty() && !value.empty()) {
                shingle_t = 0;
                shtype_t = 0;
                att_shingle_t = 0;
                att_shingle_type_t = 0;
                hamcount_t = 0;
                spamcount_t = 0;
                dlvcount_t = 0;
                pershamcount_t = 0;
                persspamcount_t = 0;
                att_spamtype_t = 0;
                att_pers_t = 0;

                if (sscanf(key.c_str(), "%" PRIx64 "-%u", &shingle_t, &shtype_t) == 2) {
                    parse_param_count = sscanf(value.c_str(), "%u-%u-%u-%u-%u,%" PRIx64 "-%u-%u-%u", &hamcount_t, &spamcount_t, &dlvcount_t, &pershamcount_t, &persspamcount_t, &att_shingle_t, &att_shingle_type_t, &att_spamtype_t, &att_pers_t);

                    shingle = shingle_t;
                    shtype = shtype_t;
                    hamcount = hamcount_t;
                    spamcount = spamcount_t;
                    dlvcount = dlvcount_t;
                    pershamcount = pershamcount_t;
                    persspamcount = persspamcount_t;
                    att_shingle = att_shingle_t;
                    att_shingle_type = att_shingle_type_t;
                    att_spamtype = att_spamtype_t;
                    att_pers = att_pers_t;

                    if ((parse_param_count >= 5) || (parse_param_count <= 7)) {
                        shingles_rpl.AddRequest();
                        shstat = PutM(abonent_ip, shingle, shtype, hamcount, dlvcount, spamcount, pershamcount, persspamcount, att_shingle, att_shingle_type, (TShingleSpamType)att_spamtype, (TPersType)att_pers);

                        res.m_answer = res.m_answer + shstat.GetFullAnswer();
                        res.m_debug = res.m_debug + shstat.GetDebugInfo();
                    }
                }
            }

            ++it;
        }
    }

    return res;
}

bool TShingleStorageMain::ParseRequest(const TString& NumbRequest, const char* s, TWStroka& result, const TString& abonent_ip, int thread_index) {
    return ParseRequestA(NumbRequest, s, result, false, abonent_ip, thread_index, VP_NEW);
}

bool TShingleStorageMain::ParseRequestOld(const TString& NumbRequest, const char* s, TWStroka& result, const TString& abonent_ip, int thread_index) {
    bool res = false;
    const char* shingle_put_ident = "sh=put?";
    const char* shingle_get_ident = "sh=get?";
    const char* storage_get_ident = "st=get?";
    TString shingle_put_str = "";
    TString shingle_get_str = "";
    TString storage_get_str = "";
    const char* p = NULL;
    const char* pbegin = s;
    const char* pend = NULL;
    const char* pmin = NULL;
    ui32 strsize = 0;
    TString text = "";

    result.Clear();
    if (s != NULL) {
        int n = strlen(s);
        pend = s + strlen(s);
        // shingle_put_str
        p = strstr(pbegin, shingle_put_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_put_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_put_ident) + 1;
            if (strsize > 0)
                shingle_put_str = TString(p + strlen(shingle_put_ident) - 1, strsize);
        }
        // shingle_get_str
        p = strstr(pbegin, shingle_get_ident);
        if (p != NULL) {
            p = p + 1;
            shingle_get_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(shingle_get_ident) + 1;
            if (strsize > 0)
                shingle_get_str = TString(p + strlen(shingle_get_ident) - 1, strsize);
        }
        // storage_get_str
        p = strstr(pbegin, storage_get_ident);
        if (p != NULL) {
            p = p + 1;
            storage_get_str = "";
            pmin = NULL;
            pmin = GetMinAddress2(p, shingle_put_ident, pmin);
            pmin = GetMinAddress2(p, shingle_get_ident, pmin);
            pmin = GetMinAddress2(p, storage_get_ident, pmin);
            if (pmin == NULL)
                pmin = pend;
            strsize = pmin - p - strlen(storage_get_ident) + 1;
            if (strsize > 0)
                storage_get_str = TString(p + strlen(storage_get_ident) - 1, strsize);
        }

        if (!shingle_put_str.empty())
            text = text + ConvertToShPutRequest(shingle_put_str);

        if (!shingle_get_str.empty())
            text = text + ConvertToShGetRequest(shingle_get_str);

        if (!storage_get_str.empty())
            text = text + ConvertToStGetRequest(storage_get_str);

        if ((LogsGroup != NULL) && (LogsGroup->GetDebugMode()) && (LogsGroup->GetConvertOldRqstLog() != NULL))
            LogsGroup->GetConvertOldRqstLog()->WriteMessage("***** %s SRCRQST[%s] DESTRQST[shinrqst?%s]\n", NumbRequest.c_str(), s, text.c_str());

        if (!text.empty())
            res = ParseRequestA(NumbRequest, text.c_str(), result, true, abonent_ip, thread_index, VP_OLD);
    }

    return res;
}

TString TShingleStorageMain::ConvertToShGetRequest(const TString& request) {
    // shingle&type&shingle&type -> shingle-type=&shingle-type=&
    TString res = "";
    const char* p1 = NULL;
    const char* p2 = NULL;
    const char* pend = NULL;
    TString st = "";
    int count = 0;
    int cntr = 0;
    TString restmp = "";
    bool err = false;
    TString shingle_s = "";
    TString type_s = "";

    if (!request.empty()) {
        pend = request.c_str() + request.length();
        p1 = request.c_str();
        p2 = strchr(p1, '&');
        while (p2 != NULL) {
            cntr++;
            count = p2 - p1;
            if (count > 0) {
                st = TString(p1, count);
                if (!st.empty()) {
                    if ((cntr % 2) > 0)
                        shingle_s = st;
                    else {
                        type_s = st;
                        restmp = restmp + shingle_s + "-" + type_s + "=&";
                        shingle_s = "";
                        type_s = "";
                    }

                } else {
                    err = true;
                    break;
                }
            } else {
                err = true;
                break;
            }

            p1 = p2 + 1;
            if (p1 < pend) {
                p2 = strchr(p1, '&');
                if (p2 == NULL)
                    p2 = pend;
            } else {
                break;
            }
        }
    }
    if (!err && !restmp.empty()) {
        if (restmp[restmp.length() - 1] != '&')
            res = "shget=" + restmp + "&";
        else
            res = "shget=" + restmp;
    }

    return res;
}

TString TShingleStorageMain::ConvertToShPutRequest(const TString& request) {
    // shingle&type&shingle&type&spam -> shingle-type=count-spamtype-pers,attendant_shingle&shingle-type=count-spamtype,attendant_shingle&
    TString res = "";
    const char* p1 = NULL;
    const char* p2 = NULL;
    const char* pend = NULL;
    TString st = "";
    int count = 0;
    int cntr = 0;
    TString restmp = "";
    bool err = false;
    TString shingle_s = "";
    TString type_s = "";
    int spampos = -1;
    TString spamtypes = "";

    if (!request.empty()) {
        for (int j = (request.length() - 1); j >= 0; j--) {
            if ((request[j] == '&') && (j != (request.length() - 1))) {
                spampos = j;
                break;
            }
        }
        if (spampos >= 0) {
            count = request.length() - spampos - 1;
            if (request[request.length() - 1] == '&')
                count--;
            if (count > 0)
                spamtypes = TString(request.c_str() + spampos + 1, count);
        }

        if (!spamtypes.empty()) {
            pend = request.c_str() + request.length();
            p1 = request.c_str();
            p2 = strchr(p1, '&');
            while (p2 != NULL) {
                cntr++;
                count = p2 - p1;
                if (count > 0) {
                    st = TString(p1, count);
                    if (!st.empty()) {
                        if ((cntr % 2) > 0)
                            shingle_s = st;
                        else {
                            type_s = st;
                            restmp = restmp + shingle_s + "-" + type_s + "=1-" + spamtypes + "-0&";
                            shingle_s = "";
                            type_s = "";
                        }

                    } else {
                        err = true;
                        break;
                    }
                } else {
                    err = true;
                    break;
                }

                p1 = p2 + 1;
                if (p1 < pend) {
                    p2 = strchr(p1, '&');
                    if (p2 == NULL)
                        p2 = pend;
                } else {
                    break;
                }
            }
            if (shingle_s.empty() || !type_s.empty())
                err = true;
        }
    }
    if (!err && !restmp.empty()) {
        if (restmp[restmp.length() - 1] != '&')
            res = "shput=" + restmp + "&";
        else
            res = "shput=" + restmp;
    }

    return res;
}

TString TShingleStorageMain::ConvertToStGetRequest(const TString& request) {
    // shingle&shingle -> shingle=&shingle=&
    TString res = "";
    const char* p1 = NULL;
    const char* p2 = NULL;
    const char* pend = NULL;
    TString st = "";
    int count = 0;
    TString restmp = "";
    bool err = false;

    if (!request.empty()) {
        pend = request.c_str() + request.length();
        p1 = request.c_str();
        p2 = strchr(p1, '&');
        while (p2 != NULL) {
            count = p2 - p1;
            if (count > 0) {
                st = TString(p1, count);
                if (!st.empty()) {
                    restmp = restmp + st + "=&";

                } else {
                    err = true;
                    break;
                }
            } else {
                err = true;
                break;
            }

            p1 = p2 + 1;
            if (p1 < pend) {
                p2 = strchr(p1, '&');
                if (p2 == NULL)
                    p2 = pend;
            } else {
                break;
            }
        }
    }
    if (!err && !restmp.empty()) {
        if (restmp[restmp.length() - 1] != '&')
            res = "stget=" + restmp + "&";
        else
            res = "stget=" + restmp;
    }

    return res;
}

void TShingleStorageMain::ProxySetShingles(const TString& abonent_ip, TShingleStatList& shingle_list) {
    TShingleStatListIt it;

    it = shingle_list.begin();
    while (it != shingle_list.end()) {
        if (!(*it).IsEmpty()) {
            if ((*it).ShingleForm() == TShingleStat::TSHSHINGLE) {
                SetM(abonent_ip, (*it).Shingle(), (*it).Type(),
                     (*it).Ham(TODAY), (*it).Malic(TODAY), (*it).Spam(TODAY), (*it).PersHam(TODAY), (*it).PersSpam(TODAY), (*it).Weight(TODAY),
                     (*it).Ham(YESTERDAY), (*it).Malic(YESTERDAY), (*it).Spam(YESTERDAY), (*it).PersHam(YESTERDAY), (*it).PersSpam(YESTERDAY), (*it).Weight(YESTERDAY),
                     0, 0);

            } else if ((*it).ShingleForm() == TShingleStat::TSHSTORAGE) {
                AddStorage((*it).Shingle(), (*it).GetSTData());
            }
        }

        ++it;
    }
}

TShingleStat TShingleStorageMain::GetPublic(const TString& abonent_ip, ui64 shingle, ui16 type) {
    return Get(abonent_ip, shingle, type, 0, 0);
}

bool TShingleStorageMain::ClearShingle(const TString& abonent_ip, ui64 shingle, ui16 type, bool& busy) {
    return ClearShingleA(abonent_ip, shingle, type, busy);
}

void TShingleStorageMain::Midnight() {
    oldrequest_all.Midnight();
    newrequest_all.Midnight();
    shingles_shget_old.Midnight();
    shingles_shget_new.Midnight();
    shingles_shput_old.Midnight();
    shingles_shput_new.Midnight();
    shingles_stget_old.Midnight();
    shingles_stget_new.Midnight();
    shingles_rpl.Midnight();
    request_rpl.Midnight();
    request_prx.Midnight();
    shingles_prxget.Midnight();
    shingles_prxput.Midnight();

    m_cluster_delayobj_days.Midnight();
    m_rpl_delayobj_days.Midnight();

    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            typesdata[i]->SetMidnightEvent();
    }
}

void TShingleStorageMain::EventTick() {
    oldrequest_all.CalcCPS();
    newrequest_all.CalcCPS();
    shingles_shget_old.CalcCPS();
    shingles_shget_new.CalcCPS();
    shingles_shput_old.CalcCPS();
    shingles_shput_new.CalcCPS();
    shingles_stget_old.CalcCPS();
    shingles_stget_new.CalcCPS();
    shingles_rpl.CalcCPS();
    request_rpl.CalcCPS();
    request_prx.CalcCPS();
    shingles_prxget.CalcCPS();
    shingles_prxput.CalcCPS();
    LoadWeights();

    WritedumpTickEvent();
}

void TShingleStorageMain::Shutdown() {
    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            typesdata[i]->Shutdown();
    }
    SaveFollowShingles();
}

void TShingleStorageMain::GetStat(TShingleStorageStat& value) {
    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            value.typesdata[i] = typesdata[i]->GetStat();
    }
    if (SSObj != NULL)
        value.storage_stat = SSObj->GetStat();

    value.lastwritedumpevent = m_last_writedumpevent;
    value.writedumptype = writedumptype;
    value.periodwritedumpsec = periodwritedumpsec;
    value.periodwritedumpsec_slow = periodwritedumpsec_slow;
}

void TShingleStorageMain::SetWriteDumpEvent() {
    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            typesdata[i]->SetWriteDumpEvent();
    }
}

void ShedulerShinStorage(void* par) {
    ui32 writedumpdelay = 0;

    TShingleStorageMain* ss = (TShingleStorageMain*)par;
    if (ss != NULL) {
        ss->WriteThreadIDToLog();

        while (!ss->ShedulerShouldStop()) {
            ss->Sleep();
            ss->ShedulerAction();
        }

        ss->ShedulerStopped(true);
    }
}

void TShingleStorageMain::ShedulerStopped(bool Stopped) {
    m_Mutex.Acquire();
    m_StopSheduler = !Stopped;
    m_Mutex.Release();
}

bool TShingleStorageMain::ShedulerStopped() {
    return !m_StopSheduler;
}

void TShingleStorageMain::ShedulerAction() {
    delobj.DeleteAllData();

    for (int i = 0; i < MAX_COUNT_TYPES; i++) {
        if (typesdata[i] != NULL)
            typesdata[i]->ShedulerAction();
    }
}

void TShingleStorageMain::StartSheduler() {
    m_Mutex.Acquire();
    m_StartSheduler = true;
    m_StopSheduler = false;
    m_Sheduler = new TThread((TThread::TThreadProc)&ShedulerShinStorage, this);
    m_Sheduler->Start();
    m_Mutex.Release();
}

void TShingleStorageMain::StopSheduler() {
    m_Mutex.Acquire();
    m_StopSheduler = true;
    m_Mutex.Release();
}

void TShingleStorageMain::Sleep() {
    ::sleep(m_SleepTime);
}

void TShingleStorageMain::Close() {
    if (m_StartSheduler) {
        StopSheduler();
        while (!ShedulerStopped())
            ::sleep(1);
    }
    if (m_Sheduler != NULL) {
        delete m_Sheduler;
        m_Sheduler = NULL;
    }
}

void TShingleStorageMain::Cleanup(int type) {
    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            typesdata[type]->SetCleanupEvent();
    } else {
        for (int i = 0; i < MAX_COUNT_TYPES; i++) {
            if (typesdata[i] != NULL)
                typesdata[i]->SetCleanupEvent();
        }
    }
}

void TShingleStorageMain::WritedumpTickEvent() {
    if (writedumptype == 0) // записываем на диск через период времени все типы (единовременная запись всех типов)
    {
        time_t tdiff = time(NULL) - m_last_writedumpevent;
        if (tdiff >= periodwritedumpsec) {
            Writedump(-1);

            m_actionmutex.Acquire();
            m_last_writedumpevent = time(NULL);
            m_actionmutex.Release();
        }

    } else  // записываем на диск каждый тип через некоторый перид времени (размазанная запись типов)
    {
        int index = -1;

        index = wd_obg.NeedWriteDump();
        if ((index >= 0) && (index < MAX_COUNT_TYPES))
            Writedump(index);
    }
}

void TShingleStorageMain::Writedump(int type) {
    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            typesdata[type]->SetWriteDumpEvent();
    } else {
        for (int i = 0; i < MAX_COUNT_TYPES; i++) {
            if (typesdata[i] != NULL)
                typesdata[i]->SetWriteDumpEvent();
        }
    }
}

bool TShingleStorageMain::LoadWeights() {
    bool res = true;

    if ((time(NULL) - m_last_load_weight) >= 10) {
        if (!weights_dir_name.empty()) {
            char *e = NULL, *b = NULL;
            ui64 shingle = 0;
            ui8 type = 0;
            int temp_type = 0;
            float weight = 0;
            ui32 count = 0;
            ui32 bad_count = 0;
            ui32 busy_count = 0;
            ui32 update_count = 0;
            ui32 new_count = 0;
            ui32 t = 0;
            char m_RecordBuffer[256];
            FILE* m_WeightsFile = NULL;
            int lerr = 0;

            TFileEntitiesList fileList(TFileEntitiesList::EM_FILES_SLINKS);

            fileList.Fill(weights_dir_name, "", "", 1);

            t = CShingleTime::GetMs();

            while (res) {
                const char * fname = fileList.Next();

                if (NULL == fname)
                    break;

                const TString m_WeightsFileName = JoinFsPaths(weights_dir_name, fname);

//--- отсекаем файлы, начинающиеся с точки ---
#ifdef _win_
                char dliteral = '\\';
#else
                char dliteral = '/';
#endif
                TString m_WeightsFileName_temp = m_WeightsFileName;
                ReverseInPlace(m_WeightsFileName_temp);
                const char* pt = strchr(m_WeightsFileName_temp.c_str(), dliteral);
                if ((pt != NULL) && (pt > m_WeightsFileName_temp.c_str())) {
                    char symb_t = *(pt - 1);
                    if (symb_t == '.')
                        continue;
                }
                //--------------------------------------------

                m_WeightsFile = fopen(m_WeightsFileName.c_str(), "r");

                if (NULL == m_WeightsFile) {
                    if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                        LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KERROR, "Can't open weights file %s", m_WeightsFileName.c_str());
                    return false;
                }

                while (fgets(m_RecordBuffer, sizeof(m_RecordBuffer), m_WeightsFile)) {
                    b = m_RecordBuffer;
                    e = strchr(m_RecordBuffer, ' ');
                    if (NULL == e) {
                        res = false;
                        break;
                    }
                    shingle = strtoull(b, &e, 16);

                    b = ++e;
                    e = strchr(m_RecordBuffer, ' ');
                    if (NULL == e) {
                        res = false;
                        break;
                    }

                    temp_type = strtoul(b, &e, 10);
                    if (temp_type >= MAX_COUNT_TYPES) {
                        res = false;
                        break;
                    }
                    type = (ui8)temp_type;

                    b = ++e;
                    e = strchr(m_RecordBuffer, '\n');
                    if (NULL == e) {
                        res = false;
                        break;
                    }
                    weight = (float)strtod(b, &e);

                    count++;

                    lerr = SetWeight("service", shingle, type, weight);
                    switch (lerr) {
                        case 1:
                            busy_count++;
                            break;
                        case 2:
                            update_count++;
                            break;
                        case 3:
                            new_count++;
                            break;
                        default:
                            bad_count++;
                    };
                }
                fclose(m_WeightsFile);
                remove(m_WeightsFileName.c_str());
            }

            if (res) {
                char text[512];

                t = CShingleTime::GetMs() - t;
                memset(text, 0, sizeof(text));
                snprintf(text, sizeof(text) - 1, "Weights loaded - OK (all=%d: busy=%u, update=%u, new=%u, bad=%u) in %u msec", count, busy_count, update_count, new_count, bad_count, t);
                if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                    LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KMESSAGE, "%s", text);

            } else {
                if ((LogsGroup != NULL) && (LogsGroup->GetShinglesLog() != NULL))
                    LogsGroup->GetShinglesLog()->WriteMessageAndDataStatus(KERROR, "Can't load weights");
            }

        } else {
            res = false;
        }

        m_last_load_weight = time(NULL);
        m_last_load_weight = time(NULL);
    }

    return res;
}

void TShingleStorageMain::WriteThreadIDToLog() {
    int thread_id = -1;

#ifndef _win_
    thread_id = syscall(SYS_gettid);
#endif

    if ((LogsGroup != NULL) && (LogsGroup->GetDebugInfoLog() != NULL))
        LogsGroup->GetDebugInfoLog()->WriteMessageAndDataStatus(KMESSAGE, "THREADID % 9d: SHINSTORAGE_SHEDULER_THREAD", thread_id);
}

bool TShingleStorageMain::ExistsFollowShingle(ui64 shingle, int type, ui32& tick) {
    bool res = false;

    tick = 0;
    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            res = typesdata[type]->ExistsFollowShingle(shingle, tick);
    }

    return res;
}

bool TShingleStorageMain::AddFollowShingle(ui64 shingle, int type) {
    bool res = false;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            res = typesdata[type]->AddFollowShingle(shingle);
    }
    SaveFollowShingles();

    return res;
}

bool TShingleStorageMain::DeleteFollowShingle(ui64 shingle, int type) {
    bool res = false;

    if ((type >= 0) && (type < MAX_COUNT_TYPES)) {
        if (typesdata[type] != NULL)
            res = typesdata[type]->DeleteFollowShingle(shingle);
    }
    SaveFollowShingles();

    return res;
}

void TShingleStorageMain::GetListFollowShingles(TFollowShinglesList& list) {
    list.clear();
    for (int i = 0; i < MAX_COUNT_TYPES; i++)
        typesdata[i]->GetListFollowShingles(list);
}

void TShingleStorageMain::SaveFollowShingles() {
    FILE* handle = NULL;
    TFollowShinglesList list;
    TFollowShinglesListIt it;
    ui32 count = 0;
    TString shingles_str = "";
    TString filename = m_followsh_filename;

    if (!filename.empty()) {
        handle = fopen(filename.c_str(), "wb");
        if (handle != NULL) {
            GetListFollowShingles(list);
            it = list.begin();
            while (it != list.end()) {
                fprintf(handle, "%016" PRIx64 "-%03d: %llu\n", (*it).m_shingle, (*it).m_type, (*it).m_get_count);
                count = IncMax32(count, 1);
                if (!shingles_str.empty())
                    shingles_str = shingles_str + ", 0x" + ShingleToStroka((*it).m_shingle) + "-" + IntToStroka((*it).m_type);
                else
                    shingles_str = shingles_str + "0x" + ShingleToStroka((*it).m_shingle) + "-" + IntToStroka((*it).m_type);

                ++it;
            }

            fclose(handle);
            handle = NULL;

            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "FOLLOWSH: write %u record (%s)", count, shingles_str.c_str());

        } else {
            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "FOLLOWSH: cant's open file for write (%s)", filename.c_str());
        }

    } else {
        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "FOLLOWSH: write, file unknown");
    }
}

void TShingleStorageMain::ReadFollowShingles(const TString& filename) {
    FILE* handle = NULL;
    char tbuff[256];
    ui64 shingle = 0;
    int type = 0;
    ui32 count = 0;
    TString shingles_str = "";

    if (!filename.empty()) {
        handle = fopen(filename.c_str(), "rb");
        if (handle != NULL) {
            while (fgets(tbuff, sizeof(tbuff) - 1, handle)) {
                if (sscanf(tbuff, "%" PRIx64 "-%d", &shingle, &type) == 2) {
                    AddFollowShingle(shingle, type);
                    count = IncMax32(count, 1);
                    if (!shingles_str.empty())
                        shingles_str = shingles_str + ", 0x" + ShingleToStroka(shingle) + "-" + IntToStroka(type);
                    else
                        shingles_str = shingles_str + "0x" + ShingleToStroka(shingle) + "-" + IntToStroka(type);
                }
            }

            fclose(handle);
            handle = NULL;

            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "FOLLOWSH: read %u record (%s)", count, shingles_str.c_str());

        } else {
            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "FOLLOWSH: cant's open file for read (%s)", filename.c_str());
        }

    } else {
        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "FOLLOWSH: read, file unknown");
    }
}

void TShingleStorageMain::SetLongDelayMs(ui32 delay_ms) {
    m_longdelayMutex.Acquire();

    m_longdelayms = delay_ms;

    m_longdelayMutex.Release();
}

ui32 TShingleStorageMain::GetLongDelayMs() {
    ui32 res = 0;

    // m_longdelayMutex.Acquire();

    res = m_longdelayms;

    // m_longdelayMutex.Release();

    return res;
}

//********************************************************************************************************************************
