//
// Created by luckybug on 03.11.17.
//

#include <library/cpp/http/server/http_ex.h>
#include <library/cpp/http/server/response.h>

#include <util/datetime/base.h>

#include "handler_server.h"
#include "events.h"

class TRequest : public TClientRequest {
public:
    bool Reply(void* tsr) override {
        TStringBuf view = TStringBuf(RequestString).After(' ').RBefore(' ');

        TCgiParameters CgiParam;
        TStringBuf request, cgi;
        if (!view.TrySplit('?', request, cgi))
            request = view;
        else
            CgiParam.Scan(cgi);

        const auto start = Now();

        try {
            Master->Reply(request, THandleContext(request, CgiParam, ParsedHeaders, Output(), Input()), tsr);

            const auto processingDuration = (Now() - start).MilliSeconds();
            Master->AccessLog << request << ' ' << CgiParam.QuotedPrint() << ' ' << HTTP_OK << ' ' << processingDuration;
            Master->RaiseEvent(request, ToString<int>(HTTP_OK), Now() - start);

        } catch (const THttpError& e) {
            const auto processingDuration = (Now() - start).MilliSeconds();
            Master->AccessLog << request << ' ' << CgiParam.QuotedPrint() << ' ' << e.code << ' ' << processingDuration;
            Master->ErrorLog << request << ' ' << CgiParam.QuotedPrint() << ' ' << e.code << ' ' << processingDuration << ' ' << CurrentExceptionMessage() << TBackTrace::FromCurrentException().PrintToString();
            ProcessException(e.code, request, Now() - start);
        } catch (...) {
            const auto processingDuration = (Now() - start).MilliSeconds();
            Master->AccessLog << request << ' ' << CgiParam.QuotedPrint() << ' ' << HTTP_INTERNAL_SERVER_ERROR << ' ' << processingDuration;
            Master->ErrorLog << request << ' ' << CgiParam.QuotedPrint() << " - " << processingDuration << ' ' << CurrentExceptionMessage() << TBackTrace::FromCurrentException().PrintToString();
            ProcessException(HTTP_INTERNAL_SERVER_ERROR, request, Now() - start);
        }

        return true;
    }

    explicit TRequest(THandlerServer* master) noexcept : Master(master) {}

private:
    void ProcessException(HttpCodes code, const TStringBuf& request, TDuration duration) {
        const auto msg = CurrentExceptionMessage() + TBackTrace::FromCurrentException().PrintToString();;
        THttpResponse(HTTP_INTERNAL_SERVER_ERROR).SetContent(msg).OutTo(Output());

        Master->RaiseEvent(request, ToString<int>(code), duration);
    }

private:
    THandlerServer* Master{};
};

THandlerServer& THandlerServer::On(const TString& request, THandlerPtr handler) {
    Handles.emplace_back(request, handler);
    ResortCbs();
    ErrorLog << TLOG_INFO << Handles.size() << " register " << request;
    return *this;
}

THandlerServer& THandlerServer::On(const TString& request, THandlerFunction handler) {
    return On(request, MakeAtomicShared<TLambdaFunctor>(std::move(handler)));
}

THandlerServer& THandlerServer::OnEvent(TEventHandlerPtr handler) {
    EventHandler = std::move(handler);
    return *this;
}

void THandlerServer::RaiseEvent(const TStringBuf& handle, const TStringBuf& status, TDuration duration) {
    if (!EventHandler)
        return;

    try {
        EventHandler->OnEvent(handle, status, duration);
    } catch (...) {
        ErrorLog << TLOG_ERR << "error in raise event: " << CurrentExceptionMessage() << TBackTrace::FromCurrentException().PrintToString();
    }
}

void THandlerServer::RaiseEvent(const TString& comment) {
    RaiseEvent({}, comment, {});
}

void THandlerServer::ReopenLog() {
    AccessLog.ReopenLog();
    ErrorLog.ReopenLog();
    ErrorLog << TLOG_INFO << "*log reopen";
}

void THandlerServer::OnFailRequest(int failstate) {
    RaiseEvent(NEvents::FAIL_REQ);
    ErrorLog << TLOG_INFO << "fail request: " << failstate << ' ' << this->GetError()
         << "; client count: " << GetClientCount()
         << "; req queue size: " << GetRequestQueueSize()
         << "; fail queue size: " << GetFailQueueSize();
}

void THandlerServer::OnFailRequestEx(const TFailLogData& d) {
    RaiseEvent(NEvents::FAIL_REQ);
    ErrorLog << TLOG_INFO << "fail request: " << d.failstate << ' ' << d.url << ' ' << this->GetError()
         << "; client count: " << GetClientCount()
         << "; req queue size: " << GetRequestQueueSize()
         << "; fail queue size: " << GetFailQueueSize();
}

void THandlerServer::OnException() {
    RaiseEvent(NEvents::EXCEPTION);
    ErrorLog << TLOG_ERR << __FILE__ << ':' << __LINE__ << ' ' << CurrentExceptionMessage() << TBackTrace::FromCurrentException().PrintToString();;
}

void THandlerServer::OnMaxConn() {
    RaiseEvent(NEvents::MAX_CONN);
    ErrorLog << TLOG_ERR << "max conn";
}

void THandlerServer::OnListenStart() {
    ErrorLog << TLOG_INFO << "start listen";
}

void THandlerServer::OnListenStop() {
    ErrorLog << TLOG_INFO << "stop listen";
}

void THandlerServer::OnWait() {
    ErrorLog << TLOG_INFO << "wait ";
}

void THandlerServer::Reply(const TStringBuf& request, THandleContext&& handleContext, void* tsr) {
    auto it = std::find_if(Handles.cbegin(), Handles.cend(), [&request](const std::pair<TString, THandlerPtr>& p) {
        const bool r = request.StartsWith(p.first);
        return r;
    });

    if (Handles.cend() == it)
        ythrow THttpError{HTTP_NOT_FOUND} << request << " not found";

    handleContext.request.SkipPrefix(it->first);
    it->second->Reply(std::move(handleContext), tsr);
}

TClientRequest* THandlerServer::CreateClient() {
    return new TRequest(this);
}

void THandlerServer::ResortCbs() {
    std::sort(Handles.begin(), Handles.end(), [](const std::pair<TString, THandlerPtr>& a, const std::pair<TString, THandlerPtr>& b) {
        return a.first.Size() > b.first.Size();
    });
}

void TStopServerHandler::Reply(THandleContext&& handleContext, void*) {
    PingHandler->Disable();
    // L3 balancers check backend availability every 10 seconds
    Sleep(TDuration::Seconds(15));
    Server->Shutdown();
    THttpResponse(HTTP_OK).OutTo(handleContext.output);
};

TStopServerHandler::TStopServerHandler(THandlerServer* server, TAtomicSharedPtr<TPingHandler> pingHandler) noexcept
    : Server(server)
    , PingHandler(pingHandler)
{
}

void TReopenLogHandler::Reply(THandleContext&& handleContext, void*) {
    Server->ReopenLog();
    THttpResponse(HTTP_OK).OutTo(handleContext.output);
};

TReopenLogHandler::TReopenLogHandler(THandlerServer* server) noexcept
    : Server(server) {
}

TPingHandler::TPingHandler()
    : Enabled(1)
{
}

void TPingHandler::Reply(THandleContext&& handleContext, void*) {
    if (IsEnabled()) {
        THttpResponse(HTTP_OK).SetContent(TString{"pong"}).OutTo(handleContext.output);
    } else {
        THttpResponse(HTTP_SERVICE_UNAVAILABLE).SetContent(TString{"ping disabled"}).OutTo(handleContext.output);
    }
}

bool TPingHandler::IsEnabled() const {
    return AtomicGet(Enabled);
}

void TPingHandler::Disable() {
    AtomicSet(Enabled, 0);
}

