#pragma once

#include "options.h"

#include <mail/so/spamstop/tools/so-common/prof.h>
#include <mail/so/spamstop/tools/so-common/sputil.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/logger/filter.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/logger/stream.h>

#include <util/generic/fwd.h>
#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/stream/str.h>
#include <util/string/builder.h>
#include <library/cpp/deprecated/atomic/atomic.h>

#include <utility>

struct THandleContext {
    THandleContext(
        TStringBuf request,
        const TCgiParameters& cgiParameters,
        const TVector<std::pair<TString, TString>>& headers,
        THttpOutput& output,
        THttpInput& input) noexcept
        : request(request)
        , cgiParameters(cgiParameters)
        , headers(headers)
        , output(output)
        , input(input) {
    }

    THandleContext(THandleContext&&) noexcept = default;

    TStringBuf request;
    const TCgiParameters& cgiParameters;
    const TVector<std::pair<TString, TString>>& headers;
    THttpOutput& output;

    TBlob ExtractData() const {
        if (input.HasExpect100Continue())
            output.SendContinue();

        ui64 contentLength = 0;
        if (!input.ContentEncoded() && input.GetContentLength(contentLength)) {
            TBuffer buf(contentLength);
            buf.Resize(input.Load(buf.Data(), contentLength));
            return TBlob::FromBuffer(buf);
        } else {
            return TBlob::FromStream(input);
        }
    }

private:
    THttpInput& input;
};

Y_DECLARE_OUT_SPEC(inline, THandleContext, s, v) {
    s << "request: " << v.request << ";"
                                        "params: "
      << v.cgiParameters.QuotedPrint() << ";"
                                          "headers: [";

    for (const auto& h : v.headers) {
        s << h.first << ':' << h.second << ',';
    }

    s << ']';
}

struct THttpError : public yexception {
    explicit THttpError(HttpCodes code) noexcept
        : code(code) {
    }
    HttpCodes code;
};


class THandlerFunctor {
public:
    virtual void Reply(THandleContext&& handleContext, void* tsr) = 0;
    virtual ~THandlerFunctor() = default;
};

class TEventHandler {
public:
    virtual void OnEvent(const TStringBuf& handle, const TStringBuf& status, TDuration duration) = 0;
    virtual ~TEventHandler() = default;
};

using THandlerFunction = std::function<void(THandleContext&& handleContext, void* tsr)>;
class TLambdaFunctor : public THandlerFunctor {
public:
    void Reply(THandleContext&& handleContext, void* tsr) override {
        F(std::move(handleContext), tsr);
    }

    explicit TLambdaFunctor(THandlerFunction f)
        : F(std::move(f)) {
    }

private:
    THandlerFunction F;
};

using THandlerPtr = TAtomicSharedPtr<THandlerFunctor>;
using TEventHandlerPtr = TAtomicSharedPtr<TEventHandler>;

class TClientRequest;
class THandlerServer : private THttpServer::ICallBack, public THttpServer {
    friend class TRequest;

public:
    THandlerServer& On(const TString& request, THandlerPtr handler);
    THandlerServer& On(const TString& request, THandlerFunction handler);

    THandlerServer& OnEvent(TEventHandlerPtr handler);
    void RaiseEvent(const TStringBuf& handle, const TStringBuf& status, TDuration duration);
    void RaiseEvent(const TString& comment);

    void ReopenLog();

    explicit THandlerServer(
        const TServerOptions& options = {},
        IThreadFactory* pool = nullptr)
        : THttpServer(this, options, pool)
        , AccessLog(CreateOwningThreadedLogBackend(options.AccessLog))
        , ErrorLog(CreateOwningThreadedLogBackend(options.ErrorLog)) {
        constexpr auto formatter = [](ELogPriority priority, TStringBuf data) -> TString {
            return TStringBuilder{} << Now() << ' ' << priority << ':' << data << '\n';
        };

        AccessLog.SetFormatter(formatter);
        ErrorLog.SetFormatter(formatter);
    }

private:
    void OnFailRequest(int failstate) override;
    void OnFailRequestEx(const TFailLogData& d) override;
    void OnException() override;
    void OnMaxConn() override;
    void OnListenStart() override;
    void OnListenStop() override;
    void OnWait() override;
    TClientRequest* CreateClient() override;
    void ResortCbs();

    void Reply(const TStringBuf& request, THandleContext&& handleContext, void* tsr);

private:
    TVector<std::pair<TString, THandlerPtr>> Handles;
    TEventHandlerPtr EventHandler;

    TLog AccessLog;
    TLog ErrorLog;
};

class TPingHandler : public THandlerFunctor {
public:
    TPingHandler();

    void Reply(THandleContext&& handleContext, void* tsr) override;

    bool IsEnabled() const;

    void Disable();

private:
    TAtomic Enabled;
};

class TStopServerHandler : public THandlerFunctor {
public:
    void Reply(THandleContext&& handleContext, void* tsr) override;
    explicit TStopServerHandler(THandlerServer* server, TAtomicSharedPtr<TPingHandler> pingHandler) noexcept;

private:
    THandlerServer* Server;
    TAtomicSharedPtr<TPingHandler> PingHandler;
};

class TReopenLogHandler : public THandlerFunctor {
public:
    void Reply(THandleContext&& handleContex, void* tsrt) override;
    explicit TReopenLogHandler(THandlerServer* server) noexcept;

private:
    THandlerServer* Server;
};

