
#include "static_serve.h"
#include <library/cpp/http/server/response.h>
#include <util/stream/file.h>
#include <util/generic/bt_exception.h>

const TString & MimeTypeByExtension(const TString & ext) {

    static const TString text = "text/plain";
    static const THashMap<TString, TString> types = {
            {"js", "application/javascript"},
            {"css", "text/css"},
            {"png", "image/png"},
            {"html", "text/html"},
    };

    auto it = types.find(ext);

    if(types.cend() == it)
        return text;

    return it->second;
}

void TStaticServe::Reply(THandleContext && handleContext, void*) {

    TFsPath f = rootDir / handleContext.request;
    f.Fix();

    if(!f.Exists() || !f.IsFile()) {
        ythrow THttpError{HTTP_NOT_FOUND} << handleContext.request << " not found";
    }

    TGuard<TMutex> cacheGuard(cacheMutex);

    auto cachedIt = fileCache.find(f.GetPath());

    if(fileCache.cend() == cachedIt) {
        bool success = false;
        std::tie(cachedIt, success) = fileCache.emplace(f.GetPath(), TBlob::FromFile(f));

        if(!success)
            ythrow TWithBackTrace<yexception>() << "cannot create cache for " << f;
    }

    THttpResponse(HTTP_OK).SetContent(
            {(char*)cachedIt->second.Data(), cachedIt->second.Size()},
            MimeTypeByExtension(f.GetExtension())
    ).OutTo(handleContext.output);
}

void TStaticServe::CleanUp() {
    TGuard<TMutex> cacheGuard(cacheMutex);

    fileCache.clear();
}

TStaticServe::TStaticServe(const TFsPath & rootDir) : rootDir(rootDir) {
    Y_VERIFY(rootDir.Exists(), "path must exists");
    Y_VERIFY(rootDir.IsDirectory(), "path must be directory");
}
