#pragma once

#include "handler_server.h"
#include <mail/so/spamstop/tools/so-common/stats_consumer.h>
#include <mail/so/spamstop/tools/so-common/hist.h>


template<class TResource, class TMaster>
class TContextManager {
public:
    struct TExecutionContext {
        TExecutionContext(TMaster& master, TResource& resource) noexcept : Master(master), Resource(resource) {}
        TMaster& Master;
        TResource& Resource;
    };

protected:
    TList<TResource> Resources;
    TRWMutex Mutex;

public:
    TExecutionContext* CreateThreadSpecificResource() {
        TWriteGuard g(Mutex);
        return new TExecutionContext(static_cast<TMaster&>(*this), Resources.emplace_back());
    }

    void DestroyThreadSpecificResource(TExecutionContext* context) {
        TWriteGuard g(Mutex);

        Resources.remove_if([context](const auto& h) {
            return std::addressof(h) == std::addressof(context->Resource);
        });
        delete context;
    }
};

template<class ... TManagers>
class TMultiContextManager{
    template<class TManagersTuple>
    static auto CreateHelper(TManagersTuple&& managers) {
        return std::apply([](auto&&... m){
            return std::make_tuple(m.CreateThreadSpecificResource()...);
        }, std::forward<TManagersTuple>(managers));
    }
public:
    using TExecutionContext = decltype(CreateHelper(std::tuple<TManagers...>{}));

    TExecutionContext* CreateThreadSpecificResource() {
        return new TExecutionContext{CreateHelper(managers)};
    }

    void DestroyThreadSpecificResource(TExecutionContext* context) {
        std::apply([](auto&&... c){
            (c->Master.DestroyThreadSpecificResource(c), ...);
        }, *context);
        delete context;
    }

    template <size_t I>
    decltype(auto) Get() {
        return std::get<I>(managers);
    }

private:
    std::tuple<TManagers...> managers;
};


class TProfiledContext : public TContextManager<TStatsListSafe, TProfiledContext>{
public:
    THashMap<TString, THist<TDuration>> MakeHist(const THashMap<TString, TVector<TDuration>>& levelsBySignal);

private:
    TList<NProf::TNamedItem> GetStats();
};

template<class TContextManager>
class TContextServer : public THandlerServer {
public:

    TContextManager ContextManager;

    using THandlerServer::THandlerServer;

    void* CreateThreadSpecificResource() override {
        return ContextManager.CreateThreadSpecificResource();
    }

    void DestroyThreadSpecificResource(void* resourceRaw) override {
        auto context = reinterpret_cast<typename TContextManager::TExecutionContext*>(resourceRaw);
        ContextManager.DestroyThreadSpecificResource(context);
    }
};
