#include <util/generic/bt_exception.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include <util/stream/printf.h>
#include <library/cpp/logger/all.h>
#include <util/string/ascii.h>
#include <mail/so/libs/syslog/so_log.h>
#include "util/system/flock.h"
#include "SperrorHolder.h"

TSpLogger::TSpLogger()
{
    log_level = SO_ERRORS_AND_WARNINGS;
    utilLoggerPtr.ResetBackend(MakeHolder<TNullLogBackend>());

    is_open = false;
    use_syslog = false;
    extFilterLog = nullptr;

    pFileBackend = nullptr;
}

constexpr bool TSpLogger::CanLog(ELogPriority priority) {
    return ((priority < TLOG_NOTICE) ||
        ((priority == TLOG_NOTICE) && (log_level > SO_ERRORS_ONLY)) ||
        ((priority == TLOG_INFO) && (log_level > SO_ERRORS_AND_WARNINGS)) ||
        ((priority == TLOG_DEBUG) && (log_level > SO_LOG_ALL_BUT_DEBUG)));
}

constexpr bool TSpLogger::CanLog(int rawPriority) {
    return CanLog(static_cast<ELogPriority>(rawPriority));
}

void TSpLogger::Openlog(TSOLogLevel log_levelA, const TString& fileName, bool bAddFormatter)
{
    log_level = log_levelA;

    if (utilLoggerPtr.IsOpen() && !utilLoggerPtr.IsNullLog())
        return;

    utilLoggerPtr.CloseLog();

    if (!fileName || AsciiEqualsIgnoreCase(fileName, "stdout"))
    {
        pStreamBackend = new TStreamLogBackend(&Cout);
        utilLoggerPtr.ResetBackend(THolder<TLogBackend>(pStreamBackend));
    }
    else
    {
        pFileBackend = new TFileLogBackend(fileName);
        utilLoggerPtr.ResetBackend(THolder<TLogBackend>(pFileBackend));
    }

    if (bAddFormatter)
        utilLoggerPtr.SetFormatter(TimePidFormatter);

    is_open = true;
}

void TSpLogger::Closelog()
{
    if (utilLoggerPtr.IsOpen())
        utilLoggerPtr.CloseLog();
}

void TSpLogger::SetSPKLog(TBaseLogClass *extFilterLogA)
{
    extFilterLog = extFilterLogA;
}

void TSpLogger::splog(int priority, const char *msg, ...)
{
    if(!CanLog(priority))
        return;
    TString buff;
    {
        TStringOutput stream(buff);

        va_list args;
        va_start(args, msg);
        Printf(stream, msg, args);
        va_end(args);
    }

    if (use_syslog)
        Syslog(ELogPriority (priority)) << buff;

    if (!utilLoggerPtr.IsNullLog())
        utilLoggerPtr << buff;

    if (extFilterLog)
        extFilterLog->WriteMessageBUFF(buff.c_str(), buff.size());
}

void TSpLogger::spprint(const char *msg, ...)
{
    TString buff;
    {
        TStringOutput stream(buff);

        va_list args;
        va_start(args, msg);
        Printf(stream, msg, args);
        va_end(args);
    }
    spprintlen(buff.c_str(), buff.size());
}

void TSpLogger::spprintlen(const char *msg, size_t len)
{
    const TStringBuf view(msg, len);

    if (!utilLoggerPtr.IsNullLog())
        utilLoggerPtr << view;


    if (extFilterLog)
        extFilterLog->WriteMessageBUFF(msg, len);
}
// ------ TSpLogger end

TSOLogLevel Int2TSOLogLevel(int val) {
    if (val < SO_ERRORS_ONLY || val > SO_LOG_EVERYTHING)
        throw yexception() << "wrong log level: " << val;
    return static_cast<TSOLogLevel>(val);
}

TSpLoggers::~TSpLoggers()
{
    CloseLogs();
}

void TSpLoggers::CloseLogs()
{
    for (int i = 0; i < SO_MAX_LOGGER; i++)
        loggers[i].Closelog();
}

TSpLogger * TSpLoggers::GetSpLoggerPtr(TSOLoggers loggerID)
{
    if (loggerID < SO_MAX_LOGGER)
        return &loggers[loggerID];

    return nullptr;
}

TSpLogger *pDaemonLogger = nullptr;
TSpLogger *pFilterLogger = nullptr;
TSpLogger *pHttpLogger   = nullptr;
