#pragma once

#include <util/generic/string.h>
#include <util/generic/hash.h>
#include <util/generic/bt_exception.h>
#include <util/system/mutex.h>
#include <util/system/defaults.h>
#include <util/stream/input.h>
#include <util/stream/str.h>
#include <library/cpp/charset/doccodes.h>
#include <dict/dictutil/dictutil.h>

#include <array>

namespace DictionaryShingler {
    enum SpamType { ST_UNKNOWN = -1,
                    ST_HAM,
                    ST_SPAM,
                    ST_COMPLHAM,
                    ST_COMPLSPAM,
                    ST_VIRUS,
                    ST_FISHING,
                    ST_FISHING_YAN,
                    ST_HACKED,
                    ST_EXPERT_COMPLHAM,
                    ST_EXPERT_COMPLSPAM,
                    ST_COUNT };
    enum RequestType { RT_UNKNOWN = -1,
                       RT_ABUSE,
                       RT_PUT,
                       RT_GET,
                       RT_VIRUS,
                       RT_FISHING,
                       RT_FISHING_YAN,
                       RT_HACKED,
                       RT_COUNT };
    enum MailType { MT_INBOX = 0,
                    MT_OUTBOX,
                    MT_COUNT };

    template <RequestType rt>
    bool ReqSpamCompatible(SpamType);

#define FOR_DICT_REQUEST_TYPE(it) for (size_t it = 0; it < DictionaryShingler::RT_COUNT; it++)
#define FOR_DICT_SPAM_TYPE(it) for (size_t it = 0; it < DictionaryShingler::ST_COUNT; it++)
#define FOR_DICT_MAIL_TYPE(it) for (size_t it = 0; it < DictionaryShingler::MT_COUNT; it++)

    const TString RequestTypeRepresentation[] = {
        "ABUSE",
        "PUT",
        "GET",
        "VIRUS",
        "FISHING",
        "FISHING_YANDEX",
        "HACKED",
    };

    const TString SpamTypeRepresentation[] = {
        "HAM",
        "SPAM",
        "COMPLHAM",
        "COMPLSPAM",
        "VIRUS",
        "FISHING",
        "FISHING_YANDEX",
        "HACKED",
        "EXPERT_COMPLHAM",
        "EXPERT_COMPLSPAM",
    };

    const TString SpamTypeDBIdrepresentation[] = {
        "ham",
        "spam",
        "compl_ham",
        "compl_spam",
        "virus",
        "fishing",
        "fishing_yan",
        "hacked",
        "expert_compl_ham",
        "expert_compl_spam",
    };

    const TString MailTypeRepresentation[] = {
        "DICTIONARY_INBOX",
        "DICTIONARY_OUTBOX",
    };


    inline TStringBuf Command(RequestType type) {
        switch(type) {
            case RT_PUT: return "dict_put"sv;
            case RT_ABUSE: return "dict_abuse"sv;
            case RT_GET: return "dict_get"sv;
            case RT_FISHING: return "dict_fishing"sv;
            case RT_VIRUS: return "dict_virus"sv;
            case RT_FISHING_YAN: return "dict_fishing_yan"sv;
            case RT_HACKED: return "dict_hacked"sv;
            default:
                ythrow TWithBackTrace<yexception>() << "unknown type " << RequestTypeRepresentation[type];
        }
    }
    inline TStringBuf LogCode(RequestType type) {
        switch(type) {
            case RT_PUT: return "DP"sv;
            case RT_ABUSE: return "DA"sv;
            case RT_GET: return "DG"sv;
            case RT_FISHING: return "DF"sv;
            case RT_VIRUS: return "DV"sv;
            case RT_FISHING_YAN: return "DFY"sv;
            case RT_HACKED: return "DH"sv;
            default:
                ythrow TWithBackTrace<yexception>() << "unknown type " << RequestTypeRepresentation[type];
        }
    }

    typedef int SpamTypeDBIdSizeMustBeEqualST_COUNT[(sizeof(SpamTypeDBIdrepresentation) / sizeof(SpamTypeDBIdrepresentation[0]) == ST_COUNT) ? 1 : -1];
    typedef int SpamTypeRepresentationSizeMustBeEqualST_COUNT[(sizeof(SpamTypeRepresentation) / sizeof(SpamTypeRepresentation[0]) == ST_COUNT) ? 1 : -1];
    typedef int RequestTypeRepresentationSizeMustBeEqualRT_COUNT[(sizeof(RequestTypeRepresentation) / sizeof(RequestTypeRepresentation[0]) == RT_COUNT) ? 1 : -1];
    typedef int MailTypeRepresentationSizeMustBeEqualMT_COUNT[(sizeof(MailTypeRepresentation) / sizeof(MailTypeRepresentation[0]) == MT_COUNT) ? 1 : -1];

    TString SpamTypeDBId(SpamType st);
    ;
    TString SpamTypeToStroka(SpamType value);
    TString MailTypeToStroka(MailType value);

    class Counters {
    public:
        static const size_t fieldsCount = ST_COUNT + 2;

        bool empty() const;
        void clear();

        template <SpamType st>
        ui32 get() const {
            return fields[st];
        };
        template <SpamType st>
        ui32& set() {
            return fields[st];
        };

        ui32 getFirstTime() const {
            return firstTime;
        }
        ui32& setFirstTime() {
            return firstTime;
        }
        ui32 getLastTime() const {
            return lastTime;
        }
        ui32& setLastTime() {
            return lastTime;
        }

        friend IOutputStream& operator<<(IOutputStream& stream, const Counters& c);
        TString ToLog() const;

        void merge(const Counters& newCounters);

        bool incrBySpamType(SpamType st, ui32 val);
        ui32 getBySpamType(SpamType st) const;
        void setBySpamType(SpamType st, const ui32& val);
        ui32& setBySpamType(SpamType st);

        Counters()
            : firstTime(0)
            , lastTime(0)
        {
            std::fill(fields.begin(), fields.end(), 0);
        }
        Counters(ui32 firstTime, ui32 lastTime, ui32 ham, ui32 spam, ui32 complHam, ui32 complSpam, ui32 virus, ui32 fishing, ui32 fishing_yan, ui32 hacked, ui32 expertComplHam, ui32 expertComplSpam)
            : firstTime(firstTime)
            , lastTime(lastTime)
        {
            fields[ST_HAM] = ham;
            fields[ST_SPAM] = spam;
            fields[ST_COMPLHAM] = complHam;
            fields[ST_COMPLSPAM] = complSpam;
            fields[ST_VIRUS] = virus;
            fields[ST_FISHING] = fishing;
            fields[ST_FISHING_YAN] = fishing_yan;
            fields[ST_HACKED] = hacked;
            fields[ST_EXPERT_COMPLHAM] = expertComplHam;
            fields[ST_EXPERT_COMPLSPAM] = expertComplSpam;
        }

    protected:
        ui32 firstTime;
        ui32 lastTime;
        std::array<ui32, ST_COUNT> fields;
    };

    class Stat {
    public:
        enum Time { TODAY = 0,
                    YESTERDAY,
                    HISTORY,
                    COUNT };
        template <Time>
        struct TimeToString {
            static const TString str;
        };

    private:
        std::array<Counters, COUNT> daily;
        ui64 flags;

    public:
        void clear() {
            daily[TODAY].clear();
            daily[YESTERDAY].clear();
            daily[HISTORY].clear();
            flags = 0;
        }

        template <Time t>
        const Counters& get() const {
            return daily[t];
        }
        template <Time t>
        Counters& set() {
            return daily[t];
        }
        template <Time t, DictionaryShingler::SpamType st>
        ui32 get() const {
            return daily[t].get<st>();
        }
        template <Time t, DictionaryShingler::SpamType st>
        ui32& set() {
            return daily[t].set<st>();
        }

        template <Time t>
        ui32 getFirstTime() const {
            return get<t>().getFirstTime();
        }
        template <Time t>
        ui32& setFirstTime() {
            return set<t>().setFirstTime();
        }
        template <Time t>
        ui32 getLastTime() const {
            return get<t>().getLastTime();
        }
        template <Time t>
        ui32& setLastTime() {
            return set<t>().setLastTime();
        }

        template <Time t>
        ui32 getHam() const {
            return get<t, ST_HAM>();
        }
        template <Time t>
        ui32& setHam() {
            return set<t, ST_HAM>();
        }
        template <Time t>
        ui32 getSpam() const {
            return get<t, ST_SPAM>();
        }
        template <Time t>
        ui32& setSpam() {
            return set<t, ST_SPAM>();
        }
        template <Time t>
        ui32 getComplHam() const {
            return get<t, ST_COMPLHAM>();
        }
        template <Time t>
        ui32& setComplHam() {
            return set<t, ST_COMPLHAM>();
        }
        template <Time t>
        ui32 getComplSpam() const {
            return get<t, ST_COMPLSPAM>();
        }
        template <Time t>
        ui32& setComplSpam() {
            return set<t, ST_COMPLSPAM>();
        }
        template <Time t>
        ui32 getVirus() const {
            return get<t, ST_VIRUS>();
        }
        template <Time t>
        ui32& setVirus() {
            return set<t, ST_VIRUS>();
        }
        template <Time t>
        ui32 getFishing() const {
            return get<t, ST_FISHING>();
        }
        template <Time t>
        ui32& setFishing() {
            return set<t, ST_FISHING>();
        }
        template <Time t>
        ui32 getFishingYandex() const {
            return get<t, ST_FISHING_YAN>();
        }
        template <Time t>
        ui32& setFishingYandex() {
            return set<t, ST_FISHING_YAN>();
        }
        template <Time t>
        ui32 getHacked() const {
            return get<t, ST_HACKED>();
        }
        template <Time t>
        ui32& setHacked() {
            return set<t, ST_HACKED>();
        }
        template <Time t>
        ui32 getExpertComplHam() const {
            return get<t, ST_EXPERT_COMPLHAM>();
        }
        template <Time t>
        ui32& setExpertComplHam() {
            return set<t, ST_EXPERT_COMPLHAM>();
        }
        template <Time t>
        ui32 getExpertComplSpam() const {
            return get<t, ST_EXPERT_COMPLSPAM>();
        }
        template <Time t>
        ui32& setExpertComplSpam() {
            return set<t, ST_EXPERT_COMPLSPAM>();
        }

        //    typedef int ItsTimeToThinkHereBecauseLanguagesCountGreaterThenBitsCountInUI64[(LANG_MAX > (sizeof(ui64)*8)) ? -1 : 1];
        bool isLanguageFits(ELanguage language) const {
            if (language == LANG_UNK)
                return getLanguageRaw() == 0;
            return getLanguageRaw() & (1ul << (language - 1));
        }
        ui64 getLanguageRaw() const {
            return flags;
        }
        ui64& setLanguageRaw() {
            return flags;
        }
        void setLanguage(ELanguage language) {
            if (language)
                flags |= (1ul << (language - 1));
        }

        TString ToLog() const {
            TStringStream res;

            res << "t=(" << daily[TODAY].ToLog() << "),"
                << "y=(" << daily[YESTERDAY].ToLog() << "),"
                << "h=(" << daily[HISTORY].ToLog() << "),"
                << "f=" << flags;

            return res.Str();
        }

        Stat()
            : flags(0)
        {
        }
        Stat(const Counters& today, const Counters& yesterday, const Counters& history, ui64 flags)
            : flags(flags)
        {
            daily[TODAY] = today;
            daily[YESTERDAY] = yesterday;
            daily[HISTORY] = history;
        }
    };

    class Request {
        Stat stat;
        ui64 shingle;
        TUtf16String word;
        MailType mailType;
        SpamType spamtype;

    public:
        ui64 const& GetShingle() const {
            return shingle;
        };
        ui64 GetHash() const;
        ui64& SetShingle() {
            return shingle;
        }

        MailType GetMailType() const {
            return mailType;
        };
        MailType& SetMailType() {
            return mailType;
        }

        const TUtf16String& sWord() const {
            return word;
        };
        SpamType GetSpamType() const {
            return spamtype;
        };
        bool IsValid() const {
            return (shingle > 0) && !word.empty();
        }

        friend IOutputStream& operator<<(IOutputStream& out, const Request& req);
        static bool ParseRequestFromString(const char* str, Request& result);

        Stat& SetCounters();
        void SetCounters(const Stat&);
        const Stat& GetCounters() const;

        Request();
        Request(const TUtf16String& word, ELanguage language, SpamType spamtype, MailType mailType = MT_INBOX);
        TString ToLogShindata() const;
        TString ToLog() const;

        static ui64 CalcShingle(const TUtf16String& word);
    };

} /* namespace DictionaryShingler */
