#include <array>
#include <util/generic/vector.h>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <util/datetime/base.h>
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include "freemail.h"
#include "ktime.h"

struct THashEmailID {
    template <bool yandex>
    size_t operator()(const NFreeMail::TEmailInfo::TEmailID<yandex>& value) const {
        return FnvHash<ui64>(value) ^ (yandex? 0: IntHash(static_cast<size_t>(1)));
    }
};

template<> struct THash<NFreeMail::TEmailInfo::UUID> : public THashEmailID {};
template<> struct THash<NFreeMail::TEmailInfo::External> : public THashEmailID {};

namespace NFreeMail {
    TRecepientInfo::TRecepientInfo(const TString& address)
        : TRecepientInfo(address, 0, 0)
    { }

    TRecepientInfo::TRecepientInfo(const TString& address, ui32 send, ui32 recv)
        : address(address), send(send), recieve(recv)
    { }

    THamSpamInfo::THamSpamInfo(ui32 ham, ui32 spam)
        : ham(ham), spam(spam)
    { }

    TInfo::TInfo(ui32 create, ui32 update, ui32 get, const THamSpamInfo& send, const THamSpamInfo& receive, const THamSpamInfo& complaint,
        const THamSpamInfo& uniq_compluser, ui32 uniq_geozone, ui32 active_days, ui32 max_recipients_count)
        : get_time(get), create_time(create), update_time(update), send(send), receive(receive), complaint(complaint)
        , unique_complaint_user(uniq_compluser), unique_geo_zone(uniq_geozone), active_days(active_days), max_recipients_count(max_recipients_count)
    { }

    template<typename T>
    static T GetElapsedTime(T begin, T end) {
        return (begin >= end) ? 0 : end - begin;
    }

    ui32 TInfo::GetCreateElapsedTime() const {
        return GetElapsedTime(create_time, get_time);
    }

    ui32 TInfo::GetUpdateElapsedTime() const {
        return GetElapsedTime(update_time, get_time);
    }

    TDayCounter::TDayCounter()
        : TDayCounter(0, 0)
    { }

    TDayCounter::TDayCounter(ui32 today, ui32 total)
        : today(today), total(total)
    { }

    ui32 TPDDInfo::GetElapsedTime() const {
        return static_cast<ui32>(::NFreeMail::GetElapsedTime(static_cast<ui64>(first_time), TInstant::Now().Seconds()) / 86400);
    }

    TBounceInfo::TBounceInfo()
        : TBounceInfo({}, {}, 0 , 0, 0)
    { }

    TBounceInfo::TBounceInfo(const TDayCounter& spam, const TDayCounter& unknown, ui32 last_type, ui32 last_date, ui32 get_date)
        : spam(spam), unknown(unknown), last_type(last_type), last_date(last_date), get_date(get_date)
    { }

    ui32 TBounceInfo::GetElapsedTime() const {
        return ::NFreeMail::GetElapsedTime(last_date, get_date);
    }

    TEmailInfo::TEmailInfo(const UUID& uuid) : m_info(uuid)
    { }

    TEmailInfo::TEmailInfo(const External& email) : m_info(email)
    { }

    TEmailInfo TEmailInfo::Create(const TString& uuid, const TString& external, bool isYandex) {
        return isYandex ? TEmailInfo(TEmailInfo::UUID(uuid)): TEmailInfo::External(external);
    }

    struct EmptyVisitor {
        template<typename T> bool operator()(const T& value) {
            return value.empty();
        }
    };

    bool TEmailInfo::Empty() const {
        EmptyVisitor visitor;
        return std::visit(visitor, m_info);
    }

    bool TEmailInfo::IsYandex() const {
        return std::holds_alternative<UUID>(m_info);
    }

    ui64 TEmailInfo::GetUUID() const {
        Y_ASSERT(IsYandex());
        return FromStringWithDefault<size_t>(std::get<UUID>(m_info), 0);
    }

    ui64 TEmailInfo::GetShingle() const {
        Y_ASSERT(IsYandex());
        return GetHash();
    }

    ui64 TEmailInfo::GetHash() const {
        return std::visit([](const auto& value) {
            using T = std::decay_t<decltype(value)>;
            return THash<T>{}(value);
        }, m_info);
    }
}

IOutputStream& operator<<(IOutputStream& stream, const NFreeMail::THamSpamInfo& value) {
    return stream << value.ham << ' ' << value.spam;
}

IOutputStream& operator<<(IOutputStream& stream, const NFreeMail::TDayCounter& value) {
    return stream << value.today << ' ' << value.total;
}
