/*
 * BBShClient.cpp
 *
 */

#include "BBShClient.h"

#include <mail/so/libs/syslog/so_log.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>

#include <library/cpp/http/fetch/exthttpcodes.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <util/charset/utf8.h>
#include <util/charset/wide.h>
#include <util/generic/scope.h>
#include <util/generic/variant.h>
#include <util/string/builder.h>
#include <util/string/strip.h>

namespace NFuncClient {
    CBB::TRequest CBB::MakeRequest(const TLogin& login, const NBlackbox2::TOptions& options, const TString& sid) {
        TStringBuf view(login);
        view = StripString(view).Before(' ');
        view.SkipPrefix("<");
        view.ChopSuffix(">");
        // Blackbox encodes whole address as punycode, because it could contain domain name
        TUtf32String cleanText;
        const unsigned char* start = (const unsigned char*) view.Data();
        const unsigned char* end = (const unsigned char*) view.Data() + view.Size();
        while (start < end) {
            wchar32 c;
            if (ReadUTF8CharAndAdvance(c, start, end) != RECODE_OK) {
                cleanText += BROKEN_RUNE;
                ++start;
            } else {
                cleanText += c;
            }
        }
        return TRequest{InfoRequest(NBlackbox2::TLoginSid(WideToUTF8(cleanText), sid), "127.0.0.1", options)};
    }

    CBB::TRequest CBB::MakeRequest(const TUid& uid, const NBlackbox2::TOptions& options) {
        return TRequest{InfoRequest(uid, "127.0.0.1", options)};
    }

    struct TRequestMaker{
        const NBlackbox2::TOptions& Options;
        explicit TRequestMaker(const NBlackbox2::TOptions& options) noexcept : Options(options) {}

        CBB::TRequest operator()(const TLogin& login) {
            return CBB::MakeRequest(login, Options);
        }

        CBB::TRequest operator()(const TUid& uid) {
            return CBB::MakeRequest(uid, Options);
        }
    };

    CBB::TRequest CBB::MakeRequest(const TBbKey& key, const NBlackbox2::TOptions& options) {
        return std::visit(TRequestMaker(options), key);
    }

    static TMaybe<TString> GetByQuery(const NBlackbox2::TResponse& response, const TString& query) {
        TString val;
        if(response.GetImpl()->GetIfExists(query, val))
            return std::move(val);

        return Nothing();
    }

    TMaybe<TString> CBB::GetSuidFromBBResponse(const NBlackbox2::TResponse& response) {
        return GetByQuery(response, "dbfield[@id='subscription.suid.2']");
    }

    TMaybe<TString> CBB::GetCountryFromBBResponse(const NBlackbox2::TResponse& response) {
        return GetByQuery(response, "dbfield[@id='userinfo.country.uid']");
    }

    TMaybe<TInstant> CBB::GetRegDateFromBBResponse(const NBlackbox2::TResponse& response) {
        TInstant regDate;
        if(auto s = GetByQuery(response, "dbfield[@id='userinfo.reg_date.uid']"); s && TInstant::TryParseIso8601(*s, regDate))
            return regDate;
        return Nothing();
    }

    TMaybe<TString> CBB::GetPhoneFromBBResponse(const NBlackbox2::TResponse& response) {
        return GetByQuery(response, "//attributes//attribute[@type='36']");
    }

    THashSet<TString> CBB::GetPddAliasesFromBBResponse(const NBlackbox2::TResponse& response) {
        auto parts = response.GetImpl()->GetParts("//aliases//alias[@type='8']");

        THashSet<TString> aliases(parts.Size());
        for(int i = 0; i < parts.Size(); i++) {
            aliases.emplace_noresize(parts[i].asString());
        }
        return aliases;
    }

    THolder<NBlackbox2::TResponse> CBB::GetUserInfo(const TRequest& request, const TLog& logger) {
        NCurl::TSimpleArtifacts artifacts;

        if(const auto error = Perform(artifacts, TRequestClient::TRequest{}.SetRequest(TStringBuilder{} << "blackbox?" << request))) {
            logger << (TLOG_ERR) << *error << request;
            return {};
        }

        const auto& responseStr = artifacts.body.Str();

        try {
            if(auto response = NBlackbox2::InfoResponse(responseStr)) {
                return response;
            }
            PushError();
            logger << (TLOG_ERR) << "Empty response for " << request << " : " << responseStr;
            return {};
        } catch (...) {
            PushError();
            logger << (TLOG_ERR) << CurrentExceptionMessageWithBt() << " response:" << responseStr << ' ' << " for request " << request;
            return {};
        }
    }
} /* namespace NFuncClient */
