/*
 * ComplShClient.cpp
 *
 *  Created on: 16 сент. 2016 г.
 *      Author: luckybug
 */

#include <mail/so/spamstop/tools/so-clients/general_shingler_get.h>
#include <mail/so/spamstop/tools/so-clients/general_shingler_put.h>
#include <mail/so/spamstop/tools/so-clients/kshingle.h>
#include <util/generic/scope.h>
#include <util/string/builder.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include "ComplShClient.h"

namespace NFuncClient {

    THashMap<std::pair<ui64, int>, TUserWeightsPair> TUserWeightsRequesterNg::GetUserWeights(const TLog& logger) const {
        NCurl::TSimpleArtifacts artifacts;
        if (auto error = TRequestClient::Perform(artifacts, TRequestClient::TRequest{}
                .SetRequest("api/async/so/get-user-weights"))) {
            logger << (TLOG_ERR) << *error;
            return {};
        }

        NJson::TJsonValue jsonValue;
        if(!NJson::ReadJsonTree(artifacts.body.Str(), &jsonValue, false)) {
            logger << (TLOG_ERR) << "cannot parse json from " << artifacts.body.Str();
            PushError();
            return {};
        }

        try {
            THashMap<std::pair<ui64, int>, TUserWeightsPair> weights;
            for(const NJson::TJsonValue & val : jsonValue.GetArraySafe()) {
                const ui64 shingle = static_cast<ui64>(val["shingle"].GetIntegerRobust());
                const int type = static_cast<int>(val["type"].GetUIntegerRobust());

                const auto creationTime = TInstant::Seconds(val["timeset"].GetUIntegerRobust());
                const auto lastUp = TInstant::Seconds(val["lastset"].GetUIntegerRobust());
                const auto duration = TDuration::Hours(val["duration"].GetUIntegerRobust());
                const auto shouldBeMalic = val["should_be_malic"].GetBooleanRobust();
                const auto weight = val["weight"].GetDoubleRobust();
                auto user = val["user"].GetStringRobust();

                TUserWeightsPair& target = weights[std::make_pair(shingle, type)];

                (val["so_type"].GetStringSafe() == "in" ? target.In : target.Out) =
                        TUserWeights(creationTime, lastUp, duration, shouldBeMalic, weight, std::move(user));
            }

            return weights;
        } catch (...) {
            logger << (TLOG_ERR) << "failed parsing json " << artifacts.body.Str() << ':' << CurrentExceptionMessageWithBt();
            PushError();
            return {};
        }
    }

    bool TCompl::Get(TCntComplShingleRequestList &shlist, const TLog& logger) const {
        THashMap<std::pair<ui64, int>, TCntComplShingleRequest *> links;

        NGeneralShingler::TGetRequest request;
        {
            NJson::TJsonValue::TArray fields;

            fields.emplace_back();
            auto *shinglerArr = &fields.back()["shingle"].SetType(NJson::JSON_ARRAY).GetArraySafe();
            auto *typeArr = &fields.back()["type"].SetType(NJson::JSON_ARRAY).GetArraySafe();

            for (TCntComplShingleRequest &shingle : shlist) {
                shinglerArr->emplace_back(shingle.Shingle());
                typeArr->emplace_back(shingle.Type());

                links.emplace(std::make_pair(shingle.Shingle(), shingle.Type()), &shingle);
            }

            request.AddMessage(TVector<TString>{"today_abuses", "history_abuses"}, std::move(fields));
        }

        if (!Getter.Perform(request, logger))
            return false;

        for (const auto &scheme : request.GetAnswer().GetArraySafe()) {
            const auto &schemeName = scheme["scheme"].GetStringSafe();

            for (const auto &fields : scheme["find"].GetArraySafe()) {
                const ui64 shingle = fields["shingle"].GetUIntegerSafe();
                const int type = static_cast<int>(fields["type"].GetUIntegerSafe());

                auto reqIt = links.find(std::make_pair(shingle, type));

                if (links.cend() == reqIt)
                    continue;

                TCntComplShingleRequest &req = *reqIt->second;

                if (schemeName == "today_abuses") {
                    req.SetTodayCounters(
                            fields["firsttime"].GetUIntegerRobust(),
                            fields["lasttime"].GetUIntegerRobust(),
                            fields["ham"].GetUIntegerRobust(),
                            fields["spam"].GetUIntegerRobust(),
                            fields["malic"].GetUIntegerRobust(),
                            fields["pers_ham"].GetUIntegerRobust(),
                            fields["pers_spam"].GetUIntegerRobust(),
                            fields["complaint_ham"].GetUIntegerRobust(),
                            fields["complaint_spam"].GetUIntegerRobust(),
                            fields["virus_count"].GetUIntegerRobust(),
                            fields["uniq_sd_spam"].GetUIntegerRobust(),
                            fields["uniq_sd_ham"].GetUIntegerRobust(),
                            fields["expert_spam"].GetUIntegerRobust(),
                            fields["expert_ham"].GetUIntegerRobust());
                } else if (schemeName == "history_abuses") {
                    req.SetHistoryCounters(
                            fields["firsttime"].GetUIntegerRobust(),
                            fields["lasttime"].GetUIntegerRobust(),
                            fields["ham"].GetUIntegerRobust(),
                            fields["spam"].GetUIntegerRobust(),
                            fields["malic"].GetUIntegerRobust(),
                            fields["pers_ham"].GetUIntegerRobust(),
                            fields["pers_spam"].GetUIntegerRobust(),
                            fields["complaint_ham"].GetUIntegerRobust(),
                            fields["complaint_spam"].GetUIntegerRobust(),
                            fields["virus_count"].GetUIntegerRobust(),
                            fields["uniq_sd_spam"].GetUIntegerRobust(),
                            fields["uniq_sd_ham"].GetUIntegerRobust(),
                            fields["expert_spam"].GetUIntegerRobust(),
                            fields["expert_ham"].GetUIntegerRobust(),
                            fields["day_count_with_complaint"].GetUIntegerRobust(),
                            fields["day_count_with_virus"].GetUIntegerRobust());
                }
            }
        }

        return true;
    }

    NGeneralShingler::TPutRequest TCompl::MakePutRequest(const TCntComplShingleRequestList& shlist, const TString& spam) {
        NGeneralShingler::TPutRequest request;

        NJson::TJsonValue::TArray fields;
        NJson::TJsonValue::TArray daysWithAbusesFields;
        for (const TCntComplShingleRequest &shingle : shlist) {
            auto &newReqFields = fields.emplace_back();
            auto &newReqDAFields = daysWithAbusesFields.emplace_back();

            newReqFields["shingle"] = newReqDAFields["shingle"] = shingle.Shingle();
            newReqFields["type"] = newReqDAFields["type"] = shingle.Type();

            if (shingle.GetUniqSdSpam())
                newReqFields["uniq_sd_spam"] = shingle.GetUniqSdSpam();
            if (shingle.GetUniqSdHam())
                newReqFields["uniq_sd_ham"] = shingle.GetUniqSdHam();

            if (spam == SPAMTYPETEXT_HAM1) {
                newReqFields["complaint_ham"] = 1;
            } else if (spam == SPAMTYPETEXT_SPAM1 || spam == SPAMTYPETEXT_UNSUBSCRIBE1) {
                newReqFields["complaint_spam"] = 1;
            }

            newReqDAFields["day_count_with_complaint"] = 1;
        }

        request.AddMessage(TVector<TString>{"today_abuses", "history_abuses"}, std::move(fields));

        request.AddMessage("history_abuses_days", std::move(daysWithAbusesFields));

        return request;
    }

    bool TCompl::Update(const NGeneralShingler::TPutRequest& request, const TLog& logger) const {
        return Updater.Perform(request, logger);
    }
} /* namespace NFuncClient */
