/*
 * FreemailShClient.cpp
 *
 *  Created on: 21 сент. 2016 г.
 *      Author: luckybug
 */

#include "FreemailShClient.h"
#include <mail/so/spamstop/tools/so-clients/general_shingler_get.h>
#include <mail/so/spamstop/tools/so-clients/general_shingler_put.h>
#include <library/cpp/xml/document/xml-document.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>

namespace NFuncClient {
    static const size_t geo_period = 30;
    static const size_t day_period = 90;
    static const size_t day_slice = 15;
    static const size_t slice_period = day_period / day_slice;

    namespace detail {
        static NJson::TJsonValue ToMap(const NJson::TJsonValue& data) {
            NJson::TJsonValue result(NJson::JSON_MAP);
            for (const auto& item : data.GetArraySafe()) {
                const auto value = item["find"];
                if (value.IsArray() && !value.GetArraySafe().empty())
                    result[item["scheme"].GetString()] = value;
            }
            return result;
        }

        static NJson::TJsonValue::TArray ToArray(const NJson::TJsonValue& value) {
            return NJson::TJsonValue::TArray(1, value);
        }

        static void FillCurrent(const NJson::TJsonValue& data, ui32& time) {
            time = data.GetArraySafe().begin()->GetMapSafe().begin()->second.GetUIntegerRobust();
        }

        using TFreemailPutRequest = NGeneralShingler::TPutRequest;

        class TFreemailGetRequest : public NGeneralShingler::TGetRequest {
        public:
            TFreemailGetRequest(const NFreeMail::TEmailInfo& key, TMaybe<NFreeMail::TInfo>& info, TMaybe<NFreeMail::TBounceInfo>& bounce)
                : m_info(info)
                , m_bounce(bounce) {
                NJson::TJsonValue::TArray local(1);
                local.front()["uuid"] = key.GetShingle();
                auto& date = local.front()["date"];

                date = slice_period;
                AddMessage("counters_get", local);

                date = geo_period;
                AddMessage("geo_get", local);

                date = day_period;
                for (const auto& name : {"time_get", "complaint_get"})
                    AddMessage(name, local);

                local.front()["uuid"] = key.GetUUID();
                AddMessage("bounce_get", local);

                AddMessage("current", NJson::TJsonValue::TArray());
            }

        private:
            static void FillTime(const NJson::TJsonValue& data, NFreeMail::TInfo& info) {
                if (data.IsArray()) {
                    Y_ASSERT(!data.GetArraySafe().empty());
                    const auto& value = data.GetArraySafe()[0];
                    info.create_time = value["create_date"].GetIntegerRobust();
                    info.update_time = value["date"].GetIntegerRobust();
                    info.active_days = value["active_days"].GetIntegerRobust();
                }
            }

            static void FillCounters(const NJson::TJsonValue& data, NFreeMail::TInfo& info) {
                for (const auto& item : data.GetArray()) {
                    info.send.ham += item["send_ham"].GetIntegerRobust();
                    info.send.spam += item["send_spam"].GetIntegerRobust();

                    info.receive.ham += item["receive_ham"].GetIntegerRobust();
                    info.receive.spam += item["receive_spam"].GetIntegerRobust();

                    info.complaint.ham += item["complaint_ham"].GetIntegerRobust();
                    info.complaint.spam += item["complaint_spam"].GetIntegerRobust();

                    const ui32 max_rcpt = std::max<ui32>(item["recepients_count"].GetIntegerRobust(), item["recepients_max_count"].GetIntegerRobust());
                    info.max_recipients_count = std::max<ui32>(info.max_recipients_count, max_rcpt);
                }
            }

            static void FillComplaint(const NJson::TJsonValue& data, NFreeMail::TInfo& info) {
                const auto& array = data.GetArray();
                info.unique_complaint_user.spam = std::count_if(array.begin(), array.end(), [](const auto& v) {
                    return v["spam"].GetBooleanRobust();
                });
                info.unique_complaint_user.ham = array.size() - info.unique_complaint_user.spam;
            }

            static void FillGeo(const NJson::TJsonValue& data, NFreeMail::TInfo& info) {
                info.unique_geo_zone = data.GetArray().size();
            }

            void FillInfo(const NJson::TJsonValue& data) {
                if (data.Has("time_get")) {
                    m_info.ConstructInPlace();
                    FillGeo(data["geo_get"], *m_info);
                    FillTime(data["time_get"], *m_info);
                    FillCurrent(data["current"], m_info->get_time);
                    FillCounters(data["counters_get"], *m_info);
                    FillComplaint(data["complaint_get"], *m_info);
                }
            };

            void FillBounce(const NJson::TJsonValue& data) {
                const auto& array = data["bounce_get"];
                if (array.IsArray()) {
                    Y_ASSERT(!array.GetArraySafe().empty());
                    const auto& value = array.GetArraySafe()[0];

                    m_bounce.ConstructInPlace();
                    m_bounce->spam.today = value["today_spam"].GetIntegerRobust();
                    m_bounce->spam.total = value["total_spam"].GetIntegerRobust();

                    m_bounce->unknown.today = value["today_unknown"].GetIntegerRobust();
                    m_bounce->unknown.total = value["total_unknown"].GetIntegerRobust();

                    m_bounce->last_type = value["last_type"].GetIntegerRobust();
                    m_bounce->last_date = value["date"].GetIntegerRobust();

                    FillCurrent(data["current"], m_bounce->get_date);
                }
            }

            void ParseAnswer(const NJson::TJsonValue& value) final {
                auto data = ToMap(value);
                FillInfo(data);
                FillBounce(data);
            }

            TMaybe<NFreeMail::TInfo>& m_info;
            TMaybe<NFreeMail::TBounceInfo>& m_bounce;
        };

        class TPDDGetRequest : public NGeneralShingler::TGetRequest {
        public:
            TPDDGetRequest(const NJson::TJsonValue& key, TMaybe<NFreeMail::TPDDInfo>& pdd)
                : m_pdd(pdd) {
                NJson::TJsonValue value;
                value["request"] = "so/domaininfo/";
                value["param"] = key;
                AddMessage("pdd", ToArray(value));

                auto array = ToArray(key);
                AddMessage("domain", array);

                array.emplace_back(NJson::JSON_MAP)["date"] = day_period;
                AddMessage("domain_all", array);
            }

        private:
            TString getLogCode() const override {
                return "PDDGET";
            };

            static void FillToday(const NJson::TJsonValue& data, NFreeMail::TPDDInfo& info) {
                if (data.IsArray()) {
                    Y_ASSERT(!data.GetArraySafe().empty());
                    const auto& value = data.GetArraySafe()[0];
                    info.bounce.today = value["count"].GetIntegerRobust();
                }
            }

            static void FillHost(const NJson::TJsonValue& data, NFreeMail::TPDDInfo& info) {
                if (data.IsArray()) {
                    Y_ASSERT(!data.GetArraySafe().empty());
                    const auto& value = data.GetArraySafe()[0];
                    info.host = value["domain"].GetString();
                }
            }

            static void FillTotal(const NJson::TJsonValue& data, NFreeMail::TPDDInfo& info) {
                for (const auto& item : data.GetArray()) {
                    info.bounce.total += item["count"].GetIntegerRobust();
                }
            }

            void FillDomain(const NJson::TJsonValue& data) {
                if (m_pdd.Defined() && data.Has("domain_all")) {
                    FillToday(data["domain"], *m_pdd);
                    FillTotal(data["domain_all"], *m_pdd);
                    FillHost(data["domain_all"], *m_pdd);
                }
            }

            template <typename T, typename S = T>
            static T Get(const NXml::TConstNode& root, const TString& name) {
                return static_cast<T>(root.Node(name, true).Value<S>(Default<T>()));
            }

            static void FillPDDFromXml(const NXml::TConstNode& root, NFreeMail::TPDDInfo& info) {
                info.karma = Get<ui32>(root, "karma");
                info.first_time = Get<ui32, double>(root, "firsttime");
                info.mailbox_count = Get<ui32>(root, "mailboxcnt");
                info.admin_uid = Get<TString>(root, "admlogin");
                info.ip = TKIPv6(Get<TString>(root, "ip"));
                info.org_id = Get<TString>(root, "org_id");

                for (const auto& node : root.Nodes("domains/domain"))
                    info.domains.emplace_back(node.Value<TString>());
            }

            void FillPDDFromXml(const TString& xmldata) {
                const NXml::TDocument xml(xmldata, NXml::TDocument::String);

                const auto root = xml.Root();
                if (root.Name() == "pddinfo" && root.Node("error", true).IsNull()) {
                    m_pdd.ConstructInPlace();
                    FillPDDFromXml(root, *m_pdd);
                }
            }

            void FillPDD(const NJson::TJsonValue& data) {
                if (data.IsArray()) {
                    Y_ASSERT(!data.GetArraySafe().empty());
                    const auto& value = data.GetArraySafe()[0];

                    if (value["code"].GetIntegerRobust() == 200) {
                        FillPDDFromXml(value["data"].GetStringRobust());
                    }
                }
            }

            void ParseAnswer(const NJson::TJsonValue& value) final {
                auto data = ToMap(value);
                FillPDD(data["pdd"]);
                FillDomain(data);
            }

            TMaybe<NFreeMail::TPDDInfo>& m_pdd;
        };

        static bool IsSpam(NFreeMail::ESpamType type) {
            return type == NFreeMail::MALIC || type == NFreeMail::SPAM;
        }

        static NJson::TJsonValue GetRecepientsMaxMap() {
            NJson::TJsonValue value(NJson::JSON_MAP);
            value["f1"] = "recepients_count";
            value["f2"] = "recepients_max_count";
            return value;
        }

        static const auto recepientsMaxMap = GetRecepientsMaxMap();
        static void AddInfoToRequest(TFreemailPutRequest& request, const ui64& uuid, const TString& geo, bool spam, size_t recepients) {
            NJson::TJsonValue base;
            base["uuid"] = uuid;

            {
                NJson::TJsonValue value(base);
                request.AddMessage("time", ToArray(value));

                value[spam ? "send_spam" : "send_ham"] = recepients;
                value["recepients_count"] = recepients;
                request.AddMessage("counters", ToArray(value));
            }

            {
                NJson::TJsonValue value(base);
                value["active_days"] = 1;
                request.AddMessage("active", ToArray(value));
            }

            if (!geo.empty()) {
                NJson::TJsonValue value(base);
                value["geo"] = to_upper(geo);
                request.AddMessage("geo", ToArray(value));
            }

            {
                NJson::TJsonValue value(base);
                value["recepients_count"] = 0;
                value["recepients_max_count"] = recepientsMaxMap;
                request.AddMessage("recepients_max", ToArray(value));
            }
        }

        static void AddRecepientsInfoToRequest(TFreemailPutRequest& request, bool spam, const TVector<NFreeMail::TEmailInfo>& recepients) {
            const TString key(spam ? "receive_spam" : "receive_ham");

            NJson::TJsonValue::TArray messages;
            for (const auto& recepient : recepients) {
                if (recepient.IsYandex() && !recepient.Empty()) {
                    auto& message = messages.emplace_back();
                    message["uuid"] = recepient.GetShingle();
                    message[key] = 1;
                }
            }

            if (!messages.empty())
                request.AddMessage("counters", messages);
        }
    } // namespace detail

    bool TFreeMail::Put(const NFreeMail::TEmailInfo& key, const TString& geo, NFreeMail::ESpamType type, const TVector<NFreeMail::TEmailInfo>& recepients, const TLog& logger) {
        if (type == NFreeMail::UNKNOWN || key.Empty())
            return false;

        const bool spam = detail::IsSpam(type);
        detail::TFreemailPutRequest request;

        if (key.IsYandex()) {
            detail::AddInfoToRequest(request, key.GetShingle(), geo, spam, recepients.size());
        }

        detail::AddRecepientsInfoToRequest(request, spam, recepients);
        return Updater.Perform(request, logger);
    }

    bool TFreeMail::Complaint(const NFreeMail::TEmailInfo& sender, const NFreeMail::TEmailInfo& complainant, NFreeMail::ESpamType type, const TLog& logger) {
        if (type == NFreeMail::UNKNOWN || sender.Empty())
            return false;

        if (!sender.IsYandex())
            return true;

        NJson::TJsonValue value;
        value["uuid"] = sender.GetShingle();
        const bool spam = detail::IsSpam(type);

        detail::TFreemailPutRequest request;
        request.AddMessage("time", detail::ToArray(value));

        NJson::TJsonValue counters = value;
        counters[spam ? "complaint_spam" : "complaint_ham"] = 1;
        request.AddMessage("counters", detail::ToArray(counters));

        value["spam"] = spam;
        value["hash"] = complainant.GetHash();
        request.AddMessage("complaint", detail::ToArray(value));

        return Updater.Perform(request, logger);
    }

    bool TFreeMail::Get(const NFreeMail::TEmailInfo& key, TMaybe<NFreeMail::TInfo>& info, TMaybe<NFreeMail::TBounceInfo>& bounce, const TLog& logger) {
        if (!key.IsYandex() || key.Empty())
            return false;

        detail::TFreemailGetRequest request(key, info, bounce);
        return Getter.Perform(request, logger);
    }

    bool TFreeMail::Get(const TString& domain, TMaybe<NFreeMail::TPDDInfo>& pdd, const TLog& logger) {
        if (domain.empty())
            return false;

        NJson::TJsonValue value;
        value["domain"] = domain;

        detail::TPDDGetRequest request(value, pdd);
        return Getter.Perform(request, logger);
    }
} /* namespace NFuncClient */
