/*
 * BBShClient.cpp
 *
 */

#include <library/cpp/json/json_reader.h>
#include <library/cpp/http/fetch/exthttpcodes.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include "RblClient.h"
#include <mail/so/spamstop/tools/so-clients/SperrorHolder.h>
#include <util/string/builder.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <util/generic/scope.h>
#include <mail/so/libs/syslog/so_log.h>

namespace NFuncClient {

    TRbl::TResponse::TResponse(NJson::TJsonValue js)
        : Original(std::move(js)) {
        {
            if(auto geobase = Original.GetValueByPath("infos.geobase")) {
                if(auto regionInfo = geobase->GetValueByPath("region_info")) {
                    IsoName = (*regionInfo)["iso_name"].GetStringSafe();
                    City = (*regionInfo)["city"].GetStringSafe();
                }

                if(auto ipTraits = geobase->GetValueByPath("ip_traits")) {
                    IsTorFlag = (*ipTraits)["is_tor"].GetBooleanSafe();

                    if (const auto& asnList = (*ipTraits)["asn_list"].GetArraySafe()) {
                        lastAsnItem = asnList.back().GetStringSafe();
                    }
                }
            }
        }

        if(auto checks = Original.GetValueByPath("checks")) {
            if(auto it = checks->GetValueByPath("combined-bl"))
                InCombinedBlackList = it->GetUIntegerRobust();
            if(auto it = checks->GetValueByPath("fast-reject"))
                InFastReject = it->GetUIntegerRobust();
        }
    }

    const NJson::TJsonValue& TRbl::TResponse::GetOriginal() const {
        return Original;
    }

    const TString& TRbl::TResponse::GetIsoName() const {
        return IsoName;
    }

    const TString& TRbl::TResponse::GetCity() const {
        return City;
    }

    bool TRbl::TResponse::IsTor() const {
        return IsTorFlag;
    }

    const TMaybe<TString>& TRbl::TResponse::GetLastAsnItem() const {
        return lastAsnItem;
    }

    const TRbl::TQueryFields TRbl::LISTS_ONLY(LISTS);
    const TRbl::TQueryFields TRbl::GEO_ONLY(GEO);
    const TRbl::TQueryFields TRbl::FAST_REJECT_ONLY(FAST_REJECT);

    std::variant<TRbl::TResponse, NCurl::TError> TRbl::Perform(TString ip,
                                                           TString service,
                                                           const TQueryFields& fields,
                                                           bool asValues) const {
        NCurl::TSimpleArtifacts artifacts;
        {
            TCgiParameters parameters;

            parameters.InsertUnescaped("ip", std::move(ip));

            if(fields.Test(GEO)) {
                parameters.InsertUnescaped("info", "geobase");
            }

            if(fields.Test(LISTS)) {
                parameters.InsertUnescaped("check", "combined-bl");
            }

            if(fields.Test(FAST_REJECT)) {
                parameters.InsertUnescaped("check", "fast-reject");
            }

            parameters.InsertUnescaped("service", std::move(service));
            if(asValues)
                parameters.InsertUnescaped("get-values", "1");

            if (auto error = TRequestClient::Perform(artifacts, TRequestClient::TRequest{}
                                                                    .SetRequest(TStringBuilder{} << "check?" << parameters.QuotedPrint()))) {
                return std::move(*error);
            }
        }

        return TRbl::TResponse(NJson::ReadJsonTree(&artifacts.body, true));
    }

    std::variant<THashMap<TString, TRbl::TResponse>, NCurl::TError> TRbl::Perform(const TVector<TString>& ips, TString service,
                                                                              const TQueryFields& fields,
                                                                              bool asValues) const {
        NCurl::TSimpleArtifacts artifacts;
        {
            TCgiParameters parameters;

            for (const auto& ip : ips) {
                parameters.InsertUnescaped("ip", ip);
            }

            if(fields.Test(GEO)) {
                parameters.InsertUnescaped("info", "geobase");
            }

            if(fields.Test(LISTS)) {
                parameters.InsertUnescaped("check", "combined-bl");
            }

            if(fields.Test(FAST_REJECT)) {
                parameters.InsertUnescaped("check", "fast-reject");
            }

            parameters.InsertUnescaped("service", std::move(service));

            if(asValues)
                parameters.InsertUnescaped("get-values", "1");

            if (auto error = TRequestClient::Perform(artifacts, TRequestClient::TRequest{}
                                                                    .SetRequest(TStringBuilder{} << "check-all?" << parameters.QuotedPrint()))) {
                return std::move(*error);
            }
        }

        THashMap<TString, TRbl::TResponse> responses;
        const auto jsAnswer = NJson::ReadJsonTree(&artifacts.body, true);
        for (const auto& [ip, js] : jsAnswer.GetMapSafe()) {
            responses.emplace(ip, js);
        }
        return responses;
    }
} /* namespace NFuncClient */
