#pragma once

#include <util/generic/variant.h>
#include <library/cpp/enumbitset/enumbitset.h>
#include "single_request_client.h"

namespace NFuncClient {
    class TRbl : TRequestClient {
    public:
        class TResponse {
        public:
            explicit TResponse(NJson::TJsonValue js);

            const NJson::TJsonValue& GetOriginal() const;

            const TString& GetIsoName() const;

            const TString& GetCity() const;

            bool IsTor() const;

            size_t IsInCombinedBlackList() const {
                return InCombinedBlackList;
            }

            size_t IsInFastReject() const {
                return InFastReject;
            }

            const TMaybe<TString>& GetLastAsnItem() const;
        private:
            NJson::TJsonValue Original;
            TString IsoName;
            TString City;
            bool IsTorFlag{};
            TMaybe<TString> lastAsnItem;
            size_t InCombinedBlackList{};
            size_t InFastReject{};
        };

        enum EQueryFields {
            GEO,
            LISTS,
            FAST_REJECT,
        };
        using TQueryFields = TEnumBitSet<EQueryFields, static_cast<int>(GEO), static_cast<int>(FAST_REJECT) + 1>;
        static const TQueryFields LISTS_ONLY;
        static const TQueryFields GEO_ONLY;
        static const TQueryFields FAST_REJECT_ONLY;

        using TRequestClient::TRequestClient;
        [[nodiscard]] std::variant<TResponse, NCurl::TError> Perform(TString ip, TString service, const TQueryFields& fields, bool asValues) const;
        [[nodiscard]] std::variant<THashMap<TString, TResponse>, NCurl::TError> Perform(const TVector<TString>& ips, TString service, const TQueryFields& fields, bool asValues) const;
    };
} /* namespace NFuncClient */

template<class TResponseHandler, class TErrorHandler>
void Visit(std::variant<NFuncClient::TRbl::TResponse, NCurl::TError>& responseOrError,
           TResponseHandler && responseHandler,
           TErrorHandler && errorHandler) {
    struct TLocalVsisitor {
        void operator()(NFuncClient::TRbl::TResponse& response) {
            ResponseHandler(std::move(response));
        }
        void operator()(NCurl::TError& error) {
            ErrorHandler(std::move(error));
        }

        TLocalVsisitor(TResponseHandler && responseHandler, TErrorHandler && errorHandler) noexcept
        : ResponseHandler(std::forward<TResponseHandler>(responseHandler))
        , ErrorHandler(std::forward<TErrorHandler>(errorHandler))
        {}

        TResponseHandler ResponseHandler;
        TErrorHandler ErrorHandler;
    };

    std::visit(TLocalVsisitor(std::forward<TResponseHandler>(responseHandler), std::forward<TErrorHandler>(errorHandler)), responseOrError);
};

template<class TResponsesHandler, class TErrorHandler>
void Visit(std::variant<THashMap<TString, NFuncClient::TRbl::TResponse>, NCurl::TError>& responsesOrError,
           TResponsesHandler && responsesHandler,
           TErrorHandler && errorHandler) {

    struct TLocalVsisitor {
        void operator()(THashMap<TString, NFuncClient::TRbl::TResponse>& responses) {
            ResponsesHandler(std::move(responses));
        }
        void operator()(NCurl::TError& error) {
            ErrorHandler(std::move(error));
        }

        TLocalVsisitor(TResponsesHandler && responsesHandler, TErrorHandler && errorHandler) noexcept
                : ResponsesHandler(std::forward<TResponsesHandler>(responsesHandler))
                , ErrorHandler(std::forward<TErrorHandler>(errorHandler))
        {}

        TResponsesHandler ResponsesHandler;
        TErrorHandler ErrorHandler;
    };

    std::visit(TLocalVsisitor(std::forward<TResponsesHandler>(responsesHandler), std::forward<TErrorHandler>(errorHandler)), responsesOrError);
};

template <>
struct hash<NFuncClient::TRbl::TQueryFields> {
    inline size_t operator()(const NFuncClient::TRbl::TQueryFields& fields) const noexcept {
        return fields.GetHash();
    }
};

