
#include <util/string/builder.h>
#include <library/cpp/json/writer/json.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include <util/generic/scope.h>
#include <library/cpp/json/json_reader.h>
#include <util/string/strip.h>
#include <mail/so/spamstop/tools/text2shingles/lib/text2shingles.h>
#include <mail/so/libs/syslog/so_log.h>
#include "SherlockClient.h"

namespace NFuncClient {
    const TString& TSherlock::TRequest::GetHtml() const {
        return Html;
    }

    const TString& TSherlock::TRequest::GetQueueId() const {
        return QueueId;
    }

    const TString& TSherlock::TRequest::GetFrom() const {
        return From;
    }

    const TString& TSherlock::TRequest::GetSubject() const {
        return Subject;
    }

    const TVector<TString>& TSherlock::TRequest::GetUids() const {
        return Uids;
    }

    const TString& TSherlock::TRequest::GetService() const {
        return Service;
    }

    TSherlock::TRequest::TRequest(
        TString html,
        TString queueId,
        TString from,
        TString subject,
        TString service,
        TVector<TString> uids) noexcept
        : Html(std::move(html))
        , QueueId(std::move(queueId))
        , From(std::move(from))
        , Subject(std::move(subject))
        , Service(std::move(service))
        , Uids(std::move(uids))
        {

        TStringBuf view(QueueId);
        view.SkipPrefix("<");
        view.ChopSuffix(">");
        QueueId = TString{view};
    }

    IOutputStream& operator<<(IOutputStream& stream, const TSherlock::TResponse::THashPrinter& printer) {
        NJsonWriter::TBuf buf(NJsonWriter::HEM_DONT_ESCAPE_HTML, &stream);

        buf.BeginList();
        for (const auto& hashes : printer.Master.GetHashes()) {
            buf.BeginList();
            for (const auto& hash : hashes) {
                buf.WriteString(hash);
            }
            buf.EndList();
        }
        buf.EndList();

        return stream;
    }

    NJson::TJsonValue TSherlock::TResponse::HashesAsJson() const {
        NJson::TJsonValue res;
        for(const TVector<TString>& hashes: Hashes) {
            res.AppendValue(Vec2Json(hashes));
        }
        return res;
    }

    TSherlock::TResponse TSherlock::TResponse::Parse(const NJson::TJsonValue& js) {
        TSherlock::TResponse response;
        if(auto delta = MapFindPtr(js.GetMapSafe(), "delta")) {
            if (!delta->IsArray())
                ythrow TWithBackTrace<yexception>() << "sherlock answer must be array: " << *delta;

            for (const auto& diffs : delta->GetArray()) {
                if (!diffs.IsArray())
                    ythrow TWithBackTrace<yexception>() << "sherlock answer must be array of arrays: " << diffs;

                for (const auto& diff : diffs.GetArray()) {
                    if (!diff.IsString())
                        ythrow TWithBackTrace<yexception>() << "sherlock answer must be array of arrays of strings: " << diff;

                    const auto view = StripString(TStringBuf(diff.GetString()));

                    if (view.StartsWith('<') && view.EndsWith('>'))
                        response.Urls.emplace_back(view);
                    else
                        response.Hashes.emplace_back(NText2Shingles::Text2Shingles(view, LANG_UNK, true));
                }
            }
        }
        if(auto stableSign = MapFindPtr(js.GetMapSafe(), "stable_sign")) {
            response.StableSign = stableSign->GetUIntegerRobust();
        }

        return response;
    }

    TMaybe<TSherlock::TResponse> TSherlock::Perform(const struct NFuncClient::TSherlock::TRequest & request, const TLog& logger) {
        NCurl::TRequestContext requestContext;
        if(request.GetQueueId()) {
            requestContext.AddHeader(TStringBuilder{} << "X-REQUEST-ID: " << request.GetQueueId());
        }

        NCurl::TSimpleArtifacts artifacts;
        {
            NJsonWriter::TBuf buf;
            buf.BeginObject();
            {
                buf.WriteKey("html").WriteString(request.GetHtml())
                        .WriteKey("queueId").WriteString(request.GetQueueId())
                        .WriteKey("from").WriteString(request.GetFrom())
                        .WriteKey("subject").WriteString(request.GetSubject())
                        .WriteKey("service").WriteString(request.GetService());

                buf.WriteKey("uids");
                buf.BeginList();
                for(const auto& uid: request.GetUids()) {
                    buf.WriteString(uid);
                }
                buf.EndList();
            }
            buf.EndObject();

            if (auto error = TRequestClient::Perform(artifacts, std::move(TRequestClient::TRequest{}
                .SetRequest("route")
                .SetData(buf)))) {
                logger << (TLOG_ERR) << *error;
                return Nothing();
            }
        }

        try {
            auto response = TResponse::Parse(NJson::ReadJsonTree(&artifacts.body, true));
            return {std::move(response)};
        } catch (...) {
            logger << (TLOG_ERR) << request.GetQueueId() << " TSherlock error:" << CurrentExceptionMessageWithBt();
            PushError();
            return Nothing();
        }
    }
} // namespace NFuncClient
