#include <library/cpp/json/json_value.h>
#include <mail/so/spamstop/tools/so-clients/general_shingler_get.h>
#include <util/generic/xrange.h>
#include <mail/so/spamstop/tools/so-clients/general_shingler_put.h>
#include <util/generic/scope.h>
#include <util/string/builder.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include <mail/so/libs/syslog/so_log.h>
#include "UserReputClient.h"

//**********************************************************************************************************************************
//                                                        TUserRepShingle
//**********************************************************************************************************************************

TUserRepShingle::TUserRepShingle() {
    m_user_uid = 0;
}

TUserRepShingle::TUserRepShingle(ui64 uid) {
    m_user_uid = uid;
}

TUserRepShingle::~TUserRepShingle() {
}

namespace USRREP {

    const TFieldIdStat& GetFieldStat(TFieldId id) {
        static const TFieldIdStat fieldIdStats[UR_END] = {
            {TFieldIdStat::Both, TFieldIdStat::Date, "udate", "UDATE", "udate"},
            {TFieldIdStat::Both, TFieldIdStat::Ui32, "krm", "karma", "karma"},
            {TFieldIdStat::Both, TFieldIdStat::Ui32, "krmst", "KarmaStatus", "krmst"},
            {TFieldIdStat::Full, TFieldIdStat::Date, "fm_ft", "FM_FirstTime", "fm_ft"},
            {TFieldIdStat::Full, TFieldIdStat::Date, "fm_lt", "FM_LastTime", "fm_lt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "ch", "ComplaintHam", "cmplham"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "cs", "ComplaintSpam", "cmplspam"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "h", "Ham", "ham"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "s", "Spam", "spam"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "sh", "SendHam", "sndham"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "ss", "SendSpam", "sndspam"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "uc", "UniqueCompluser", "uniqcmpluser"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "un", "UniqueNetworks", "uniquenetwrks"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "ug", "UniqueGeo", "uniqgeo"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "obm", "OutBitMask", "outbitmsk"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "ad", "ActiveDays", "actdays"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "mrc", "MaxRecipientCount", "maxrcptcnt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "bc", "BounceCount", "bncecnt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "bsp", "BounceSpamCount", "bncespamcunt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "btur", "BounceTodayUnknownRcpt", "bncetdayunkrcpt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "btsr", "BounceTodaySpamRcpt", "bncetdayspamrcpt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "btlur", "BounceTotalUnknownRcpt", "bcetttlunkrcpt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "btlsr", "BounceTotalSpamRcpt", "bcettlspamrcpt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui8, "blrt", "BounceLastRcptType", "bcelastrcpttype"},
            {TFieldIdStat::Full, TFieldIdStat::Date, "bld", "BounceLastDate", "bcelastdate"},
            {TFieldIdStat::Full, TFieldIdStat::Date, "pddd", "PddDate", "pdddate"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "pddmb", "PddMailboxCount", "pddmboxcnt"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "pddrq", "PddResQuota", "pddresquota"},
            {TFieldIdStat::Full, TFieldIdStat::Ui8, "pddfr", "PddFreeRegistration", "pddfreereg"},
            {TFieldIdStat::Full, TFieldIdStat::Ui8, "pddimr", "PddIsManualReg", "pddismnualreg"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "pddfbt", "PddFBounceToday", "pddfbnctoday"},
            {TFieldIdStat::Full, TFieldIdStat::Ui32, "pddfbqa", "PddFBounceQAll", "pddfbounceqall"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "ld", "LastDecision", "lastdecis"},
            {TFieldIdStat::Full, TFieldIdStat::Ui8, "uu", "UnlimitUser", "unlimituser"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "tsm", "T_SpamMails", "t_spammls"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "thm", "T_HamMails", "t_hammls"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "tmm", "T_MalicMails", "t_malicmls"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "tsr", "T_SpamRecipients", "t_spamrcpnts"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "thr", "T_HamRecipients", "t_Hamrcpnts"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "tmr", "T_MalicRecipients", "t_malicrcpnts"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "tur", "T_UniqRecipients", "t_uniqrcpnts"},
            {TFieldIdStat::Full, TFieldIdStat::Ui16, "tug", "T_UniqGeo", "t_uniqgeo"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui32, "tmsm", "T_MySelfMails", "t_myselfmls"},
            {TFieldIdStat::Fast, TFieldIdStat::Date, "fldsc", "F_LastDateSecondCheck", "f_lastdatsecchk"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui32, "afhm", "A_F_HamMails", "af_hammls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui32, "afsm", "A_F_SpamMails", "af_spammls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui32, "afmm", "A_F_MalicMails", "af_malicmls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "tfhm", "T_F_HamMails", "tf_hammls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "tfsm", "T_F_SpamMails", "tf_spammls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "tfmm", "T_F_MalicMails", "tf_malicmls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "lfhm", "L_F_HamMails", "lf_hammls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "lfsm", "L_F_SpamMails", "lf_spammls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "lfmm", "L_F_MalicMails", "lf_malicmls"},
            {TFieldIdStat::Fast, TFieldIdStat::Ui16, "fcc", "F_CacheCount", "fcachecnt"},
            {TFieldIdStat::Fast, TFieldIdStat::Date, "fcd", "F_CacheDate", "fcachedt"},
            {TFieldIdStat::Unknown, TFieldIdStat::Ui32, "rsrv1", "Reserve32_1", "rsrv32_1"},
            {TFieldIdStat::Unknown, TFieldIdStat::Ui32, "rsrv2", "Reserve32_2", "rsrv32_2"},
            {TFieldIdStat::Unknown, TFieldIdStat::Ui16, "rsrv3", "Reserve16_1", "rsrv16_1"},
            {TFieldIdStat::Unknown, TFieldIdStat::Ui16, "rsrv4", "Reserve16_2", "rsrv16_2"},
        };

        return fieldIdStats[id];
    }

    TString getident(TViewIdx index, const TString& rqsttype, const TString& rasm, const TString& storfname_s, const TString& viewtext_s, const TString& logtext_s) {
        TString res = "";

        switch (index) {
            case RQSTTYPE:
                res = rqsttype; //f - full, s - fast, a - both, u - unknown
                break;
            case RASM:
                res = rasm; //s - date, 1 - ui8, 2 - ui16, 4 - ui32
                break;
            case STORFNAME:
                res = storfname_s;
                break;
            case VIEWTEXT:
                res = viewtext_s;
                break;
            case LOGTEXT:
                res = logtext_s;
                break;
        };

        return res;
    }

    TString URToText_(TFieldId fn, TViewIdx idx) {
        TString res = "";

        switch (fn) {
            case UR_UDATE:
                res = getident(idx, "a", "d", "udate", "UDATE", "udate");
                break;
            case UR_karma:
                res = getident(idx, "a", "4", "krm", "karma", "karma");
                break;
            case UR_KarmaStatus:
                res = getident(idx, "a", "4", "krmst", "KarmaStatus", "krmst");
                break;
            case UR_FM_FirstTime:
                res = getident(idx, "f", "d", "fm_ft", "FM_FirstTime", "fm_ft");
                break;
            case UR_FM_LastTime:
                res = getident(idx, "f", "d", "fm_lt", "FM_LastTime", "fm_lt");
                break;
            case UR_ComplaintHam:
                res = getident(idx, "f", "4", "ch", "ComplaintHam", "cmplham");
                break;
            case UR_ComplaintSpam:
                res = getident(idx, "f", "4", "cs", "ComplaintSpam", "cmplspam");
                break;
            case UR_Ham:
                res = getident(idx, "f", "4", "h", "Ham", "ham");
                break;
            case UR_Spam:
                res = getident(idx, "f", "4", "s", "Spam", "spam");
                break;
            case UR_SendHam:
                res = getident(idx, "f", "4", "sh", "SendHam", "sndham");
                break;
            case UR_SendSpam:
                res = getident(idx, "f", "4", "ss", "SendSpam", "sndspam");
                break;
            case UR_UniqueCompluser:
                res = getident(idx, "f", "4", "uc", "UniqueCompluser", "uniqcmpluser");
                break;
            case UR_UniqueNetworks:
                res = getident(idx, "f", "4", "un", "UniqueNetworks", "uniquenetwrks");
                break;
            case UR_UniqueGeo:
                res = getident(idx, "f", "2", "ug", "UniqueGeo", "uniqgeo");
                break;
            case UR_OutBitMask:
                res = getident(idx, "f", "4", "obm", "OutBitMask", "outbitmsk");
                break;
            case UR_ActiveDays:
                res = getident(idx, "f", "2", "ad", "ActiveDays", "actdays");
                break;
            case UR_MaxRecipientCount:
                res = getident(idx, "f", "2", "mrc", "MaxRecipientCount", "maxrcptcnt");
                break;
            case UR_BounceCount:
                res = getident(idx, "f", "2", "bc", "BounceCount", "bncecnt");
                break;
            case UR_BounceSpamCount:
                res = getident(idx, "f", "2", "bsp", "BounceSpamCount", "bncespamcunt");
                break;
            case UR_BounceTodayUnknownRcpt:
                res = getident(idx, "f", "2", "btur", "BounceTodayUnknownRcpt", "bncetdayunkrcpt");
                break;
            case UR_BounceTodaySpamRcpt:
                res = getident(idx, "f", "2", "btsr", "BounceTodaySpamRcpt", "bncetdayspamrcpt");
                break;
            case UR_BounceTotalUnknownRcpt:
                res = getident(idx, "f", "4", "btlur", "BounceTotalUnknownRcpt", "bcetttlunkrcpt");
                break;
            case UR_BounceTotalSpamRcpt:
                res = getident(idx, "f", "4", "btlsr", "BounceTotalSpamRcpt", "bcettlspamrcpt");
                break;
            case UR_BounceLastRcptType:
                res = getident(idx, "f", "1", "blrt", "BounceLastRcptType", "bcelastrcpttype");
                break;
            case UR_BounceLastDate:
                res = getident(idx, "f", "d", "bld", "BounceLastDate", "bcelastdate");
                break;
            case UR_PddDate:
                res = getident(idx, "f", "d", "pddd", "PddDate", "pdddate");
                break;
            case UR_PddMailboxCount:
                res = getident(idx, "f", "4", "pddmb", "PddMailboxCount", "pddmboxcnt");
                break;
            case UR_PddResQuota:
                res = getident(idx, "f", "4", "pddrq", "PddResQuota", "pddresquota");
                break;
            case UR_PddFreeRegistration:
                res = getident(idx, "f", "1", "pddfr", "PddFreeRegistration", "pddfreereg");
                break;
            case UR_PddIsManualReg:
                res = getident(idx, "f", "1", "pddimr", "PddIsManualReg", "pddismnualreg");
                break;
            case UR_PddFBounceToday:
                res = getident(idx, "f", "4", "pddfbt", "PddFBounceToday", "pddfbnctoday");
                break;
            case UR_PddFBounceQAll:
                res = getident(idx, "f", "4", "pddfbqa", "PddFBounceQAll", "pddfbounceqall");
                break;
            case UR_LastDecision:
                res = getident(idx, "f", "2", "ld", "LastDecision", "lastdecis");
                break;
            case UR_UnlimitUser:
                res = getident(idx, "f", "1", "uu", "UnlimitUser", "unlimituser");
                break;
            case UR_T_SpamMails:
                res = getident(idx, "f", "2", "tsm", "T_SpamMails", "t_spammls");
                break;
            case UR_T_HamMails:
                res = getident(idx, "f", "2", "thm", "T_HamMails", "t_hammls");
                break;
            case UR_T_MalicMails:
                res = getident(idx, "f", "2", "tmm", "T_MalicMails", "t_malicmls");
                break;
            case UR_T_SpamRecipients:
                res = getident(idx, "f", "2", "tsr", "T_SpamRecipients", "t_spamrcpnts");
                break;
            case UR_T_HamRecipients:
                res = getident(idx, "f", "2", "thr", "T_HamRecipients", "t_Hamrcpnts");
                break;
            case UR_T_MalicRecipients:
                res = getident(idx, "f", "2", "tmr", "T_MalicRecipients", "t_malicrcpnts");
                break;
            case UR_T_UniqRecipients:
                res = getident(idx, "f", "2", "tur", "T_UniqRecipients", "t_uniqrcpnts");
                break;
            case UR_T_UniqGeo:
                res = getident(idx, "f", "2", "tug", "T_UniqGeo", "t_uniqgeo");
                break;
            case UR_T_MySelfMails:
                res = getident(idx, "s", "4", "tmsm", "T_MySelfMails", "t_myselfmls");
                break;
            case UR_F_LastDateSecondCheck:
                res = getident(idx, "s", "d", "fldsc", "F_LastDateSecondCheck", "f_lastdatsecchk");
                break;
            case UR_A_F_HamMails:
                res = getident(idx, "s", "4", "afhm", "A_F_HamMails", "af_hammls");
                break;
            case UR_A_F_SpamMails:
                res = getident(idx, "s", "4", "afsm", "A_F_SpamMails", "af_spammls");
                break;
            case UR_A_F_MalicMails:
                res = getident(idx, "s", "4", "afmm", "A_F_MalicMails", "af_malicmls");
                break;
            case UR_T_F_HamMails:
                res = getident(idx, "s", "2", "tfhm", "T_F_HamMails", "tf_hammls");
                break;
            case UR_T_F_SpamMails:
                res = getident(idx, "s", "2", "tfsm", "T_F_SpamMails", "tf_spammls");
                break;
            case UR_T_F_MalicMails:
                res = getident(idx, "s", "2", "tfmm", "T_F_MalicMails", "tf_malicmls");
                break;
            case UR_L_F_HamMails:
                res = getident(idx, "s", "2", "lfhm", "L_F_HamMails", "lf_hammls");
                break;
            case UR_L_F_SpamMails:
                res = getident(idx, "s", "2", "lfsm", "L_F_SpamMails", "lf_spammls");
                break;
            case UR_L_F_MalicMails:
                res = getident(idx, "s", "2", "lfmm", "L_F_MalicMails", "lf_malicmls");
                break;
            case UR_F_CacheCount:
                res = getident(idx, "s", "2", "fcc", "F_CacheCount", "fcachecnt");
                break;
            case UR_F_CacheDate:
                res = getident(idx, "s", "d", "fcd", "F_CacheDate", "fcachedt");
                break;
            case UR_Reserve32_1:
                res = getident(idx, "u", "4", "rsrv1", "Reserve32_1", "rsrv32_1");
                break;
            case UR_Reserve32_2:
                res = getident(idx, "u", "4", "rsrv2", "Reserve32_2", "rsrv32_2");
                break;
            case UR_Reserve16_1:
                res = getident(idx, "u", "2", "rsrv3", "Reserve16_1", "rsrv16_1");
                break;
            case UR_Reserve16_2:
                res = getident(idx, "u", "2", "rsrv4", "Reserve16_2", "rsrv16_2");
                break;
            default:
                res = "???";
        };

        return res;
    };

    TFieldId STORFNToFieldId(const TString& storname) {
        TFieldId res = UR_END;

        for (size_t i = 0; i < UR_END; i++) {
            if (STORFN(static_cast<TFieldId>(i)) == storname) {
                res = static_cast<TFieldId>(i);
                break;
            }
        }

        return res;
    }

    TString STORFN(TFieldId fn) //return fieldname in storage
    {
        return URToText_(fn, STORFNAME);
    };

    TString VIEWFN(TFieldId fn) //return fieldname for user interface
    {
        return URToText_(fn, VIEWTEXT);
    };

    TString LOGFN(TFieldId fn) //return fieldname for log
    {
        return URToText_(fn, LOGTEXT);
    };

    TRasmType RASMF(TFieldId fn) //field type (s - date, 1 - ui8, 2 - ui16, 4 - ui32)
    {
        TRasmType res = RUI32;
        TString size_type_s = "";

        size_type_s = URToText_(fn, RASM);
        if (size_type_s == "d")
            res = RDATE;
        else if (size_type_s == "1")
            res = RUI8;
        else if (size_type_s == "2")
            res = RUI16;
        else if (size_type_s == "4")
            res = RUI32;

        return res;
    };

    TString RASMFs(TFieldId fn) {
        TString res = "";

        TRasmType value = RASMF(fn);
        switch (value) {
            case RDATE:
                res = "DATE";
                break;
            case RUI8:
                res = "UI8";
                break;
            case RUI16:
                res = "UI16";
                break;
            case RUI32:
                res = "UI32";
                break;
        };

        return res;
    };

    TUseType USEF(TFieldId fn) //use field (f - full, s - fast, a - both, u - unknown)
    {
        TUseType res = USE_UNKNOWN;
        TString rqst_type_s = "";

        rqst_type_s = URToText_(fn, RQSTTYPE);
        if (rqst_type_s == "f")
            res = USE_FULL;
        else if (rqst_type_s == "s")
            res = USE_FAST;
        else if (rqst_type_s == "a")
            res = USE_BOTH;
        else if (rqst_type_s == "u")
            res = USE_UNKNOWN;

        return res;
    };

    TString USEFs(TFieldId fn) {
        TString res = "";
        TUseType value = USEF(fn);

        switch (value) {
            case USE_UNKNOWN:
                res = "UNKNOWN";
                break;
            case USE_FULL:
                res = "FULL";
                break;
            case USE_FAST:
                res = "FAST";
                break;
            case USE_BOTH:
                res = "FULL+FAST";
                break;
        };

        return res;
    }

    bool IsFullField(TFieldId fn) {
        if ((USEF(fn) == USE_FULL) || (USEF(fn) == USE_BOTH))
            return true;
        else
            return false;
    }

    bool IsFastField(TFieldId fn) {
        if ((USEF(fn) == USE_FAST) || (USEF(fn) == USE_BOTH))
            return true;
        else
            return false;
    }

    typedef TString (*CreateNameFunction)(TFieldId);
    void CheckDublicatFNameByFunction(CreateNameFunction createName, const char* FunctionName) {
        typedef THashMap<TString, size_t> DataHash;
        typedef std::pair<DataHash::iterator, bool> InsertResult;

        DataHash data;
        for (size_t i = 0; i < UR_END; ++i) {
            const TString name = createName(static_cast<TFieldId>(i));
            Y_VERIFY(!name.empty() && name != "???", "%s function: Unknown field: %" PRISZT, FunctionName, i);

            const InsertResult result = data.insert(std::make_pair(name, i));
            Y_VERIFY(result.second, "%s function: Find fields with same name: %" PRISZT " and %" PRISZT, FunctionName, i, result.first->second);
        }
    }

    void CheckDublicatFName() {
        CheckDublicatFNameByFunction(&STORFN, "STORFN");
        CheckDublicatFNameByFunction(&VIEWFN, "VIEWFN");
        CheckDublicatFNameByFunction(&LOGFN, "LOGFN");
    }

    TString TSourceToTString(TSource value) {
        TString res = "UNKNOWN";

        switch (value) {
            case SRC_UNKNOWN:
                res = "SRC_UNKNOWN";
                break;
            case SRC_STORAGE:
                res = "SRC_STORAGE";
                break;
            case SRC_CACHE:
                res = "SRC_CACHE";
                break;
        };

        return res;
    }

    TString AnyValueReputToTString(TReputHash::const_iterator& hit, nosql::TDataCounters& counters) {
        TString recstr = "";
        TString tstr = "";

        switch ((*hit).second.Type()) {
            case nosql::AnyValue::NOTYPE:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::NOTYPE) + "''&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::NOTYPE)] = IncMax64(counters.count_by_type[static_cast<int>(nosql::AnyValue::NOTYPE)], 1);
                break;
            case nosql::AnyValue::INTEGER:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::INTEGER) + "'" + IntToStroka((*hit).second.Integer()) + "'&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::INTEGER)] = IncMax64(static_cast<int>(counters.count_by_type[nosql::AnyValue::INTEGER]), 1);
                break;
            case nosql::AnyValue::INTEGER64:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::INTEGER64) + "'" + I64ToStroka((*hit).second.Long()) + "'&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::INTEGER64)] = IncMax64(static_cast<int>(counters.count_by_type[nosql::AnyValue::INTEGER64]), 1);
                break;
            case nosql::AnyValue::DOUBLE:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::DOUBLE) + "'" + DoubleToStr((*hit).second.Double()) + "'&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::DOUBLE)] = IncMax64(counters.count_by_type[static_cast<int>(nosql::AnyValue::DOUBLE)], 1);
                break;
            case nosql::AnyValue::VECTOR:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::VECTOR) + "''&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::VECTOR)] = IncMax64(counters.count_by_type[static_cast<int>(nosql::AnyValue::VECTOR)], 1);
                break;
            case nosql::AnyValue::STRING:
                tstr = (*hit).second.String();
                CGIEscape(tstr);
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::STRING) + "'" + tstr + "'&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::STRING)] = IncMax64(counters.count_by_type[static_cast<int>(nosql::AnyValue::STRING)], 1);
                break;
            case nosql::AnyValue::LIST:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::LIST) + "''&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::LIST)] = IncMax64(counters.count_by_type[static_cast<int>(nosql::AnyValue::LIST)], 1);
                break;
            case nosql::AnyValue::HASH:
                recstr += IntToStroka(static_cast<ui32>((*hit).first)) + "=#" + IntToStroka(nosql::AnyValue::HASH) + "''&";
                counters.count_by_type[static_cast<int>(nosql::AnyValue::HASH)] = IncMax64(counters.count_by_type[static_cast<int>(nosql::AnyValue::HASH)], 1);
                break;
        };

        return recstr;
    }

    TString ReputHashToTString(const TReputHash& hash) {
        TString recstr = "";
        TReputHash::const_iterator hit;
        nosql::TDataCounters counters;

        hit = hash.begin();
        while (hit != hash.end()) {
            recstr += AnyValueReputToTString(hit, counters);

            ++hit;
        }

        return recstr;
    }

    bool pTStringToReputHash(const char* pdatastr, size_t /*pdatastrlen*/, TReputHash& hash, TString& err_str) {
        bool res = false;
        const char* pb = nullptr;
        const char* pe = nullptr;
        const char* p1 = nullptr;
        const char* p2 = nullptr;
        const char* p3 = nullptr;
        TString idn2 = "=#";
        TString fieldname = "";
        TString fieldtype = "";
        TString fieldvalue = "";
        int fieldtype_i = 0;
        int count = 0;
        nosql::AnyValue::TYPE fieldtype_av;
        i64 value_i64_av = 0;
        int value_int_av = 0;
        double value_double_av = 0;
        TFieldId field_id;

        err_str = "";
        hash.clear();
        if (pdatastr != nullptr) {
            pb = pdatastr;
            pe = (pb != nullptr) ? strchr(pb, '&') : nullptr;
            while (pe != nullptr) {
                field_id = UR_END;
                fieldname = "";
                fieldtype = "";
                fieldvalue = "";
                fieldtype_i = 0;
                fieldtype_av = nosql::AnyValue::NOTYPE;

                p1 = strstr(pb, idn2.c_str());
                if ((p1 != nullptr) && (p1 < pe)) {
                    count = p1 - pb;
                    if (count > 0) {
                        fieldname = TString(pb, count);
                        if (!fieldname.empty())
                            field_id = static_cast<TFieldId>(atol(fieldname.c_str()));

                    } else {
                        err_str += "can't find fieldname2, ";
                    }

                    p2 = strchr(p1 + idn2.length(), '\'');
                    if ((p2 != nullptr) && (p2 < pe)) {
                        count = p2 - p1 - idn2.length();
                        if (count > 0) {
                            fieldtype = TString(p1 + idn2.length(), count);
                            if (!fieldtype.empty()) {
                                fieldtype_i = atoi(fieldtype.c_str());
                                fieldtype_av = static_cast<nosql::AnyValue::TYPE>(fieldtype_i);
                            } else {
                                err_str += "can't find fieldtype2, ";
                            }

                            p3 = strchr(p2 + 1, '\'');
                            if ((p3 != nullptr) && (p3 < pe)) {
                                count = p3 - p2 - 1;
                                fieldvalue = TString(p2 + 1, count);
                                if (!fieldvalue.empty())
                                    CGIUnescape(fieldvalue);

                                switch (fieldtype_av) {
                                    case nosql::AnyValue::INTEGER:
                                        value_int_av = 0;
                                        if ((!fieldvalue.empty()) && (sscanf(fieldvalue.c_str(), "%d", &value_int_av) == 1))
                                            hash[field_id] = static_cast<nosql::AnyValue>(value_int_av);
                                        break;
                                    case nosql::AnyValue::INTEGER64:
                                        value_i64_av = 0;
                                        if ((!fieldvalue.empty()) && (sscanf(fieldvalue.c_str(), "%" PRIi64, &value_i64_av) == 1))
                                            hash[field_id] = static_cast<nosql::AnyValue>(value_i64_av);
                                        break;
                                    case nosql::AnyValue::DOUBLE:
                                        value_double_av = 0;
                                        if (!fieldvalue.empty()) {
                                            value_double_av = atof(fieldvalue.c_str());
                                            hash[field_id] = static_cast<nosql::AnyValue>(value_double_av);
                                        }
                                        break;
                                    case nosql::AnyValue::VECTOR:
                                        err_str += VIEWFN(field_id) + ": no support VECTOR type, ";
                                        break;
                                    case nosql::AnyValue::STRING:
                                        hash[field_id] = static_cast<nosql::AnyValue>(fieldvalue);
                                        break;
                                    case nosql::AnyValue::LIST:
                                        err_str += VIEWFN(field_id) + ": no support LIST type, ";
                                        break;
                                    case nosql::AnyValue::HASH:
                                        err_str += VIEWFN(field_id) + ": no support HASH type, ";
                                        break;
                                    default:
                                        err_str += VIEWFN(field_id) + ": no define type, ";
                                };

                            } else {
                                err_str += "can't find fieldvalue1, ";
                            }

                        } else {
                            err_str += "can't find fieldtype2, ";
                        }

                    } else {
                        err_str += "can't find fieldtype1, ";
                    }

                } else {
                    err_str += "can't find fieldname1, ";
                }

                pb = pe + 1;
                pe = (pb != nullptr) ? strchr(pb, '&') : nullptr;
            }

        } else {
            err_str += "empty record data";
        }

        if (err_str.empty())
            res = true;

        return res;
    }

    bool TStringToReputHash(const TString& datastr, TReputHash& hash, TString& err_str) {
        return pTStringToReputHash(datastr.c_str(), datastr.length(), hash, err_str);
    }

    NJson::TJsonValue TCntData::ToJson() const {
        NJson::TJsonValue res;
        for (size_t i : xrange((size_t)USRREP::UR_END)) {
            if (m_setflags[i] == 0)
                continue;

            res[GetFieldStat(TFieldId(i)).name] = m_counters[i];
        }

        return res;
    }

    void TCntData::FromJson(const NJson::TJsonValue& value) {
        m_empty = true;
        for (size_t i = 0; i < (size_t)USRREP::UR_END; i++) {
            const USRREP::TFieldIdStat& stat = USRREP::GetFieldStat(USRREP::TFieldId(i));
            if (!value.Has(stat.name))
                continue;
            m_empty = false;
            m_setflags[i] = 1;
            m_counters[i] = static_cast<ui32>(value[stat.name].GetUIntegerRobust());
        }
    }

    TCntData::TCntData() {
        Clear();
    }

    void TCntData::Clear() {
        m_source = SRC_UNKNOWN;
        m_gettime = 0;
        m_empty = true;
        m_error = false;
        m_error_text = "";
        memset(m_counters, 0, sizeof(m_counters));
        memset(m_setflags, 0, sizeof(m_setflags));
    }

    TString TCntData::SerializeCounters() {
        const int tbuff_size = 1024;
        char tbuff[tbuff_size];
        ui32 intbuff = 0;
        int capacity = 0;

        tbuff[tbuff_size - 1] = 0x0;
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            capacity = tbuff_size - intbuff - 1;
            if (capacity > 0)
                intbuff += snprintf(tbuff + intbuff, capacity, "%x-", m_counters[i]);
        }

        return TString(tbuff);
    }

    TString TCntData::SerializeFlags() {
        const int tbuff_size = 256;
        char tbuff[tbuff_size];
        ui32 intbuff = 0;
        int capacity = 0;

        tbuff[tbuff_size - 1] = 0x0;
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            capacity = tbuff_size - intbuff - 1;
            if (capacity > 0)
                intbuff += snprintf(tbuff + intbuff, capacity, "%d-", m_setflags[i]);
        }

        return TString(tbuff);
    }

    bool TCntData::DeserializeCounters(const TString& data_s) {
        bool res = false;
        const char* pb = nullptr;
        const char* pe = nullptr;
        int count = 0;
        TString st = "";
        ui32 value = 0;
        int index = -1;

        memset(m_counters, 0, sizeof(m_counters));
        if (!data_s.empty()) {
            pb = data_s.c_str();
            if (pb != nullptr)
                pe = strchr(pb, '-');
            while ((pb != nullptr) && (pe != nullptr)) {
                count = pe - pb;
                if (count > 0) {
                    st = TString(pb, count);
                    if (sscanf(st.c_str(), "%x", &value) == 1) {
                        index++;
                        if ((index >= 0) && (index < USRREP::UR_END)) {
                            m_counters[index] = value;
                            res = true;

                        } else {
                            res = false;
                            break;
                        }
                    }
                }

                pb = pe + 1;
                pe = (pb != nullptr) ? strchr(pb, '-') : nullptr;
            }
        }

        return res;
    }

    bool TCntData::DeserializeFlags(const TString& data_s) {
        bool res = false;
        const char* pb = nullptr;
        const char* pe = nullptr;
        int count = 0;
        TString st = "";
        ui32 value = 0;
        ui8 vv = 0;
        int index = -1;

        memset(m_setflags, 0, sizeof(m_setflags));
        if (!data_s.empty()) {
            pb = data_s.c_str();
            if (pb != nullptr)
                pe = strchr(pb, '-');
            while ((pb != nullptr) && (pe != nullptr)) {
                count = pe - pb;
                if (count > 0) {
                    st = TString(pb, count);
                    if (sscanf(st.c_str(), "%u", &value) == 1) {
                        index++;
                        if ((index >= 0) && (index < USRREP::UR_END)) {
                            vv = (value > 0xFF) ? 0xFF : value & 0xFF;
                            m_setflags[index] = vv;
                            res = true;

                        } else {
                            res = false;
                            break;
                        }
                    }
                }

                pb = pe + 1;
                pe = (pb != nullptr) ? strchr(pb, '-') : nullptr;
            }
        }

        return res;
    }

    void TCntData::FillPutAllDebug() {
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            m_counters[i] = USRREP::UR_END - i - 1;
            m_setflags[i] = 1;
        }
    }

    void TCntData::FillPutFullDebug() {
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            if ((USEF((TFieldId)i) == USE_FULL) || (USEF((TFieldId)i) == USE_BOTH)) {
                m_counters[i] = i;
                m_setflags[i] = 1;
            }
        }
    }

    void TCntData::FillPutFullMaxDebug() {
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            if ((USEF((TFieldId)i) == USE_FULL) || (USEF((TFieldId)i) == USE_BOTH)) {
                switch (RASMF((TFieldId)i)) {
                    case RDATE:
                        m_counters[i] = 1999999999;
                        m_setflags[i] = 1;
                        break;
                    case RUI8:
                        m_counters[i] = 127;
                        m_setflags[i] = 1;
                        break;
                    case RUI16:
                        m_counters[i] = 32000;
                        m_setflags[i] = 1;
                        break;
                    case RUI32:
                        m_counters[i] = 1999999999;
                        m_setflags[i] = 1;
                        break;
                };
            }
        }
    }

    TString TCntData::PrintDataToLog() {
        TString res = "";
        const int tbuff_size = 1024;
        char tbuff[tbuff_size];
        ui32 intbuff = 0;
        int capacity = 0;

        res += TSourceToTString(m_source) + " ";
        res += "gt='" + TimeToStr(m_gettime) + "' ";
        if (m_error)
            res += "err=yes(" + m_error_text + ") ";
        else
            res += "err=no ";
        if (m_empty) {
            res += "empty ";

        } else {
            tbuff[tbuff_size - 1] = 0x0;
            for (size_t i = 0; i < USRREP::UR_END; i++) {
                capacity = tbuff_size - intbuff - 1;
                if (capacity > 0) {
                    if (m_setflags[i] > 0)
                        intbuff += snprintf(tbuff + intbuff, capacity, "%u,", m_counters[i]);
                    else
                        intbuff += snprintf(tbuff + intbuff, capacity, "-,");
                }
            }
            if (tbuff[intbuff - 1] == ',')
                tbuff[intbuff - 1] = 0x00;

            res += "cntrs='" + TString(tbuff) + "'";
        }

        return res;
    }

    TString TCntData::PrintOnlyCountersToLog() {
        TString res = "";
        const int tbuff_size = 1024;
        char tbuff[tbuff_size];
        ui32 intbuff = 0;
        int capacity = 0;

        tbuff[tbuff_size - 1] = 0x0;
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            capacity = tbuff_size - intbuff - 1;
            if (capacity > 0) {
                if (m_setflags[i] > 0)
                    intbuff += snprintf(tbuff + intbuff, capacity, "%u,", m_counters[i]);
                else
                    intbuff += snprintf(tbuff + intbuff, capacity, "-,");
            }
        }
        if (tbuff[intbuff - 1] == ',')
            tbuff[intbuff - 1] = 0x00;

        res += "cntrs='" + TString(tbuff) + "'";

        return res;
    }

    TString TCntData::PrintOnlyCountersToLog2() {
        TString res = "";
        const int tbuff_size = 4096;
        ui32 intbuff = 0;
        int capacity = 0;
        TString field_name = "";

        TVector<char> tbuff(tbuff_size);
        tbuff[tbuff_size - 1] = 0x0;
        for (size_t i = 0; i < USRREP::UR_END; i++) {
            capacity = tbuff_size - intbuff - 1;
            if (capacity > 0) {
                field_name = USRREP::LOGFN(static_cast<USRREP::TFieldId>(i));
                if (m_setflags[i] > 0)
                    intbuff += snprintf(tbuff.data() + intbuff, capacity, "%s=%u,", field_name.c_str(), m_counters[i]);
                else
                    intbuff += snprintf(tbuff.data() + intbuff, capacity, "%s=-,", field_name.c_str());
            }
        }
        if (tbuff[intbuff - 1] == ',')
            tbuff[intbuff - 1] = 0x00;

        res += "cntrs='" + TString(tbuff.data()) + "'";

        return res;
    }
} // namespace USRREP

//**********************************************************************************************************************************
//                                                        TUserRepRequest
//**********************************************************************************************************************************

TUserRepRequest::TUserRepRequest() {
    Clear();
    m_shingle = TUserRepShingle();
}

TUserRepRequest::TUserRepRequest(const char* strUid) {
    ui64 value = 0;

    Clear();
    if (strUid != nullptr) {
        if (sscanf(strUid, "%" PRIu64, &value) == 1)
            m_shingle = TUserRepShingle(value);
    }
}

TUserRepRequest::TUserRepRequest(ui64 uid) {
    m_shingle = TUserRepShingle(uid);
    Clear();
}

TUserRepRequest::~TUserRepRequest() {
}

void TUserRepRequest::ChangeUid(ui64 uid) {
    m_shingle = TUserRepShingle(uid);
}

void TUserRepRequest::ChangeUid(const char* strUid) {
    ui64 value = 0;

    if (strUid != nullptr) {
        if (sscanf(strUid, "%" PRIu64, &value) == 1)
            m_shingle = TUserRepShingle(value);
    }
}

bool TUserRepRequest::Undefined() {
    bool res = false;

    if (UserUid() == 0)
        res = true;

    return res;
}

ui64 TUserRepRequest::UserUid() const {
    return m_shingle.UserUid();
}

TString TUserRepRequest::UserUidS() {
    return m_shingle.UserUidS();
}

void TUserRepRequest::Clear() {
    m_set_counters.Clear();
    m_res_counters.Clear();
    m_res_limit_th = false;
}

bool TUserRepRequest::CheckSetsIncrs(USRREP::TReputHash& sets, USRREP::TReputHash& incrs) {
    bool res = true;
    USRREP::TReputHashIt it_s, it_i;

    it_s = sets.begin();
    while (it_s != sets.end()) {
        it_i = incrs.find((*it_s).first);
        if (it_i != incrs.end()) {
            res = false;
            break;
        }

        ++it_s;
    }

    return res;
}

bool TUserRepRequest::CheckPutFull() {
    bool res = true;

    for (size_t i = 0; i < USRREP::UR_END; i++) {
        if (m_set_counters.m_setflags[static_cast<USRREP::TFieldId>(i)] > 0) {
            if (!IsFullField(static_cast<USRREP::TFieldId>(i))) {
                res = false;
                break;
            }
        }
    }

    return res;
}

bool TUserRepRequest::CheckPutFast(USRREP::TReputHash& sets, USRREP::TReputHash& incrs) {
    bool res = true;
    USRREP::TReputHashIt it;

    if (res) {
        it = sets.begin();
        while (it != sets.end()) {
            if (!IsFastField((*it).first)) {
                res = false;
                break;
            }

            ++it;
        }
    }

    if (res) {
        it = incrs.begin();
        while (it != incrs.end()) {
            if (!IsFastField((*it).first)) {
                res = false;
                break;
            }

            ++it;
        }
    }

    return res;
}

TString TUserRepRequest::GetRequestFull() {
    return "uid=" + m_shingle.UserUidS();
}

bool TUserRepRequest::PutRequestFull(TString& request, TString& error_s) {
    bool res = false;

    error_s = "";
    request = "uid=" + m_shingle.UserUidS() + "&cntflags=" + m_set_counters.SerializeFlags() + "&cntdata=" + m_set_counters.SerializeCounters();
    res = true;

    return res;
}

TString TUserRepRequest::GetRequestFast() {
    return "uid=" + m_shingle.UserUidS();
}

TString TUserRepRequest::PutRequestFast(USRREP::TReputHash& sets, USRREP::TReputHash& incrs, bool only_cache) {
    TString dopdata = "";
    TString sets_s = USRREP::ReputHashToTString(sets);
    TString incrs_s = USRREP::ReputHashToTString(incrs);

    CGIEscape(sets_s);
    CGIEscape(incrs_s);

    if (only_cache)
        dopdata = "&onlycache=1";

    return "uid=" + m_shingle.UserUidS() + "&sets=" + sets_s + "&incrs=" + incrs_s + dopdata;
}

NJson::TJsonValue TUserRepRequest::ToJson() const {
    auto res = m_set_counters.ToJson();

    res["id"] = m_shingle.UserUid();

    return res;
}

NJson::TJsonValue TUserRepRequest::ResultToJson() const {
    return m_res_counters.ToJson();
}

bool TUserRepRequest::FillDataFromServer(const TString& /*uid*/, nosql::HashMap& data) {
    bool res = true;
    bool ok1 = false;
    bool ok2 = false;
    ui32 vers = 0;
    ui32 empty = 0;
    ui32 err = 0;
    ui32 src = 0;
    ui32 gettime = 0;
    TString err_str = "";
    TString uid_from_hash = "";
    TString serialize_cntrs = "";
    TString serialize_flags = "";
    ui32 limit_th = 0;

    limit_th = data["limit_th"].Integer();
    vers = data["vers"].Integer();
    uid_from_hash = data["uid"].String();
    src = data["src"].Integer();
    gettime = data["gtime"].Integer();
    empty = data["empty"].Integer();
    err = data["err"].Integer();
    err_str = data["err_text"].String();
    serialize_cntrs = data["cnt_data"].String();
    serialize_flags = data["cnt_flgs"].String();

    m_res_limit_th = limit_th;

    m_res_counters.Clear();
    m_res_counters.m_source = static_cast<USRREP::TSource>(src);
    m_res_counters.m_gettime = gettime;
    m_res_counters.m_empty = (bool)empty;
    m_res_counters.m_error = (bool)err;
    m_res_counters.m_error_text = err_str;
    ok1 = m_res_counters.DeserializeCounters(serialize_cntrs);
    ok2 = m_res_counters.DeserializeFlags(serialize_flags);

    res = ok1 && ok2;

    return res;
}

bool TUserRepRequest::FillDataFromGeneralServer(const NJson::TJsonValue& value) {
    m_res_counters.FromJson(value);

    return !m_res_counters.m_empty;
}

TString TUserRepRequest::PrintResCounters() {
    TString res = "";

    if (m_res_limit_th)
        res += "LIMIT_TH ";
    res += m_res_counters.PrintDataToLog();

    return res;
}

const USRREP::TCntData& TUserRepRequest::getResultRawCounters() const {
    return m_res_counters;
}

TString TUserRepRequest::PrintSetCounters() {
    return m_set_counters.PrintOnlyCountersToLog();
}

TString TUserRepRequest::PrintSetCounters2() {
    return m_set_counters.PrintOnlyCountersToLog2();
}

void TUserRepRequest::FillPutAllDebug() {
    m_set_counters.FillPutAllDebug();
}

void TUserRepRequest::FillPutFullDebug() {
    m_set_counters.FillPutFullDebug();
}

void TUserRepRequest::FillPutFullMaxDebug() {
    m_set_counters.FillPutFullMaxDebug();
}

void TUserRepRequest::SetFieldValue(USRREP::TFieldId fntype, ui32 value) {
    if (USRREP::RASMF(fntype) == USRREP::RDATE || USRREP::RASMF(fntype) == USRREP::RUI32)
        m_set_counters.m_counters[fntype] = value > MAX_U32_2BLN ? MAX_U32_2BLN : value;
    else
        m_set_counters.m_counters[fntype] = value > MAX_U16_32K ? MAX_U16_32K : value;

    m_set_counters.m_setflags[fntype] = 1;
}

bool TUserRepRequest::GetEmpty() {
    return m_res_counters.m_empty;
}

USRREP::TSource TUserRepRequest::GetSource() {
    return m_res_counters.m_source;
}

ui32 TUserRepRequest::GetReceiveDataTime() {
    return m_res_counters.m_gettime;
}

bool TUserRepRequest::GetError() {
    return m_res_counters.m_error;
}

TString TUserRepRequest::GetErrorText() {
    return m_res_counters.m_error_text;
}

bool TUserRepRequest::GetFieldValue(USRREP::TFieldId fntype, ui32& value) {
    bool res = false;

    if (m_res_counters.m_setflags[fntype] > 0) {
        value = m_res_counters.m_counters[fntype];
        res = true;
    }

    return res;
}
//**********************************************************************************************************************************
//                                                      CUserReputationClient
//**********************************************************************************************************************************

namespace NFuncClient {
    ui32 TUserReputationClient::GetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size) {
        long pos = -1;
        ui32 res = 0;
        ui8 dob = 0;

        for (ui32 i = 0; i < source_size; i++) {
            if ((source[i] == 0x0D) && ((i + 1) < source_size) && (source[i + 1] == 0x0A)) {
                pos = i;
                dob = 1;
                break;
            }
            if (source[i] == 0x0A) {
                pos = i;
                dob = 0;
                break;
            }
        }
        res = pos + 1 + dob;
        if (res == 0)
            res = source_size;
        if (res > 0) {
            memset(destination, 0, destination_size);
            if (res <= (destination_size - 1)) {
                memcpy(destination, source, res);
            } else {
                memcpy(destination, source, destination_size - 1);
            }
        }
        return res;
    }

    bool TUserReputationClient::TGetter::Perform( TUserRepRequest& userrqst, const TLog& logger) const {
        NGeneralShingler::TGetRequest request;
        {
            NJson::TJsonValue::TArray jsMessages;

            auto& res = jsMessages.emplace_back();

            res["id"] = userrqst.UserUid();

            request.AddMessage("set", std::move(jsMessages));
        }

        NCurl::TSimpleArtifacts artifacts;
        if(auto error = TRequestClient::Perform(artifacts, TRequest{}
            .SetRequest(TStringBuilder{} << request.getIdent() << '?' << request.getQueryParams())
            .SetData(request.getRequestData()))) {
            logger << (TLOG_ERR) << *error;
            return false;
        }

        request.parseFromString(artifacts.body.Str());

        const auto& answer = request.GetAnswer();
        if (!answer.IsArray())
            return false;

        const auto& results = answer.GetArray();

        if (results.empty() || !results.front().IsMap() || !results.front().Has("find"))
            return false;

        const auto& find = results.front()["find"];

        if (!find.IsArray() || find.GetArray().empty())
            return false;

        userrqst.FillDataFromGeneralServer(find.GetArray()[0]);

        return true;
    }

    bool TUserReputationClient::TFullUpdater::Perform(const TUserRepRequest& userrqst, const TLog& logger) const {
        NGeneralShingler::TPutRequest request;

        NJson::TJsonValue::TArray jsMessages;

        jsMessages.emplace_back(userrqst.ToJson());

        request.AddMessage("set", std::move(jsMessages));

        NCurl::TSimpleArtifacts artifacts;
        if(auto error = TRequestClient::Perform(artifacts, TRequest{}
            .SetRequest(TStringBuilder{} << request.getIdent() << '?' << request.getQueryParams())
            .SetData(request.getRequestData()))) {
            logger << (TLOG_ERR) << *error;
            return false;
        }

        return true;
    }

    static void AnyvalueToJson(const USRREP::TFieldIdStat& fieldIdStat, const nosql::AnyValue& src, NJson::TJsonValue& target) {
        switch (fieldIdStat.fieldType) {
            case USRREP::TFieldIdStat::Double:
                target[fieldIdStat.name] = src.Double();
                break;
            case USRREP::TFieldIdStat::Ui64:
                target[fieldIdStat.name] = src.Long();
                break;
            case USRREP::TFieldIdStat::Date:
            case USRREP::TFieldIdStat::Ui32:
            case USRREP::TFieldIdStat::Ui16:
            case USRREP::TFieldIdStat::Ui8:
                target[fieldIdStat.name] = src.Integer();
                break;
            case USRREP::TFieldIdStat::String:
                target[fieldIdStat.name] = src.String();
                break;
        }
    }

    void RepHashToJson(const USRREP::TReputHash& src, NJson::TJsonValue& target) {
        for (const auto& p : src) {
            const auto& fieldId = USRREP::GetFieldStat(p.first);
            AnyvalueToJson(fieldId, p.second, target);
        }
    }

    bool TUserReputationClient::TFastUpdater::Perform(const TUserRepRequest& userrqst, const USRREP::TReputHash& sets, const USRREP::TReputHash& incrs, const TLog& logger) const {
        if (incrs.empty() && sets.empty())
            return true;

        NGeneralShingler::TPutRequest request;

        if (!incrs.empty()) {
            NJson::TJsonValue::TArray jsMessages;

            auto& message = jsMessages.emplace_back();
            message["id"] = userrqst.UserUid();
            RepHashToJson(incrs, message);

            request.AddMessage("incr", std::move(jsMessages));
        }

        if (!sets.empty()) {
            NJson::TJsonValue::TArray jsMessages;

            auto& message = jsMessages.emplace_back();
            message["id"] = userrqst.UserUid();
            RepHashToJson(sets, message);

            request.AddMessage("set", std::move(jsMessages));
        }

        NCurl::TSimpleArtifacts artifacts;
        if(auto error = TRequestClient::Perform(artifacts, TRequest{}
            .SetRequest(TStringBuilder{} << request.getIdent() << '?' << request.getQueryParams())
            .SetData(request.getRequestData()))) {
            logger << (TLOG_ERR) << *error;
            return false;
        }

        return true;
    }
} // namespace NFuncClient

//**********************************************************************************************************************************
