#include <util/generic/ptr.h>
#include <util/string/builder.h>
#include <util/generic/scope.h>
#include <mail/so/libs/syslog/so_log.h>

#include "single_request_client.h"

TRequestClient::TArgs::TArgs() : Pool(MakeAtomicShared<NCurl::TPool>()) {}

TRequestClient::TArgs& TRequestClient::TArgs::SetPool(TAtomicSharedPtr<NCurl::TPool> pool) {
    Pool = std::move(pool);
    return *this;
}

TRequestClient::TArgs& TRequestClient::TArgs::SetTvm(TAtomicSharedPtr<NTvmAuth::TTvmClient> client) {
    TvmClient = std::move(client);
    return *this;
}

TRequestClient::TArgs TRequestClient::TArgs::PopulateWithYasm(const TStringBuf& s) const {
    if(!Config.YasmPrefix)
        return *this;

    auto args = *this;
    args.Config.YasmPrefix += s;
    return args;
}

TRequestClient::TArgs::TArgs(TClientConfig config)
    : Config(std::move(config))
    , Pool(MakeAtomicShared<NCurl::TPool>(Config.PoolParams)) {};

TRequestClient::TRequest&& TRequestClient::TRequest::SetRequest(TString value) && noexcept {
    Request = std::move(value);
    return std::move(*this);
}
TRequestClient::TRequest&& TRequestClient::TRequest::SetData(TString value) && noexcept {
    Data = std::move(value);
    return std::move(*this);
}

TRequestClient::TRequest&& TRequestClient::TRequest::SetData(const NJsonWriter::TBuf& buf) && noexcept {
    Data = std::move(buf.Str());

    return std::move(*this).SetHeaders({{"Content-Type: application/json"}});
}

TRequestClient::TRequest&& TRequestClient::TRequest::SetHeaders(TVector<TString> value) && noexcept {
    if(!Headers) {
        Headers = std::move(value);
    } else {
        for(auto& h : value) {
            Headers->emplace_back(std::move(h));
        }
    }
    return std::move(*this);
}

TRequestClient::TRequestClient(TRequestClient::TArgs args) noexcept
    : Config(std::move(args.Config))
    , Pool(std::move(args.Pool))
    , PoolTraits(Config.Ssl, Config.Retries)
    , ErrorStatter(DrillHole(Config, "err"))
    , PoolStatter(DrillHole(Config, "pool_err"))
    , Stats2xx(DrillHole(Config, "2xx"))
    , Stats4xx(DrillHole(Config, "4xx"))
    , Stats5xx(DrillHole(Config, "5xx"))
    , TvmErrorStatter(DrillHole(Config, "tvm_err"))
    , TimingStatter(DrillHistHole(Config, "time"))
    , TvmClient(Config.TvmServiceName ? std::move(args.TvmClient) : nullptr)
    , EncoderCtor(TCompressionCodecFactory::Instance().FindEncoder(Config.Compression)) {
    Y_VERIFY(!Config.Compression && !EncoderCtor || Config.Compression && EncoderCtor);
}

TMaybe<NCurl::TError> TRequestClient::Perform(TRequest request) const noexcept {
    return Perform(NCurl::TSimpleArtifacts{}, std::move(request));
}

TMaybe<NCurl::TError> TRequestClient::Perform(NCurl::TSimpleArtifacts&& artifacts, TRequest request) const {
    return Perform(artifacts, std::move(request));
}

TMaybe<NCurl::TError> TRequestClient::Perform(NCurl::TSimpleArtifacts& artifacts, TRequest request) const {
    NCurl::TRequestContext requestContext;

    requestContext
        .SetConnectTimeout(Config.ConnectTimeout)
        .SetRequestTimeout(Config.RequestTimeout)
        .SetPort(Config.Port);
    if (request.Request) {
        requestContext.SetHost(TStringBuilder{} << Config.Host << '/' << *request.Request);
    } else {
        requestContext.SetHost(Config.Host);
    }

    if (request.Data) {
        if(EncoderCtor) {
            TString encoded;
            {
                TStringOutput toEncode(encoded);
                auto encodedStream = (*EncoderCtor)(&toEncode);
                encodedStream->Write(*request.Data);
            }
            requestContext.SetPostData(std::move(encoded));
            requestContext.AddHeader("Content-Encoding: " + Config.Compression);
        } else {
            requestContext.SetPostData(std::move(*request.Data));
        }
    }

    if (request.Headers) {
        requestContext.AddHeaders(*request.Headers);
    }

    if (TvmClient) {
        try {
            requestContext.AddHeader("X-Ya-Service-Ticket: " + TvmClient->GetServiceTicketFor(Config.TvmServiceName));
        } catch (...) {
            if (TvmErrorStatter)
                TvmErrorStatter->PushSignal(1);
            return NCurl::TError() << "tvm error: " << CurrentExceptionMessageWithBt();
        }
    }
    {
        const auto now = Now();
        Y_DEFER {
            if (TimingStatter)
                TimingStatter->PushSignal((Now() - now).MilliSeconds());
        };

        NCurl::TPoolItemHolder holder;

        if (!Pool->get(PoolTraits, holder)) {
            if (PoolStatter)
                PoolStatter->PushSignal(1);
            return NCurl::TError() << "pool timeout";
        }

        NCurl::TCurl& curl = holder.Get();

        curl.Setup(std::move(requestContext));

        if (auto error = curl.Perform(artifacts)) {
            PushError();
            return error;
        }
    }

    if (artifacts.code < 300) {
        if (Stats2xx)
            Stats2xx->PushSignal(1);
        return Nothing();
    } else {
        if (artifacts.code < 500) {
            if (Stats4xx)
                Stats4xx->PushSignal(1);
        } else {
            if (Stats5xx)
                Stats5xx->PushSignal(1);
        }
        return NCurl::TError() << TStringBuilder{} << Config.Host << '/' << *request.Request << ':' << int(artifacts.code) << ' ' << artifacts.firstLine << ';' << artifacts.body.Str();
    }
}

void TRequestClient::PushError() const noexcept {
    if (ErrorStatter)
        ErrorStatter->PushSignal(1);
}

NUnistat::IHolePtr TRequestClient::DrillHole(const TClientConfig& config, const TStringBuf name) {
    if (config.YasmPrefix)
        return TUnistat::Instance().DrillFloatHole(TStringBuilder{} << config.YasmPrefix << '_' << name, "summ", NUnistat::TPriority{0});
    else
        return nullptr;
}

NUnistat::IHolePtr TRequestClient::DrillHistHole(const TClientConfig& config, const TStringBuf name) {
    if (!config.YasmPrefix)
        return nullptr;

    NUnistat::TIntervals intervals;
    for (size_t i = 0; i < size_t((config.ConnectTimeout + config.RequestTimeout).MilliSeconds()) + 10; i += 10)
        intervals.emplace_back(i);

    return TUnistat::Instance().DrillHistogramHole(TStringBuilder{} << config.YasmPrefix << '_' << name, "hgram", NUnistat::TPriority{0}, intervals);
}
