#include <mail/so/spamstop/tools/so-common/sobase64.h>
#include <mail/so/spamstop/tools/so-common/parsers.h>
#include <util/string/ascii.h>
#include <util/string/reverse.h>
#include "kshingle.h"

TString TComplSpamTypeToStroka(TComplSpamType spam) {
    TString res = "";

    switch (spam) {
        case CS_HAM:
            res = SPAMTYPETEXT_HAM1;
            break;
        case CS_SPAM:
            res = SPAMTYPETEXT_SPAM1;
            break;
        case CS_UNSUBSCRIBE:
            res = SPAMTYPETEXT_UNSUBSCRIBE1;
            break;
        case CS_UNKNOWN:
            res = "unknown";
            break;
        default:
            res = SPAMTYPETEXT_SPAM1;
    };

    return res;
}

TComplSpamType StrokaToTComplSpamType(const TString& str) {
    TComplSpamType res;

    if ((str == SPAMTYPETEXT_HAM1) || (str == "0"))
        res = CS_HAM;
    else if ((str == SPAMTYPETEXT_SPAM1) || (str == "1"))
        res = CS_SPAM;
    else if ((str == SPAMTYPETEXT_UNSUBSCRIBE1) || (str == "2"))
        res = CS_UNSUBSCRIBE;
    else
        res = CS_UNKNOWN;

    return res;
}

bool TComplSpamTypeToSpam(TComplSpamType spam) {
    bool res = false;

    if (spam == CS_HAM)
        res = false;
    else
        res = true;

    return res;
}

TString TComplSpamTypeToSpamS(TComplSpamType spam) {
    TString res = "";

    if (spam == CS_HAM)
        res = "ham";
    else
        res = "spam";

    return res;
}

//***************************************************************************************
//                      TEMailFunction - класс для работы с email адресами
//***************************************************************************************

bool TEMailFunction::IsBorderSymbol(unsigned char symb, bool domenname) const {
    if (domenname)
        return symboltable_domenname[symb];
    else
        return symboltable[symb];
}

bool TEMailFunction::IsBorderSymbolSoft(unsigned char symb, bool domenname) const {
    if (domenname)
        return symboltable_domenname_soft[symb];
    else
        return symboltable_soft[symb];
}

TStringBuf TEMailFunction::GetFirstEMailAddressA(const TStringBuf& str, TType type) const {
    TStringBuf res;
    ptrdiff_t count = 0;
    long maxcount = 0;
    char symb = 0x00;
    size_t pfirst = TStringBuf::npos;
    size_t pend = TStringBuf::npos;
    size_t lastpoint = TStringBuf::npos;
    long lsize = 0;
    ptrdiff_t rsize = 0;
    bool badpoint = false;

    const auto isBordeNotrDomain = std::bind(type == TBASE ? &TEMailFunction::IsBorderSymbol : &TEMailFunction::IsBorderSymbolSoft, this, std::placeholders::_1, false);
    const auto isBorderDomain = std::bind(type == TBASE ? &TEMailFunction::IsBorderSymbol : &TEMailFunction::IsBorderSymbolSoft, this, std::placeholders::_1, true);

    if (str) {
        size_t bstr = 0;
        while (true) {
            size_t p = str.find("@", bstr);
            if (p != TStringBuf::npos) {
                pfirst = TStringBuf::npos;
                pend = TStringBuf::npos;
                lastpoint = TStringBuf::npos;
                lsize = 0;
                rsize = 0;
                badpoint = false;
                //left border
                count = p;
                if (count > 0) {
                    for (long i = 1; i <= count; i++) {
                        symb = str[p - i];
                        if (isBordeNotrDomain(symb)) {
                            pfirst = p - i + 1;
                            break;
                        }
                    }
                }
                if (pfirst == TStringBuf::npos)
                    pfirst = 0;
                lsize = p - pfirst;
                //right border
                maxcount = str.size() - p - 1;
                count = 0;
                while (count < maxcount) {
                    count++;
                    symb = str[p + count];
                    if (symb == '.') {
                        lastpoint = p + count;
                        if (count == 1)
                            badpoint = true;
                    }
                    if (isBorderDomain(symb)) {
                        pend = p + count;
                        break;
                    }
                }
                if (pend == TStringBuf::npos)
                    pend = str.size();
                if ((lastpoint != TStringBuf::npos) && ((pend - lastpoint) < 3))
                    badpoint = true;
                rsize = pend - p;
                //create address
                count = pend - pfirst;
                if ((count > 0) && (lastpoint != TStringBuf::npos) && (lsize > 0) && (rsize > 0) && (!badpoint)) {
                    res = str.substr(pfirst, static_cast<size_t>(count));
                    break;
                }
            } else
                break;
            bstr = p + 1;
        }
    }

    return res;
}

TStringBuf TEMailFunction::GetFirstEMailAddressWitchMailtoA(const TStringBuf& str, TType type) const {
    TStringBuf res;
    TString srchstr = "mailto:";
    bool fend = false;

    if (str) {
        size_t pbegin = str.find(srchstr);
        while (pbegin != TStringBuf::npos) {
            size_t pend = str.find(srchstr, pbegin + 1);
            if (pend == TStringBuf::npos) {
                pend = str.size();
                fend = true;
            }
            res = GetFirstEMailAddressA(str.substr(pbegin + srchstr.length(), pend - pbegin - srchstr.length()), type);
            if (!res.empty())
                break;

            if (!fend) {
                pbegin = str.find(srchstr, pend + 1);
            } else
                break;
        }
    }

    return res;
}

TStringBuf TEMailFunction::GetFirstEMailAddressWitchMailto(const TStringBuf& str) const {
    return GetFirstEMailAddressWitchMailtoA(str, TBASE);
}

TStringBuf TEMailFunction::GetFirstEMailAddress(const TStringBuf& str) const {
    return GetFirstEMailAddressA(str, TBASE);
}

TStringBuf TEMailFunction::GetFirstEMailAddressSoft(const TStringBuf& str) const {
    return GetFirstEMailAddressA(str, TSOFT);
}

//***************************************************************************************
//                                Functions
//***************************************************************************************

TString TComplaintTypeToText(TComplaintType value) {
    TString res = "";

    switch (value) {
        case KCFROM:
            res = "CFROM";
            break;
        case KCFROMDOMEN:
            res = "CFROMDOMEN";
            break;
        case KCRETURNPATH:
            res = "CRETURNPATH";
            break;
        case KCMESS:
            res = "CMESS";
            break;
        case KCSUBJECT:
            res = "CSUBJECT";
            break;
        case KCBEENSENDER:
            res = "CBEENSENDER";
            break;
        case KCLIST:
            res = "CLIST";
            break;
        case KCPSNDR:
            res = "CPSNDR";
            break;
        case KCPSNDR_NOFROM:
            res = "CPSNDR_NOFROM";
            break;
        case KCUNKNOWN:
            res = "CUNKNOWN";
            break;
        default:
            res = "CUNKNOWN";
    };

    return res;
};

ui8 TComplaintTypeToInt(TComplaintType value) {
    ui8 res = 7;

    switch (value) {
        case KCFROM:
            res = 0;
            break;
        case KCFROMDOMEN:
            res = 1;
            break;
        case KCRETURNPATH:
            res = 2;
            break;
        case KCMESS:
            res = 3;
            break;
        case KCSUBJECT:
            res = 4;
            break;
        case KCBEENSENDER:
            res = 5;
            break;
        case KCLIST:
            res = 6;
            break;
        case KCPSNDR:
            res = 8;
            break;
        case KCPSNDR_NOFROM:
            res = 9;
            break;
        case KCUNKNOWN:
            res = 7;
            break;
        default:
            res = 7;
    };

    return res;
}

TComplaintType IntToTComplaintType(ui8 value) {
    TComplaintType res = KCUNKNOWN;

    switch (value) {
        case 0:
            res = KCFROM;
            break;
        case 1:
            res = KCFROMDOMEN;
            break;
        case 2:
            res = KCRETURNPATH;
            break;
        case 3:
            res = KCMESS;
            break;
        case 4:
            res = KCSUBJECT;
            break;
        case 5:
            res = KCBEENSENDER;
            break;
        case 6:
            res = KCLIST;
            break;
        case 8:
            res = KCPSNDR;
            break;
        case 9:
            res = KCPSNDR_NOFROM;
            break;
        case 7:
            res = KCUNKNOWN;
            break;
        default:
            res = KCUNKNOWN;
    };

    return res;
}

TString TStatTypeToText(TStatType value) {
    TString res = "";

    switch (value) {
        case KSFROM:
            res = "SFROM";
            break;
        case KSBEENSENDER:
            res = "SBEENSENDER";
            break;
        case KSLIST:
            res = "SLIST";
            break;
        case KSUNKNOWN:
            res = "SUNKNOWN";
            break;
        default:
            res = "SUNKNOWN";
    };

    return res;
}

ui8 TStatTypeToInt(TStatType value) {
    ui8 res = 3;

    switch (value) {
        case KSFROM:
            res = 0;
            break;
        case KSBEENSENDER:
            res = 1;
            break;
        case KSLIST:
            res = 2;
            break;
        case KSUNKNOWN:
            res = 3;
            break;
        default:
            res = 3;
    };

    return res;
}

TStatType IntToTStatType(ui8 value) {
    TStatType res = KSUNKNOWN;

    switch (value) {
        case 0:
            res = KSFROM;
            break;
        case 1:
            res = KSBEENSENDER;
            break;
        case 2:
            res = KSLIST;
            break;
        case 3:
            res = KSUNKNOWN;
            break;
        default:
            res = KSUNKNOWN;
    };

    return res;
}

TString KSTrim(const TString& s) {
    TString res = "", stemp = s;
    int nb = 0, ne = 0, nbc = 0, nec = 0;
    TString::iterator it;
    int count = 0;
    unsigned char uc = 0;

    it = stemp.begin();
    while (it != stemp.end()) {
        nb++;
        memcpy(&uc, it, sizeof(uc));
        if (uc > ' ')
            break;
        ++it;
    }

    ReverseInPlace(stemp);
    it = stemp.begin();
    while (it != stemp.end()) {
        ne++;
        memcpy(&uc, it, sizeof(uc));
        if (uc > ' ')
            break;
        ++it;
    }

    nbc = nb > 0 ? nb - 1 : 0;
    nec = ne > 0 ? ne - 1 : 0;
    count = s.size() - nbc - nec;
    if (count > 0)
        res = s.substr(nbc, count);

    return res;
}

ui32 SGetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size) {
    long pos = -1;
    ui32 res = 0;
    ui8 dob = 0;

    for (ui32 i = 0; i < source_size; i++) {
        if ((source[i] == 0x0D) && ((i + 1) < source_size) && (source[i + 1] == 0x0A)) {
            pos = i;
            dob = 1;
            break;
        }
        if (source[i] == 0x0A) {
            pos = i;
            dob = 0;
            break;
        }
    }
    res = pos + 1 + dob;
    if (res == 0)
        res = source_size;
    if (res > 0) {
        memset(destination, 0, destination_size);
        if (res <= (destination_size - 1)) {
            memcpy(destination, source, res);
        } else {
            memcpy(destination, source, destination_size - 1);
        }
    }
    return res;
}

TString ParseSenderXBeenThereFields(const TStringBuf& from, const TStringBuf& sender, const TStringBuf& xbeenthere, const TStringBuf& deliveredto) {
    TString res;
    TStringBuf beensenderaddr = TEMailFunction{}.GetFirstEMailAddress(xbeenthere);
    if (beensenderaddr.empty())
        beensenderaddr = TEMailFunction{}.GetFirstEMailAddress(sender);

    if (!beensenderaddr.empty()) {
        if (!AsciiEqualsIgnoreCase(beensenderaddr, from))
            res = to_lower(TString{beensenderaddr});
    }

    if (res.empty() && deliveredto) {
        if (!AsciiEqualsIgnoreCase(deliveredto, from))
            res = to_lower(TString{deliveredto});
    }

    return res;
}

TString ParseListFields(const TStringBuf& from, const TStringBuf& beensender, const TStringBuf& deliveredto, const TFieldListStrings& list) {
    TString res = "";
    TStringBuf reslistaddr;
    bool nocmp = true;

    const TEMailFunction EMF{};
    if (list.list != nullptr)
        reslistaddr = EMF.GetFirstEMailAddressWitchMailto(list.list);
    if (reslistaddr.empty() && (list.listpost != nullptr))
        reslistaddr = EMF.GetFirstEMailAddressWitchMailto(list.listpost);
    if (reslistaddr.empty() && (list.listowner != nullptr))
        reslistaddr = EMF.GetFirstEMailAddressWitchMailto(list.listowner);
    if (reslistaddr.empty() && (list.listsubscribe != nullptr))
        reslistaddr = EMF.GetFirstEMailAddressWitchMailto(list.listsubscribe);

    if (reslistaddr.empty() && (list.list != nullptr))
        reslistaddr = EMF.GetFirstEMailAddress(list.list);
    if (reslistaddr.empty() && (list.listpost != nullptr))
        reslistaddr = EMF.GetFirstEMailAddress(list.listpost);
    if (reslistaddr.empty() && (list.listowner != nullptr))
        reslistaddr = EMF.GetFirstEMailAddress(list.listowner);
    if (reslistaddr.empty() && (list.listsubscribe != nullptr))
        reslistaddr = EMF.GetFirstEMailAddress(list.listsubscribe);

    if (reslistaddr.empty() && (list.listother != nullptr))
        reslistaddr = EMF.GetFirstEMailAddressWitchMailto(list.listother);
    if (reslistaddr.empty() && (list.listother != nullptr))
        reslistaddr = EMF.GetFirstEMailAddress(list.listother);

    if (!reslistaddr.empty()) {
        if(!AsciiEqualsIgnoreCase(reslistaddr, from) && !AsciiEqualsIgnoreCase(reslistaddr, beensender)) {
            nocmp = true;

            size_t p = beensender.find("@");
            if (p != TStringBuf::npos) {
                size_t count = p;
                if ((count > 0) && reslistaddr.length() >= count) {

                    if (AsciiEqualsIgnoreCase(beensender.substr(0, count), reslistaddr.substr(0, count)))
                        nocmp = false;
                }
            }
            if (nocmp)
                res = to_lower(TString{reslistaddr});
        }
    }

    if ((res.empty()) && (deliveredto != nullptr)) {
        if (!AsciiEqualsIgnoreCase(deliveredto, from) && !AsciiEqualsIgnoreCase(deliveredto, beensender))
            res = to_lower(TString{deliveredto});
    }

    return res;
}

TStringBuf GetFromDomen(const TStringBuf& from, TZoneDetector& TDC) {
    const size_t p = from.find('@');

    if (p != TStringBuf::npos) {
        return TDC.GetCommonZone(from.substr(p + 1));
    }

    return {};
}

TStringBuf GetFromDomenFull(const TStringBuf& from) {
    const size_t p = from.find('@');

    if (p != TStringBuf::npos) {
        return from.substr(p + 1);
    }

    return {};
}

//***************************************************************************************
//                                  Host reputation
//***************************************************************************************

THostShingle::THostShingle() {
    m_host = "";
    m_shingle_str = "";
    m_shingle = 0;
}

THostShingle::THostShingle(ui64 shingle) {
    char sshingle[32];

    m_shingle = shingle;
    m_host = "";
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);
}

THostShingle::THostShingle(const char* host) {
    char sshingle[32];
    TString stemp = "";

    m_shingle_str = "";
    m_shingle = 0;
    m_host = KSTrim(TString(host));
    to_lower_k(m_host);
    stemp = m_host;
    memset(sshingle, 0, sizeof(sshingle));
    calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
    sscanf(sshingle, "%" PRIx64, &m_shingle);
    m_shingle_str = TString(sshingle);
}

THostRequest::THostRequest() {
    ResClear();
    m_shingle = THostShingle();
    m_type = 0;
    m_banpattrn = false;
    m_surbl = false;
    m_value = KUNDEF;
}

THostRequest::THostRequest(THostShingle shingle, ui16 type, bool banpattrn, bool surbl, TShingleValue value) {
    ResClear();
    m_shingle = shingle;
    m_type = type;
    m_banpattrn = banpattrn;
    m_surbl = surbl;
    m_value = value;
}

THostRequest::THostRequest(ui64 shingle, ui16 type) {
    ResClear();
    m_shingle = THostShingle(shingle);
    m_type = type;
    m_banpattrn = false;
    m_surbl = false;
    m_value = KUNDEF;
}

THostRequest::THostRequest(const char* host, ui16 type, bool banpattrn, bool surbl, TShingleValue value) {
    ResClear();
    m_shingle = THostShingle(host);
    m_type = type;
    m_banpattrn = banpattrn;
    m_surbl = surbl;
    m_value = value;
}

void THostRequest::ResClear() {
    m_res_empty = true;
    m_res_firsttime = 0;
    m_res_lasttime = 0;
    m_res_todaycount = 0;
    m_res_banpattrn = 0;
    m_res_surbl = 0;
}

const char* THostRequest::sShingle() const {
    return m_shingle.Shingle_str().c_str();
}

ui32 THostRequest::ResFirsttimeElapsed() const {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= m_res_firsttime)
        res = currenttime - m_res_firsttime;

    return res;
}

ui32 THostRequest::ResLasttimeElapsed() const {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= m_res_lasttime)
        res = currenttime - m_res_lasttime;

    return res;
}

TString THostRequest::GetRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0, a3 = 0, a4 = 0, a5 = 0;

    a1 = m_shingle.Shingle();
    a2 = m_type;
    a3 = m_banpattrn;
    a4 = m_surbl;
    a5 = m_value;
    memset(buff, 0, sizeof(buff));
    snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%u-%u-%u", a1, a2, a3, a4, a5);

    return TString(buff);
}

bool THostRequest::ResSetFromText(ui64 shingle, ui16 type, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0, a5 = 0, a6 = 0, a7 = 0, a8 = 0, a9 = 0, a10 = 0, a11 = 0;
    int fres = 0;

    if ((shingle == m_shingle.Shingle()) && (type == m_type)) {
        m_res_empty = empty;
        if (true /*!m_res_empty*/) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u", &a1, &a2, &a3, &a4, &a5, &a6, &a7, &a8, &a9, &a10, &a11);
            if (fres == 11) {
                //история
                m_res_firsttime = a1;
                m_res_lasttime = a2;
                m_res_todaycount = a3;
                m_res_banpattrn = a4;
                m_res_surbl = a5;

                //сегодня
                m_res_todaydata_empty = (bool)a6;
                m_res_todaydata_ham = a7;
                m_res_todaydata_spam = a8;
                m_res_todaydata_malic = a9;
                m_res_todaydata_banpattrn = a10;
                m_res_todaydata_surbl = a11;

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

//***************************************************************************************
//                                     FromTo
//***************************************************************************************

TFromToShingle::TFromToShingle() {
    m_from = "";
    m_to = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = 0;
    m_userid_newuid = "";
    m_shingle_newuid_str = "";
    m_shingle_newuid = 0;
}

TFromToShingle::TFromToShingle(ui64 shingle) {
    char sshingle[32];

    m_from = "";
    m_to = "";

    m_userid = "";
    m_shingle_str = "";
    m_shingle = shingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);

    m_userid_newuid = "";
    m_shingle_newuid_str = "";
    m_shingle_newuid = 0;
}

TFromToShingle::TFromToShingle(const TStringBuf& from, const TStringBuf& userid, const TStringBuf& to) {
    char sshingle[32];
    TString stemp = "";

    m_to = KSTrim(TString(to));
    to_lower_k(m_to);
    m_from = KSTrim(TString(from));
    to_lower_k(m_from);

    m_shingle_str = "";
    m_shingle = 0;
    m_userid = KSTrim(TString(userid));
    to_lower_k(m_userid);
    stemp = m_from + "-" + m_userid;
    memset(sshingle, 0, sizeof(sshingle));
    calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
    sscanf(sshingle, "%" PRIx64, &m_shingle);
    m_shingle_str = TString(sshingle);

    m_userid_newuid = "";
    m_shingle_newuid_str = "";
    m_shingle_newuid = 0;
}

TFromToShingle::TFromToShingle(ui64 shingle, ui64 shingle_newuid) {
    char sshingle[32];

    m_from = "";
    m_to = "";

    m_userid = "";
    m_shingle_str = "";
    m_shingle = shingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);

    m_userid_newuid = "";
    m_shingle_newuid_str = "";
    m_shingle_newuid = shingle_newuid;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle_newuid);
    m_shingle_newuid_str = TString(sshingle);
}

TFromToShingle::TFromToShingle(const TStringBuf& from, const TUserID& id, const TStringBuf& to) {
    char sshingle[32];
    TString stemp = "";

    m_to = KSTrim(TString(to));
    to_lower_k(m_to);
    m_from = KSTrim(TString(from));
    to_lower_k(m_from);

    m_shingle_str = "";
    m_shingle = 0;
    m_userid = KSTrim(TString(id.GetSuid()));
    if (!m_userid.empty()) {
        to_lower_k(m_userid);
        stemp = m_from + "-" + m_userid;
        memset(sshingle, 0, sizeof(sshingle));
        calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
        sscanf(sshingle, "%" PRIx64, &m_shingle);
        m_shingle_str = TString(sshingle);
    }

    m_shingle_newuid_str = "";
    m_shingle_newuid = 0;
    m_userid_newuid = KSTrim(TString(id.GetNewUid()));
    if (!m_userid_newuid.empty()) {
        to_lower_k(m_userid_newuid);
        stemp = m_from + "-" + m_userid_newuid;
        memset(sshingle, 0, sizeof(sshingle));
        calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
        sscanf(sshingle, "%" PRIx64, &m_shingle_newuid);
        m_shingle_newuid_str = TString(sshingle);
    }
}

TFromToRequest::TFromToRequest() {
    ResClear();
    m_shingle = TFromToShingle();
    m_value = KUNKNOWN;
}

TFromToRequest::TFromToRequest(ui64 shingle, TKMailType value) {
    ResClear();
    m_shingle = TFromToShingle(shingle);
    m_value = value;
}

TFromToRequest::TFromToRequest(ui64 shingle, ui64 shingle_newuid, TKMailType value) {
    ResClear();
    m_shingle = TFromToShingle(shingle, shingle_newuid);
    m_value = value;
}

TFromToRequest::TFromToRequest(TFromToShingle shingle, TKMailType value) {
    m_shingle = std::move(shingle);
    m_value = value;
    ResClear();
}

TFromToRequest::TFromToRequest(const char* from, const char* userid, TKMailType value, const char* to) {
    m_shingle = TFromToShingle(from, userid, to);
    m_value = value;
    ResClear();
}

TFromToRequest::TFromToRequest(const char* from, const TUserID& id, TKMailType value, const char* to) {
    m_shingle = TFromToShingle(from, id, to);
    m_value = value;
    ResClear();
}

void TFromToRequest::ResClear() {
    m_res_empty = true;
    m_res_firsttime = 0;
    m_res_lasttime = 0;
    m_res_send = 0;
    m_res_accepted = 0;
}

const char* TFromToRequest::sShingle() const {
    return m_shingle.Shingle_str().c_str();
}

const char* TFromToRequest::sShingleNewUid() const {
    return m_shingle.ShingleNewUid_str().c_str();
}

ui32 TFromToRequest::ResFirsttimeElapsed() const {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= m_res_firsttime)
        res = currenttime - m_res_firsttime;

    return res;
}

ui32 TFromToRequest::ResLasttimeElapsed() const {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= m_res_lasttime)
        res = currenttime - m_res_lasttime;

    return res;
}

TString TFromToRequest::GetRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0;
    ui64 a3 = 0;

    a1 = m_shingle.Shingle();
    a2 = m_value;
    a3 = m_shingle.ShingleNewUid();

    memset(buff, 0, sizeof(buff));
    if (a3 == 0)
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);
    else
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u--%016" PRIx64 "-", a1, a2, a3);

    return TString(buff);
}

bool TFromToRequest::ResSetFromText(ui64 shingle, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0;
    int fres = 0;

    if (shingle == m_shingle.Shingle()) {
        m_res_empty = empty;
        if (!m_res_empty) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u", &a1, &a2, &a3, &a4);
            if (fres == 4) {
                m_res_firsttime = a1;
                m_res_lasttime = a2;
                m_res_send = a3;
                m_res_accepted = a4;

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

bool TFromToRequest::ResSetFromText2(ui64 shingle_newuid, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0;
    int fres = 0;

    if (shingle_newuid == m_shingle.ShingleNewUid()) {
        m_res_empty = empty;
        if (!m_res_empty) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u", &a1, &a2, &a3, &a4);
            if (fres == 4) {
                m_res_firsttime = a1;
                m_res_lasttime = a2;
                m_res_send = a3;
                m_res_accepted = a4;

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

TString TFromToRequest::PrintData() {
    TString res = "";
    char buff[256];

    snprintf(buff, sizeof(buff) - 1, "shingle=%s shingle_newuid=%s ft=%u lt=%u ...", sShingle(), sShingleNewUid(), ResFirsttime(), ResLasttime());

    if (ResEmpty())
        res = res + "PRINT FROMTO: empty (" + TString(buff) + ")";
    else
        res = res + "PRINT FROMTO: nonempty (" + TString(buff) + ")";

    return res;
}

TString TFromToRequest::PrintData2() {
    TString res = "";
    char buff[256];

    snprintf(buff, sizeof(buff) - 1, "shingle=%s shingle_newuid=%s ft=%u lt=%u send=%u accept=%u", sShingle(), sShingleNewUid(), ResFirsttime(), ResLasttime(), ResSend(), ResAccepted());

    if (ResEmpty())
        res = res + "PRINT FROMTO:   empty    (" + TString(buff) + ")";
    else
        res = res + "PRINT FROMTO:   nonempty (" + TString(buff) + ")";

    return res;
}

//***************************************************************************************
//                                  Complaints
//***************************************************************************************

TComplaintShingle::TComplaintShingle() {
    m_host = "";
    m_from = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = 0;
    m_new_userid = "";
    m_new_shingle_str = "";
    m_new_shingle = 0;
    m_type = KCUNKNOWN;
}

TComplaintShingle::TComplaintShingle(ui64 shingle, TComplaintType type) {
    char sshingle[32];

    m_host = "";
    m_from = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = shingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);
    m_new_userid = "";
    m_new_shingle_str = "";
    m_new_shingle = 0;
    m_type = type;
}

TComplaintShingle::TComplaintShingle(ui64 shingle, TComplaintType type, ui64 newshingle) {
    char sshingle[32];

    m_host = "";
    m_from = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = shingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);
    m_new_userid = "";
    m_new_shingle_str = "";
    m_new_shingle = newshingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_new_shingle);
    m_new_shingle_str = TString(sshingle);
    m_type = type;
}

TComplaintShingle::TComplaintShingle(ui64 shingle, TComplaintType type, const TString& suid, ui64 newshingle, const TString& uid) {
    char sshingle[32];

    m_host = "";
    m_from = "";
    m_userid = suid;
    m_shingle_str = "";
    m_shingle = shingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);
    m_new_userid = uid;
    m_new_shingle_str = "";
    m_new_shingle = newshingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_new_shingle);
    m_new_shingle_str = TString(sshingle);
    m_type = type;
}

void TComplaintShingle::Init(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& userid, TComplaintType type, const TStringBuf& newuserid) {
    char sshingle[32];
    TString stemp = "";
    TString stype = "";

    m_type = type;
    switch (m_type) {
        default:
            break;
        case KCFROM:
            stype = "from-";
            break;
        case KCFROMDOMEN:
            stype = "fromdomen-";
            break;
        case KCRETURNPATH:
            stype = "returnpath-";
            break;
        case KCMESS:
            stype = "mess-";
            break;
        case KCSUBJECT:
            stype = "subj-";
            break;
        case KCBEENSENDER:
            stype = "beensender-";
            break;
        case KCLIST:
            stype = "list-";
            break;
        case KCPSNDR:
            stype = "psndr-";
            break;
        case KCPSNDR_NOFROM:
            stype = "psndrnofrom-";
            break;
    };
    m_shingle_str = "";
    m_shingle = 0;
    m_host = KSTrim(GetHost(TDC, TString(host), TString(ip)));
    to_lower_k(m_host);
    if (m_type == KCMESS)
        m_from = GetHost2(TDC, KSTrim(TString(from)));
    else if (m_type == KCPSNDR_NOFROM)
        m_from = "nofrom";
    else
        m_from = KSTrim(TString(from));
    to_lower_k(m_from);
    if (m_from.length() > 0) {
        if ((m_type == KCMESS) && IsLocalhost(m_from)) {
            m_shingle = 0;
            m_shingle_str = "";
        } else {
            m_userid = KSTrim(TString(userid));
            if ((!m_userid.empty()) && (m_userid != "0")) {
                to_lower_k(m_userid);
                stemp = stype + m_from + "-" + m_userid + "-" + m_host;
                memset(sshingle, 0, sizeof(sshingle));
                calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
                sscanf(sshingle, "%" PRIx64, &m_shingle);
                m_shingle_str = TString(sshingle);
            }

            m_new_userid = KSTrim(TString(newuserid));
            if ((!m_new_userid.empty()) && (m_new_userid != "0")) {
                to_lower_k(m_new_userid);
                stemp = stype + m_from + "-" + m_new_userid + "-" + m_host;
                memset(sshingle, 0, sizeof(sshingle));
                calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
                sscanf(sshingle, "%" PRIx64, &m_new_shingle);
                m_new_shingle_str = TString(sshingle);
            }
        }
    }
}

TComplaintShingle::TComplaintShingle(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& userid, TComplaintType type, const TStringBuf& newuserid) {
    m_host = "";
    m_from = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = 0;
    m_new_userid = "";
    m_new_shingle_str = "";
    m_new_shingle = 0;
    m_type = KCUNKNOWN;

    Init(TDC, host, ip, from, userid, type, newuserid);
}

TComplaintShingle::TComplaintShingle(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& sender, const TStringBuf& xbeenwhere, const TStringBuf& deliveredto, const TStringBuf& userid, const TStringBuf& newuserid) {
    m_host = "";
    m_from = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = 0;
    m_new_userid = "";
    m_new_shingle_str = "";
    m_new_shingle = 0;
    m_type = KCUNKNOWN;

    Init(TDC, host, ip, (ParseSenderXBeenThereFields(from, sender, xbeenwhere, deliveredto)), userid, KCBEENSENDER, newuserid);
}

TComplaintShingle::TComplaintShingle(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& beenthere, const TStringBuf& deliveredto, TFieldListStrings& list, const TStringBuf& userid, const TStringBuf& newuserid) {
    m_host = "";
    m_from = "";
    m_userid = "";
    m_shingle_str = "";
    m_shingle = 0;
    m_new_userid = "";
    m_new_shingle_str = "";
    m_new_shingle = 0;
    m_type = KCUNKNOWN;

    Init(TDC, host, ip, (ParseListFields(from, beenthere, deliveredto, list)), userid, KCLIST, newuserid);
}

TString TComplaintShingle::GetHost(TZoneDetector& TDC, const TString& host, const TString& ip) {
    TString res = "";
    size_t n = 0;
    //    size_t n1 = 0;
    size_t pointcount = 0;
    TKIPv6 ipv;

    n = host.find(".");
    if ((!host.empty()) && (n < (host.length() - 1))) {
        res = TDC.GetCommonZone(host);
        res = RemoveDigitsFromHost(res);
    } else {
        ipv = TKIPv6(ip.c_str());

        if (!ipv.Undefined()) {
            if (ipv.IsIPv4()) {
                pointcount = 0;
                for (size_t i = 0; i < ip.length(); i++) {
                    if (ip[i] == '.')
                        pointcount++;
                    if (pointcount == 3) {
                        res = ip.substr(0, i + 1) + "0";
                        break;
                    }
                }
            } else {
                res = ipv.toStroka();
            }
        }
    }
    return res;
}

TString TComplaintShingle::GetHost2(TZoneDetector& TDC, const TString& host) {
    TString res = "";
    size_t n = 0;
    //    size_t n1 = 0;
    //    size_t pointcount = 0;

    n = host.find(".");
    if ((!host.empty()) && (n < (host.length() - 1))) {
        res = TDC.GetCommonZone(host.c_str());
        res = RemoveDigitsFromHost(res);

    } else {
        res = host;
    }
    return res;
}

TString TComplaintShingle::RemoveDigitsFromHost(const TString& host) {
    TString res = "";
    TString stemp = "";
    const char* p = nullptr;
    ssize_t count = 0;
    char uc = 0;
    bool ok_write = false;
    bool was_symb = false;

    res = host;
    stemp = host;
    ReverseInPlace(stemp);
    p = strchr(stemp.c_str(), '.');
    if (p != nullptr) {
        count = stemp.length() - (p - stemp.c_str());
        if (count > 0) {
            res = "";
            for (int i = 1; i < count; i++) {
                uc = *(p + i);
                if ((uc >= '0') && (uc <= '9') && (!was_symb)) {
                    if (!ok_write) {
                        res = res + "_";
                        ok_write = true;
                    }
                } else {
                    was_symb = true;
                    res = res + TString(uc);
                }
            }
            res = TString(stemp.c_str(), p - stemp.c_str() + 1) + res;
            ReverseInPlace(res);
        }
    }

    return res;
}

bool TComplaintShingle::IsLocalhost(const TString& host) {
    bool res = false;

    if (((host.length() >= 4) && (strncmp(host.c_str(), "127.", 4) == 0)) ||
        ((host.length() >= 8) && (strncmp(host.c_str(), "192.168.", 8) == 0)) ||
        ((host.length() >= 7) && (strncmp(host.c_str(), "172.16.", 7) == 0)) ||
        ((host.length() >= 3) && (strncmp(host.c_str(), "10.", 3) == 0)) ||
        ((host.length() >= 8) && (strncmp(host.c_str(), "169.254.", 8) == 0)) ||
        ((host.length() >= 2) && (strncmp(host.c_str(), "0.", 2) == 0)) ||
        ((host.length() == 9) && (host == "localhost")))
        res = true;

    return res;
}

TComplaintRequest::TComplaintRequest() {
    m_shingle = TComplaintShingle();
    m_value = KUNKNOWN;
    ResClear();
}

TComplaintRequest::TComplaintRequest(ui64 shingle, TComplaintType type, TKMailType value) {
    m_shingle = TComplaintShingle(shingle, type);
    m_value = value;
    ResClear();
}

TComplaintRequest::TComplaintRequest(ui64 shingle, ui64 uidnew_shingle, TComplaintType type, TKMailType value) {
    m_shingle = TComplaintShingle(shingle, type, uidnew_shingle);
    m_value = value;
    ResClear();
}

TComplaintRequest::TComplaintRequest(TComplaintShingle shingle, TKMailType value) {
    m_shingle = std::move(shingle);
    m_value = value;
    ResClear();
}

TComplaintRequest::TComplaintRequest(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& userid, TKMailType value, TComplaintType type, const TStringBuf& newuserid) {
    m_shingle = TComplaintShingle(TDC, host, ip, from, userid, type, newuserid);
    m_value = value;
    ResClear();
}

TComplaintRequest::TComplaintRequest(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& sender, const TStringBuf& xbeenthere, const TStringBuf& deliveredto, const TStringBuf& userid, TKMailType value, const TStringBuf& newuserid) {
    m_shingle = TComplaintShingle(TDC, host, ip, from, sender, xbeenthere, deliveredto, userid, newuserid);
    m_value = value;
    ResClear();
}

TComplaintRequest::TComplaintRequest(TZoneDetector& TDC, const TStringBuf& host, const TStringBuf& ip, const TStringBuf& from, const TStringBuf& beenthere, const TStringBuf& deliveredto, TFieldListStrings& list, const TStringBuf& userid, TKMailType value, const TStringBuf& newuserid) {
    m_shingle = TComplaintShingle(TDC, host, ip, from, beenthere, deliveredto, list, userid, newuserid);
    m_value = value;
    ResClear();
}

void TComplaintRequest::ResClear() {
    m_res_empty = true;
    m_res_firsttime = 0;
    m_res_lasttime = 0;
    m_res_complaint_ham = 0;
    m_res_complaint_spam = 0;
    m_res_ham = 0;
    m_res_spam = 0;
    m_res_history = 0;
    m_res_replace_to_ham = 0;
    m_res_replace_to_spam = 0;
    m_res_lasttime_change = 0;
}

const char* TComplaintRequest::sShingle() const {
    return m_shingle.Shingle_str().c_str();
}

const char* TComplaintRequest::sShingleNew() {
    return m_shingle.NewShingle_str().c_str();
}

ui32 TComplaintRequest::ResFirsttime() {
    return m_res_firsttime & 0x3FFF; //младшие 14 бит
}

ui32 TComplaintRequest::ResFirsttimeElapsed() {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= ResFirsttime())
        res = currenttime - ResFirsttime();

    return res;
}

ui32 TComplaintRequest::ResLasttime() {
    ui32 res = 0;

    res = m_res_lasttime & 0x3FFF; //младшие 14 бит

    return res;
}

ui32 TComplaintRequest::ResLasttimeElapsed() {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= ResLasttime())
        res = currenttime - ResLasttime();

    return res;
}

ui32 TComplaintRequest::ResHam() {
    ui32 res = 0;

    res = m_res_ham;
    if (((m_value == KOUTMAILHAM) || (m_value == KINMAILHAM)) && (res > 0))
        res--;

    return res;
}

ui32 TComplaintRequest::ResSpam() {
    ui32 res = 0;

    res = m_res_spam;
    if (((m_value == KOUTMAILSPAM) || (m_value == KINMAILSPAM)) && (res > 0))
        res--;

    return res;
}

TString TComplaintRequest::GetRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0;
    TString a3 = "";
    ui64 a4 = 0;
    TString a5 = "";

    a1 = m_shingle.Shingle();
    a2 = TPackShingleType(m_shingle.Type(), (int)m_value).PackValue();
    memset(buff, 0, sizeof(buff));
    if (a4 != 0)
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s-%016" PRIx64 "-%s", a1, a2, a3.c_str(), a4, a5.c_str());
    else
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);

    return TString(buff);
}

TString TComplaintRequest::GetRequestNew() const {
    char buff[256];
    ui64 a1 = 0;
    ui32 a2 = 0;
    TString a3 = "";
    ui64 a4 = 0;
    TString a5 = "";

    a1 = m_shingle.Shingle();
    a2 = TPackShingleType(m_shingle.Type(), (int)m_value).PackValue();
    a3 = TString(m_shingle.Userid());
    a4 = m_shingle.NewShingle();
    a5 = TString(m_shingle.NewUserid());
    memset(buff, 0, sizeof(buff));
    if (a4 != 0)
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s-%016" PRIx64 "-%s", a1, a2, a3.c_str(), a4, a5.c_str());
    else
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s", a1, a2, a3.c_str());

    return TString(buff);
}

TString TComplaintRequest::GetCorrectRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0;

    a1 = m_shingle.Shingle();
    memset(buff, 0, sizeof(buff));
    if (m_value == KINMAILHAM) {
        a2 = TPackShingleType(m_shingle.Type(), (int)KINMAILSPAM).PackValue();
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);
    } else if (m_value == KINMAILSPAM) {
        a2 = TPackShingleType(m_shingle.Type(), (int)KINMAILHAM).PackValue();
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);
    }

    return TString(buff);
}

TString TComplaintRequest::GetCorrectRequestNew() const {
    char buff[256];
    ui64 a1 = 0;
    ui32 a2 = 0;
    TString a3 = "";
    ui64 a4 = 0;
    TString a5 = "";

    a1 = m_shingle.Shingle();
    a3 = TString(m_shingle.Userid());
    a4 = m_shingle.NewShingle();
    a5 = TString(m_shingle.NewUserid());
    memset(buff, 0, sizeof(buff));
    if (a4 != 0) {
        if (m_value == KINMAILHAM) {
            a2 = TPackShingleType(m_shingle.Type(), (int)KINMAILSPAM).PackValue();
            snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s-%016" PRIx64 "-%s", a1, a2, a3.c_str(), a4, a5.c_str());
        } else if (m_value == KINMAILSPAM) {
            a2 = TPackShingleType(m_shingle.Type(), (int)KINMAILHAM).PackValue();
            snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s-%016" PRIx64 "-%s", a1, a2, a3.c_str(), a4, a5.c_str());
        }

    } else {
        if (m_value == KINMAILHAM) {
            a2 = TPackShingleType(m_shingle.Type(), (int)KINMAILSPAM).PackValue();
            snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s", a1, a2, a3.c_str());
        } else if (m_value == KINMAILSPAM) {
            a2 = TPackShingleType(m_shingle.Type(), (int)KINMAILHAM).PackValue();
            snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u-%s", a1, a2, a3.c_str());
        }
    }

    return TString(buff);
}

/*void TComplaintRequest::ResSet(bool empty, ui16 firsttime, ui16 lasttime, ui8 complaint_ham, ui8 complaint_spam, ui8 ham, ui8 spam, ui32 history, ui8 replace_to_ham, ui8 replace_to_spam, ui16 lasttimechange)
{
   m_res_empty           = empty;
   m_res_firsttime       = firsttime;
   m_res_lasttime        = lasttime;
   m_res_complaint_ham   = complaint_ham;
   m_res_complaint_spam  = complaint_spam;
   m_res_ham             = ham;
   m_res_spam            = spam;
   m_res_history         = history;
   m_res_replace_to_ham  = replace_to_ham;
   m_res_replace_to_spam = replace_to_spam;
   m_res_lasttime_change = lasttimechange;
}*/

bool TComplaintRequest::ResSetFromText(ui64 shingle, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0, a5 = 0, a6 = 0, a7 = 0, a8 = 0, a9 = 0, a10 = 0;
    int fres = 0;
    //    int dbga = 0;

    //    if ((shingle == 0xbc17bb52e0c2fd3b) || (m_shingle.Shingle() == 0xbc17bb52e0c2fd3b) || (m_shingle.NewShingle() == 0xbc17bb52e0c2fd3b))
    //        dbga = dbga;

    if (shingle == m_shingle.Shingle()) {
        m_res_empty = empty;
        if (!m_res_empty) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u-%u-%u-%u-%u-%u-%u", &a1, &a2, &a3, &a4, &a5, &a6, &a7, &a8, &a9, &a10);
            if (fres == 10) {
                m_res_firsttime = a1;
                m_res_lasttime = a2;
                m_res_complaint_ham = a3;
                m_res_complaint_spam = a4;
                m_res_ham = a5;
                m_res_spam = a6;
                m_res_history = a7;
                m_res_replace_to_ham = a8;
                m_res_replace_to_spam = a9;
                m_res_lasttime_change = a10;

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

bool TComplaintRequest::ResSetFromText2(ui64 shingle_newuid, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0, a5 = 0, a6 = 0, a7 = 0, a8 = 0, a9 = 0, a10 = 0;
    int fres = 0;

    if (shingle_newuid == m_shingle.NewShingle()) {
        m_res_empty = empty;
        if (!m_res_empty) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u-%u-%u-%u-%u-%u-%u", &a1, &a2, &a3, &a4, &a5, &a6, &a7, &a8, &a9, &a10);
            if (fres == 10) {
                m_res_firsttime = a1;
                m_res_lasttime = a2;
                m_res_complaint_ham = a3;
                m_res_complaint_spam = a4;
                m_res_ham = a5;
                m_res_spam = a6;
                m_res_history = a7;
                m_res_replace_to_ham = a8;
                m_res_replace_to_spam = a9;
                m_res_lasttime_change = a10;

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

TString TComplaintRequest::PrintData() {
    TString res = "";
    char buff[256];
    TString stype = "";

    stype = TComplaintTypeToText(Type());
    snprintf(buff, sizeof(buff) - 1, "type=%s shingle=%s ft=%u(%u) lt=%u(%u) ...", stype.c_str(), sShingle(), ResFirsttime(), ResFullFirstTime(), ResLasttime(), ResFullLasttime());

    if (ResEmpty())
        res = res + "PRINT COMPL: empty (" + TString(buff) + ")";
    else
        res = res + "PRINT COMPL: nonempty (" + TString(buff) + ")";

    return res;
}

TString TComplaintRequest::PrintData2() {
    TString res = "";
    char buff[1024];
    TString stype = "";

    stype = TComplaintTypeToText(Type());
    snprintf(buff, sizeof(buff) - 1, "type=%11s shingle=%s(suid=%s) shinglenew=%s(uid=%s) ft=%u lt=%u cmpl_ham=%u cmpl_spam=%u ham=%u spam=%u history=%u rt_ham=%u rt_spam=%u lt_chng=%u pr_a=%d pr_m=%d pr_u=%d", stype.c_str(), sShingle(), Userid(), sShingleNew(), UseridNew(), ResFirsttime(), ResLasttime(), ResComplaintHam(), ResComplaintSpam(), ResHam(), ResSpam(), ResHistory(), ResReplaceToHam(), ResReplaceToSpam(), ResLasttimeChange(), (int)ResAutoAction(), (int)ResManualAction(), (int)ResUnsubscribe());

    if (ResEmpty())
        res = res + "PRINT COMPL:    empty    (" + TString(buff) + ")";
    else
        res = res + "PRINT COMPL:    nonempty (" + TString(buff) + ")";

    return res;
}

//***************************************************************************************
//                                  From statistik
//***************************************************************************************

TFromStatShingle::TFromStatShingle() {
    m_from = "";
    m_shingle_str = "";
    m_shingle = 0;
    m_type = KSUNKNOWN;
}

TFromStatShingle::TFromStatShingle(ui64 shingle, TStatType type) {
    char sshingle[32];

    m_from = "";
    m_shingle_str = "";
    m_shingle = shingle;
    memset(sshingle, 0, sizeof(sshingle));
    snprintf(sshingle, sizeof(sshingle) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(sshingle);
    m_type = type;
}

void TFromStatShingle::Init(const TStringBuf& from, TStatType m_typeA) {
    TString stype = "";

    m_type = m_typeA;
    switch (m_type) {
        default:
            break;
        case KSFROM:
            stype = "from-";
            break;
        case KSBEENSENDER:
            stype = "beensender-";
            break;
        case KSLIST:
            stype = "list-";
            break;
    };
    m_shingle_str = "";
    m_shingle = 0;
    m_from = KSTrim(TString(from));
    to_lower_k(m_from);

    if (m_from.length() > 0) {
        char sshingle[32];
        TString stemp = stype + m_from;
        memset(sshingle, 0, sizeof(sshingle));
        calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);

        sscanf(sshingle, "%" PRIx64, &m_shingle);
        m_shingle_str = TString(sshingle);
    }
}

TFromStatShingle::TFromStatShingle(const TStringBuf& from, TStatType type) {
    Init(from, type);
}

/*TFromStatShingle::TFromStatShingle(const char *from)
{
   Init(from, KSFROM);
}*/

TFromStatShingle::TFromStatShingle(const TStringBuf& from, const TStringBuf& sender, const TStringBuf& xbeenwhere, const TStringBuf& deliveredto) {
    Init((ParseSenderXBeenThereFields(from, sender, xbeenwhere, deliveredto)), KSBEENSENDER);
}

TFromStatShingle::TFromStatShingle(const TStringBuf& from, const TStringBuf& beenthere, const TStringBuf& deliveredto, TFieldListStrings& list) {
    Init((ParseListFields(from, beenthere, deliveredto, list)).c_str(), KSLIST);
}

TFromStatRequest::TFromStatRequest() {
    m_shingle = TFromStatShingle();
    m_value = KUNKNOWN;
    ResClear();
}

TFromStatRequest::TFromStatRequest(ui64 shingle, TStatType type, TKMailType value) {
    m_shingle = TFromStatShingle(shingle, type);
    m_value = value;
    ResClear();
}

TFromStatRequest::TFromStatRequest(const TStringBuf& from, TStatType type, TKMailType value) {
    m_shingle = TFromStatShingle(from, type);
    m_value = value;
    ResClear();
}

/*TFromStatRequest::TFromStatRequest(const char *from, TKMailType value)
{
   m_shingle = TFromStatShingle(from);
   m_value   = value;
   ResClear();
}*/

TFromStatRequest::TFromStatRequest(const TStringBuf& from, const TStringBuf& sender, const TStringBuf& xbeenwhere, const TStringBuf& deliveredto, TKMailType value) {
    m_shingle = TFromStatShingle(from, sender, xbeenwhere, deliveredto);
    m_value = value;
    ResClear();
}

TFromStatRequest::TFromStatRequest(const TStringBuf& from, const TStringBuf& beenthere, const TStringBuf& deliveredto, TFieldListStrings& list, TKMailType value) {
    m_shingle = TFromStatShingle(from, beenthere, deliveredto, list);
    m_value = value;
    ResClear();
}

void TFromStatRequest::ResClear() {
    m_res_empty = true;
    m_res_firsttime = 0;
    m_res_lasttime = 0;
    m_res_complaint_ham = 0;
    m_res_complaint_spam = 0;
    m_res_ham = 0;
    m_res_spam = 0;
    m_res_history = 0;
    m_res_replace_to_ham = 0;
    m_res_replace_to_spam = 0;
    m_res_uniquser_ham = 0;
    m_res_uniquser_spam = 0;
    m_res_uniquser = 0;
    m_res_lasttime_change = 0;
}

const char* TFromStatRequest::sShingle() const {
    return m_shingle.Shingle_str().c_str();
}

ui32 TFromStatRequest::ResFirsttimeElapsed() const {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= m_res_firsttime)
        res = currenttime - m_res_firsttime;

    return res;
}

ui32 TFromStatRequest::ResLasttimeElapsed() const {
    ui32 res = 0;
    ui32 currenttime = 0;

    currenttime = (ui32)(kday_t(time(nullptr)));
    if (currenttime >= m_res_lasttime)
        res = currenttime - m_res_lasttime;

    return res;
}

ui32 TFromStatRequest::ResHam() const {
    ui32 res = 0;

    res = m_res_ham;
    if (((m_value == KOUTMAILHAM) || (m_value == KINMAILHAM)) && (res > 0))
        res--;

    return res;
}

ui32 TFromStatRequest::ResSpam() const {
    ui32 res = 0;

    res = m_res_spam;
    if (((m_value == KOUTMAILSPAM) || (m_value == KINMAILSPAM)) && (res > 0))
        res--;

    return res;
}

TString TFromStatRequest::GetRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0;

    a1 = m_shingle.Shingle();
    a2 = TPackShingleType2(m_shingle.Type(), (int)m_value).PackValue();
    memset(buff, 0, sizeof(buff));
    snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);

    return TString(buff);
}

TString TFromStatRequest::GetCorrectRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0;

    a1 = m_shingle.Shingle();
    memset(buff, 0, sizeof(buff));
    if (m_value == KINMAILHAM) {
        a2 = TPackShingleType2(m_shingle.Type(), (int)KINMAILSPAM).PackValue();
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);
    } else if (m_value == KINMAILSPAM) {
        a2 = TPackShingleType2(m_shingle.Type(), (int)KINMAILHAM).PackValue();
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);
    }

    return TString(buff);
}

/*void TFromStatRequest::ResSet(bool empty, ui16 firsttime, ui16 lasttime, ui32 complaint_ham, ui32 complaint_spam, ui32 ham, ui32 spam, ui32 history, ui32 replace_to_ham, ui32 replace_to_spam, ui32 uniquser_ham, ui32 uniquser_spam, ui32 uniquser, ui16 lasttimechange)
{
   m_res_empty           = empty;
   m_res_firsttime       = firsttime;
   m_res_lasttime        = lasttime;
   m_res_complaint_ham   = complaint_ham;
   m_res_complaint_spam  = complaint_spam;
   m_res_ham             = ham;
   m_res_spam            = spam;
   m_res_history         = history;
   m_res_replace_to_ham  = replace_to_ham;
   m_res_replace_to_spam = replace_to_spam;
   m_res_uniquser_ham    = uniquser_ham;
   m_res_uniquser_spam   = uniquser_spam;
   m_res_uniquser        = uniquser;
   m_res_lasttime_change = lasttimechange;
}*/

bool TFromStatRequest::ResSetFromText(ui64 shingle, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0, a5 = 0, a6 = 0, a7 = 0, a8 = 0, a9 = 0, a10 = 0, a11 = 0, a12 = 0, a13 = 0;
    int fres = 0;

    if (shingle == m_shingle.Shingle()) {
        m_res_empty = empty;
        if (!m_res_empty) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u", &a1, &a2, &a3, &a4, &a5, &a6, &a7, &a8, &a9, &a10, &a11, &a12, &a13);
            if (fres == 13) {
                m_res_firsttime = a1;
                m_res_lasttime = a2;
                m_res_complaint_ham = a3;
                m_res_complaint_spam = a4;
                m_res_ham = a5;
                m_res_spam = a6;
                m_res_history = a7;
                m_res_replace_to_ham = a8;
                m_res_replace_to_spam = a9;
                m_res_uniquser_ham = a10;
                m_res_uniquser_spam = a11;
                m_res_uniquser = a12;
                m_res_lasttime_change = a13;

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

TString TFromStatRequest::PrintData() const {
    TString res = "";
    char buff[256];
    TString stype = "";

    stype = TStatTypeToText(Type());
    snprintf(buff, sizeof(buff) - 1, "type=%s shingle=%s ft=%u lt=%u ...", stype.c_str(), sShingle(), ResFirsttime(), ResLasttime());

    if (ResEmpty())
        res = res + "PRINT FROMSTAT: empty (" + TString(buff) + ")";
    else
        res = res + "PRINT FROMSTAT: nonempty (" + TString(buff) + ")";

    return res;
}

TString TFromStatRequest::PrintData2() const {
    TString res = "";
    char buff[1024];
    TString stype = "";

    stype = TStatTypeToText(Type());
    snprintf(buff, sizeof(buff) - 1, "type=%11s shingle=%s ft=%u lt=%u cmpl_ham=%u cmpl_spam=%u ham=%u spam=%u history=%u rt_ham=%u rt_spam=%u lt_chng=%u pr_a=%d pr_m=%d uniq=%u uniq_ham=%u uniq_spam=%u", stype.c_str(), sShingle(), ResFirsttime(), ResLasttime(), ResComplaintHam(), ResComplaintSpam(), ResHam(), ResSpam(), ResHistory(), ResReplaceToHam(), ResReplaceToSpam(), ResLasttimeChange(), (int)ResAutoAction(), (int)ResManualAction(), ResUniquser(), ResUniquserHam(), ResUniquserSpam());

    if (ResEmpty())
        res = res + "PRINT FROMSTAT: empty (" + TString(buff) + ")";
    else
        res = res + "PRINT FROMSTAT: nonempty (" + TString(buff) + ")";

    return res;
}

//***************************************************************************************
//                                   TDLVShingles
//***************************************************************************************

TDLVShingles::TDLVShingles() {
    m_shingle = 0;
    m_type = 0;
    m_shingle_str = "";
}

TDLVShingles::TDLVShingles(ui64 shingleA, ui8 typeA) {
    char buff[32];

    m_shingle = shingleA;
    m_type = typeA;
    memset(buff, 0, sizeof(buff));
    snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "", m_shingle);
    m_shingle_str = TString(buff);
}

TDLVShinglesRequest::TDLVShinglesRequest() {
    m_shingle = TDLVShingles();
    ResClear();
}

TDLVShinglesRequest::TDLVShinglesRequest(ui64 shingleA, ui8 typeA) {
    m_shingle = TDLVShingles(shingleA, typeA);
    ResClear();
}

TDLVShinglesRequest::TDLVShinglesRequest(const TStringBuf& shingleA, ui8 typeA) {
    if (shingleA != nullptr) {
        ui64 v_shingle = 0;
        auto len = shingleA.size();

        if ((len > 0) && (len < 20)) {
            if (sscanf(shingleA.data(), "%" PRIx64, &v_shingle) > 0)
                m_shingle = TDLVShingles(v_shingle, typeA);
        }
    }
    ResClear();
}

void TDLVShinglesRequest::ResClear() {
    m_res_empty = true;
    m_res_ham = 0;
    m_res_spam = 0;
    m_res_pr = 0;
    m_res_inwhitelist = false;
    m_res_today_ham = 0;
    m_res_today_spam = 0;
    m_res_yesterday_ham = 0;
    m_res_yesterday_spam = 0;

    m_res_weight = 0;
    m_res_exist_weight = 0;
    m_res_time_set_weigth = 0;
    m_res_last_set_weigth = 0;
    m_res_setweigths_count = 0;
    m_res_during = 0;

    m_res_weight_in = 0;
    m_res_exist_weight_in = 0;
    m_res_time_set_weigth_in = 0;
    m_res_last_set_weigth_in = 0;
    m_res_setweigths_count_in = 0;
    m_res_during_in = 0;
}

const char* TDLVShinglesRequest::sShingle() const {
    return m_shingle.Shingle_str().c_str();
}

TString TDLVShinglesRequest::GetRequest() const {
    char buff[64];
    ui64 a1 = 0;
    ui32 a2 = 0;

    a1 = m_shingle.Shingle();
    a2 = m_shingle.Type();
    memset(buff, 0, sizeof(buff));
    snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 "-%u", a1, a2);

    return TString(buff);
}

bool TDLVShinglesRequest::ResSetFromText(ui64 shingle, ui16 type, bool empty, TString& text) {
    bool res = false;
    ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0, a5 = 0, a6 = 0, a7 = 0, a9 = 0, a10 = 0, a11 = 0, a12 = 0, a13 = 0, a15 = 0, a16 = 0, a17 = 0, a18 = 0, a19 = 0;
    float a8 = 0, a14 = 0;
    int fres = 0;

    if ((shingle == m_shingle.Shingle()) && (type == m_shingle.Type())) {
        m_res_empty = empty;
        if (!m_res_empty) {
            fres = sscanf(text.c_str(), "%u-%u-%u-%u-%u-%u-%u-%f-%u-%u-%u-%u-%u-%f-%u-%u-%u-%u-%u", &a1, &a2, &a3, &a4, &a5, &a6, &a7, &a8, &a9, &a10, &a11, &a12, &a13, &a14, &a15, &a16, &a17, &a18, &a19);
            if (fres >= 8) {
                m_res_ham = a1;
                m_res_spam = a2;
                m_res_pr = a3;
                if ((a3 & 0x01) > 0)
                    m_res_inwhitelist = true;
                m_res_today_ham = a4;
                m_res_today_spam = a5;
                m_res_yesterday_ham = a6;
                m_res_yesterday_spam = a7;
                m_res_weight = a8;

                if (fres >= 13) {
                    m_res_exist_weight = a9;
                    m_res_time_set_weigth = a10;
                    m_res_last_set_weigth = a11;
                    m_res_setweigths_count = a12;
                    m_res_during = a13;

                    if (fres >= 19) {
                        m_res_weight_in = a14;
                        m_res_exist_weight_in = a15;
                        m_res_time_set_weigth_in = a16;
                        m_res_last_set_weigth_in = a17;
                        m_res_setweigths_count_in = a18;
                        m_res_during_in = a19;
                    }
                }

                res = true;
            } else {
                m_res_empty = true;
            }
        }
    }

    return res;
}

TString TDLVShinglesRequest::PrintData() const {
    TString res = "";
    char buff[256];
    TString stype = "";

    snprintf(buff, sizeof(buff) - 1, "shingle=%s type=%d ...", sShingle(), Type());

    if (ResEmpty())
        res = res + "PRINT DLVSH: empty (" + TString(buff) + ")";
    else
        res = res + "PRINT DLVSH: nonempty (" + TString(buff) + ")";

    return res;
}

//****************************************************************************************
//                                TShinglesGroupClass
//****************************************************************************************

TString TShinglesGroupClass::CreateRequestToLongshinglerWOUid(TString& errtext) {
    return CreateRequestToLongshinglerA(errtext);
}

TString TShinglesGroupClass::CreateRequestToLongshinglerA(TString& errtext) {
    TString res = "";
    char tbuff[64];
    TString request_otherdata = "";
    TString request_parsedsh = "";
    TString request_compl = "";
    TString request_stat = "";
    TString messid_s = "";
    TString queueid_s = "";
    TString complid_s = "";
    TString err_id = "";

    if (IsGoodData(err_id)) //данных достаточно и заполнены верно
    {
        complid_s = complaint_id;
        CGIEscape(complid_s);

        res = res + "spam=" + TComplSpamTypeToStroka(spam) + "&";
        if (!complid_s.empty())
            res = res + "cid=" + complid_s + "&";

        if ((!messid.empty()) && (!id.GetSuid().empty()) && (data > 0) && (dlv_time > 0)) {
            //other data
            if (!messid.empty()) {
                messid_s = messid;
                CGIEscape(messid_s);
            }
            if (!queueid.empty()) {
                queueid_s = queueid;
                CGIEscape(queueid_s);
            }
            request_otherdata = request_otherdata + "mid=" + messid_s + "&";
            request_otherdata = request_otherdata + "qid=" + queueid_s + "&";
            request_otherdata = request_otherdata + "date=" + UI32ToStroka(data) + "&";
            request_otherdata = request_otherdata + "suid=" + id.GetSuid() + "&";
            request_otherdata = request_otherdata + "dlvt=" + UI32ToStroka(dlv_time) + "&";

            if (shlist.size() > 0) {
                //parsed shingles
                auto it = shlist.begin();
                while (it != shlist.end()) {
                    memset(tbuff, 0, sizeof(tbuff));
                    snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "-%d-%u-%u", (*it).shingle, (*it).type, (*it).spam, (*it).ham);
                    request_parsedsh = request_parsedsh + TString(tbuff) + "&";

                    ++it;
                }
                if (!request_parsedsh.empty())
                    request_parsedsh = "shdata=" + request_parsedsh;
            }

            res = res + request_otherdata;
            res = res + request_parsedsh;
        }

        //compl & fromstat shingles
        if ((from_shingle != 0) || (fromdomen_shingle != 0) || (returnpath_shingle != 0) || (mess_shingle != 0) || (subj_shingle != 0) || (beensender_compl_shingle != 0) || (list_compl_shingle != 0) || (psndr_compl_shingle != 0) || (psndr_nofrom_compl_shingle != 0) ||
            (from_shingle_coll != 0) || (fromdomen_shingle_coll != 0) || (returnpath_shingle_coll != 0) || (mess_shingle_coll != 0) || (subj_shingle_coll != 0) || (beensender_compl_shingle_coll != 0) || (list_compl_shingle_coll != 0) || (psndr_compl_shingle_coll != 0) || (psndr_nofrom_compl_shingle_coll != 0) ||
            (from_shingle_stat != 0) || (beensender_stat_shingle != 0) || (list_stat_shingle != 0)) {
            //жалобы
            if ((from_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", from_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCFROM, spam).PackValueS();
            }
            if ((fromdomen_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", fromdomen_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCFROMDOMEN, spam).PackValueS();
            }
            if ((returnpath_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", returnpath_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCRETURNPATH, spam).PackValueS();
            }
            if ((mess_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", mess_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCMESS, spam).PackValueS();
            }
            if ((subj_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", subj_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCSUBJECT, spam).PackValueS();
            }
            if ((beensender_compl_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", beensender_compl_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCBEENSENDER, spam).PackValueS();
            }
            if ((list_compl_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", list_compl_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCLIST, spam).PackValueS();
            }
            if ((psndr_compl_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", psndr_compl_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCPSNDR, spam).PackValueS();
            }
            if ((psndr_nofrom_compl_shingle != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", psndr_nofrom_compl_shingle);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCPSNDR_NOFROM, spam).PackValueS();
            }

            //жалобы сборщика
            if ((from_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", from_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCFROM, spam).PackValueS();
            }
            if ((fromdomen_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", fromdomen_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCFROMDOMEN, spam).PackValueS();
            }
            if ((returnpath_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", returnpath_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCRETURNPATH, spam).PackValueS();
            }
            if ((mess_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", mess_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCMESS, spam).PackValueS();
            }
            if ((subj_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", subj_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCSUBJECT, spam).PackValueS();
            }
            if ((beensender_compl_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", beensender_compl_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCBEENSENDER, spam).PackValueS();
            }
            if ((list_compl_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", list_compl_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCLIST, spam).PackValueS();
            }
            if ((psndr_compl_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", psndr_compl_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCPSNDR, spam).PackValueS();
            }
            if ((psndr_nofrom_compl_shingle_coll != 0)) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", psndr_nofrom_compl_shingle_coll);
                if (!request_compl.empty())
                    request_compl = request_compl + "&";
                request_compl = request_compl + TString(tbuff) + "-" + TPackShingleType(KCPSNDR_NOFROM, spam).PackValueS();
            }

            //статистика
            if (from_shingle_stat != 0) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", from_shingle_stat);
                if (!request_stat.empty())
                    request_stat = request_stat + "&";
                request_stat = request_stat + TString(tbuff) + "-" + TPackShingleType2(KSFROM, spam).PackValueS();
            }
            if (beensender_stat_shingle != 0) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", beensender_stat_shingle);
                if (!request_stat.empty())
                    request_stat = request_stat + "&";
                request_stat = request_stat + TString(tbuff) + "-" + TPackShingleType2(KSBEENSENDER, spam).PackValueS();
            }
            if (list_stat_shingle != 0) {
                memset(tbuff, 0, sizeof(tbuff));
                snprintf(tbuff, sizeof(tbuff) - 1, "%016" PRIx64 "", list_stat_shingle);
                if (!request_stat.empty())
                    request_stat = request_stat + "&";
                request_stat = request_stat + TString(tbuff) + "-" + TPackShingleType2(KSLIST, spam).PackValueS();
            }

            if (!request_compl.empty())
                res = res + "ac=" + request_compl + "&";
            if (!request_stat.empty())
                res = res + "as=" + request_stat + "&";
        }

        if (res.empty()) {
            if (request_otherdata.empty()) {
                if (!errtext.empty())
                    errtext = errtext + ", ";
                errtext = errtext + "no defined other fields";
            }
            if (request_parsedsh.empty()) {
                if (!errtext.empty())
                    errtext = errtext + ", ";
                errtext = errtext + "no parsed shingles";
            }
            if (request_compl.empty()) {
                if (!errtext.empty())
                    errtext = errtext + ", ";
                errtext = errtext + "no compl shingles";
            }
            if (request_stat.empty()) {
                if (!errtext.empty())
                    errtext = errtext + ", ";
                errtext = errtext + "no stat shingles";
            }
        }
    } else
        errtext = "no full data: " + err_id;

    return res;
}

bool TShinglesGroupClass::IsGoodData(TString& err_id) const {
    bool res = true;
    bool empty_compl_shingles = false;
    bool empty_compl_shingles_coll = false;
    bool empty_stat_shingles = false;

    err_id = "";

    if (!definespam) {
        res = false;
        err_id = err_id + "1";
    }

    if ((from_shingle == 0) && (fromdomen_shingle == 0) && (returnpath_shingle == 0) && (mess_shingle == 0) && (subj_shingle == 0) && (beensender_compl_shingle == 0) && (list_compl_shingle == 0))
        empty_compl_shingles = true;
    if ((from_shingle_coll == 0) && (fromdomen_shingle_coll == 0) && (returnpath_shingle_coll == 0) && (mess_shingle_coll == 0) && (subj_shingle_coll == 0) && (beensender_compl_shingle_coll == 0) && (list_compl_shingle_coll == 0))
        empty_compl_shingles_coll = true;
    if ((from_shingle_stat == 0) && (beensender_stat_shingle == 0) && (list_stat_shingle == 0))
        empty_stat_shingles = true;

    if (empty_compl_shingles && empty_compl_shingles_coll && empty_stat_shingles) {
        if (empty_compl_shingles)
            err_id = err_id + "2";
        if (empty_compl_shingles_coll)
            err_id = err_id + "3";
        if (empty_stat_shingles)
            err_id = err_id + "4";

        res = false;
    }

    //mailtype
    if (messid.empty()) {
        res = false;
        err_id = err_id + "5";
    }
    //queueid
    //from
    //spam
    if (data == 0) {
        res = false;
        err_id = err_id + "6";
    }
    if (id.GetSuid().empty()) {
        res = false;
        err_id = err_id + "7";
    }
    if (dlv_time == 0) {
        res = false;
        err_id = err_id + "8";
    }
    //shlist
    //sendersuid

    if (complaint_id.empty()) {
        res = false;
        err_id = err_id + "9";
    }

    return res;
}

bool TShinglesGroupClass::SetComplData(const TString& messidA, const TString& queueidA, TComplSpamType spamA, time_t dateA, const TUserID& idA, time_t dlv_timeA, const TString& fromA,
                                       const TUserID& sender_idA, const TString& complaint_idA, const TString& spmHdr, ui32 uiFlags, const TString& psndA) {
    spam = spamA;
    definespam = true;
    messid = messidA;
    queueid = queueidA;
    data = dateA;
    id = idA;
    dlv_time = dlv_timeA;
    from = fromA;
    sender_id = sender_idA;
    complaint_id = complaint_idA;
    ui_flags = uiFlags;
    spamHdr = spmHdr;
    psnd = psndA;

    return true;
}

//*************************************************************************************************************************
//                                  TComplaintFromUser
//*************************************************************************************************************************

bool TComplaintFromUser::IsGood() const {
    bool res = true;

    if (m_login.empty())
        res = false;
    if (m_suid.empty())
        res = false;
    if (m_ip.empty())
        res = false;
    if (m_data.empty())
        res = false;
    if (m_type.empty())
        res = false;
    if (m_move.empty())
        res = false;
    if (m_mail.empty())
        res = false;

    return res;
}

bool TComplaintFromUser::SetParams(const TString& login, const TString& suid, const TString& ip, const TString& tsCompl, const TString& type, const TString& move, const TString& mail, const TString& dlvlog) {
    bool res = true;

    *this = {};
    m_enb = true;
    m_login = login;
    m_suid = suid;
    m_ip = ip;
    m_data = tsCompl;
    m_type = type;
    m_move = move;
    m_mail = mail;
    m_dlvlog = dlvlog;

    if (!IsGood())
        res = false;

    return res;
}

TString TComplaintFromUser::GetType() const {
    return BoolToStroka(TComplSpamTypeToSpam(StrokaToTComplSpamType((m_type))));
}

TString TComplaintFromUser::GetTypeStr() const {
    return m_type;
}

TString TComplaintFromUser::GetComplainee() const {
    return m_login;
}

//*************************************************************************************************************************
//                                  TParseResponce
//*************************************************************************************************************************

void TParseResponce::ParseResponce(const TString& s) {
    const ui32 sLength = s.size();
    static const char identhost[] = "<HOST";
    static const char identfromto[] = "<FROMTO";
    static const char identcompl[] = "<COMPL";
    static const char identfromstat[] = "<FROMST";
    static const char identdlvsh[] = "<SHDLV";
    static const char identfreemail[] = "<FREEMAIL";
    static const char identbounce[] = "<BOUNCE";
    static const char identpdd[] = "<PDD";
    static const char identsender[] = "<SENDER";
    static const char identdomain[] = "<DOMAIN";
    static const char identpsndr[] = "<PSNDR";
    static const char identshingle[] = "shingle='";
    static const char identtype[] = "type='";
    static const char identnull[] = "null='";
    static const char identbody[] = "body='";
    static const char identstat[] = "stat='";
    static const char identprofile[] = "profile='";
    static const char indentshinglenewuid[] = "shingle_newuid='";
    const char *pbegin = nullptr;
    const char *pend = nullptr;
    ptrdiff_t count = 0;
    bool f1 = false, f2 = false, f3 = false, f4 = false, f5 = false, f6 = false, f7 = false;

    //responce type
    if ((sLength >= sizeof(identhost) - 1) && (!memcmp(s.c_str(), identhost, sizeof(identhost) - 1)))
        resptype = KTHOST;
    else if ((sLength >= sizeof(identfromto) - 1) && (!memcmp(s.c_str(), identfromto, sizeof(identfromto) - 1)))
        resptype = KTFROMTO;
    else if ((sLength >= sizeof(identcompl) - 1) && (!memcmp(s.c_str(), identcompl, sizeof(identcompl) - 1)))
        resptype = KTCOMPL;
    else if ((sLength >= sizeof(identfromstat) - 1) && (!memcmp(s.c_str(), identfromstat, sizeof(identfromstat) - 1)))
        resptype = KTFROMSTAT;
    else if ((sLength >= sizeof(identdlvsh) - 1) && (!memcmp(s.c_str(), identdlvsh, sizeof(identdlvsh) - 1)))
        resptype = KTDLVSH;
    else if ((sLength >= sizeof(identfreemail) - 1) && (!memcmp(s.c_str(), identfreemail, sizeof(identfreemail) - 1)))
        resptype = KTFREEMAIL;
    else if ((sLength >= sizeof(identbounce) - 1) && (!memcmp(s.c_str(), identbounce, sizeof(identbounce) - 1)))
        resptype = KTBOUNCE;
    else if ((sLength >= sizeof(identpdd) - 1) && (!memcmp(s.c_str(), identpdd, sizeof(identpdd) - 1)))
        resptype = KTPDD;
    else if ((sLength >= sizeof(identsender) - 1) && (!memcmp(s.c_str(), identsender, sizeof(identsender) - 1)))
        resptype = KTSENDER;
    else if ((sLength >= sizeof(identdomain) - 1) && (!memcmp(s.c_str(), identdomain, sizeof(identdomain) - 1)))
        resptype = KTDOMAIN;
    else if ((sLength >= sizeof(identpsndr) - 1) && (!memcmp(s.c_str(), identpsndr, sizeof(identpsndr) - 1)))
        resptype = KTPSNDR;
    else
        resptype = KTUNKNOWN;

    //shingle
    pbegin = strstr(s.c_str(), identshingle);
    if (pbegin != nullptr) {
        pend = strchr(pbegin + sizeof(identshingle) - 1, '\'');
        if (pend != nullptr) {
            count = pend - pbegin - (sizeof(identshingle) - 1);
            if (count > 0) {
                sscanf(pbegin + sizeof(identshingle) - 1, "%" PRIx64, &shingle);
                f1 = true;
            }
        }
    }
    if ((f1) && (shingle == 0)) {
        pbegin = strstr(s.c_str(), indentshinglenewuid);
        if (pbegin != nullptr) {
            pend = strchr(pbegin + sizeof(indentshinglenewuid) - 1, '\'');
            if (pend != nullptr) {
                count = pend - pbegin - (sizeof(indentshinglenewuid) - 1);
                if (count > 0) {
                    sscanf(pbegin + sizeof(indentshinglenewuid) - 1, "%" PRIx64, &shingle2);

                    f7 = true;
                }
            }
        }
    }

    //type
    pbegin = strstr(s.c_str(), identtype);
    if (pbegin != nullptr) {
        pend = strchr(pbegin + sizeof(identtype) - 1, '\'');
        if (pend != nullptr) {
            count = pend - pbegin - (sizeof(identtype) - 1);
            if (count > 0) {
                sscanf(pbegin + sizeof(identtype) - 1, "%i", &type);

                f2 = true;
            }
        }
    }

    //empty (null)
    pbegin = strstr(s.c_str(), identnull);
    if (pbegin != nullptr) {
        pend = strchr(pbegin + sizeof(identnull) - 1, '\'');
        if (pend != nullptr) {
            count = pend - pbegin - (sizeof(identnull) - 1);
            if (count > 0) {
                empty = (bool) atoi(pbegin + sizeof(identnull) - 1);

                f3 = true;
            }
        }
    }

    //body
    pbegin = strstr(s.c_str(), identbody);
    if (pbegin != nullptr) {
        pend = strchr(pbegin + sizeof(identbody) - 1, '\'');
        if (pend != nullptr) {
            count = pend - pbegin - (sizeof(identbody) - 1);
            if ((count > 0) && (count < 64000)) {
                body = TString(pbegin + sizeof(identbody) - 1, count);
                f4 = true;
            }
        }
    }

    //stat
    pbegin = strstr(s.c_str(), identstat);
    if (pbegin != nullptr) {
        pend = strchr(pbegin + sizeof(identstat) - 1, '\'');
        if (pend != nullptr) {
            count = pend - pbegin - (sizeof(identstat) - 1);
            if ((count > 0) && (count < 64000)) {
                stat = TString(pbegin + sizeof(identstat) - 1, count);
                f5 = true;
            }
        }
    }

    //profile
    pbegin = strstr(s.c_str(), identprofile);
    if (pbegin != nullptr) {
        pend = strchr(pbegin + sizeof(identprofile) - 1, '\'');
        if (pend != nullptr) {
            count = pend - pbegin - (sizeof(identprofile) - 1);
            if ((count > 0) && (count < 64000)) {
                profile = TString(pbegin + sizeof(identprofile) - 1, count);
                f6 = true;
            }
        }
    }

    switch (resptype) {
        case KTHOST:
        case KTDLVSH:
            if (!(f1 && f2 && f3 && f4))
                empty = true;
            break;
        case KTFROMTO:
        case KTCOMPL:
        case KTFROMSTAT:
        case KTPDD:
            if (!((f1 || f7) && f3 && f4))
                empty = true;
            break;
        case KTFREEMAIL:
        case KTSENDER:
        case KTDOMAIN:
        case KTPSNDR:
            if (!(f1 && f3 && f4 /* && f5*/))
                empty = true;
            break;
        case KTBOUNCE:
            if (!(f3 && f4))
                empty = true;
            break;
        default:
            empty = true;
    };
}

//****************************************************************************************
