/*
 * tsenderrepenv.cpp
 *
 *  Created on: 23 мар. 2017 г.
 *      Author: luckybug
 */

#include <algorithm>
#include <util/stream/format.h>
#include <util/system/defaults.h>
#include "tsenderrepenv2.h"

namespace NSenderReputation {
    //****************************************************************************************************
    //                                     TSenderShingle
    //****************************************************************************************************

    bool TShingle::operator==(const TShingle& anotherShingle) const {
        return m_shingle_email == anotherShingle.m_shingle_email &&
               m_shingle_domain == anotherShingle.m_shingle_domain &&
               m_shingle_psndr == anotherShingle.m_shingle_psndr;
    }

    IOutputStream& operator<<(IOutputStream& stream, const TShingle& shingle) {
        if(shingle.ShingleEmail())
            stream << Hex(*shingle.ShingleEmail());
        else
            stream << "0x0";
        stream << "(" << shingle.m_shingle_email << ")";

        if(shingle.ShingleDomain())
            stream << Hex(*shingle.ShingleDomain());
        else
            stream << "0x0";
        stream << "(" << shingle.m_shingle_domain << ")";

        if(shingle.ShinglePSndr())
            stream << Hex(*shingle.ShinglePSndr());
        else
            stream << "0x0";
        stream << "(" << shingle.m_shingle_psndr << ")";

        return stream;
    }

    TShingle::TShingle(TString rawEmail, TSenderBitset use_shingles)
        : m_shingle_email()
        , m_shingle_domain()
        , m_shingle_psndr()
    {
        const TEmail email(NormalizeEmail(std::move(rawEmail)));

        if (!email.Email().empty()) {
            if (use_shingles.Test(TSenderType::Sender)) {
                m_shingle_email = ShingleFromStroka(email.Email());
            }

            if (email.Domain().Defined() && use_shingles.Test(TSenderType::Domain)) {
                const TString& domain = *email.Domain();
                m_shingle_domain = ShingleFromStroka(domain);
            }
        }
    }
    TShingle::TShingle(TString email, const TString& psndr, TSenderBitset use_shingles)
        : TShingle(std::move(email), use_shingles)
    {
        if (!psndr.empty() && use_shingles.Test(TSenderType::Paysender)) {
            m_shingle_psndr = ShingleFromStroka(psndr);
        }
    }

    TString TShingle::NormalizeEmail(TString email) {
        if (!email)
            return "";
        to_lower_k(email);
        return email;
    }

    void TDBSender::Clear() {
        *this = {};
    }

    template<typename... Ts> struct TDoNothing{
        static size_t fill(std::tuple<Ts...> && /*tupleOfTuples*/, const NJson::TJsonValue & /*val*/) {
            return 0;
        }
    };

    template<int index, typename... Ts> struct TTotalsFiller{
        static size_t fill(std::tuple<Ts...> && tupleOfTuples, const NJson::TJsonValue & val) {
            auto & tuple = std::get<index>(tupleOfTuples);
            const auto & name = std::get<0>(tuple);
            auto & value = std::get<1>(tuple);

            size_t totalSet = 0;

            if(val.Has(name)) {
                value = val[name].GetUIntegerRobust();
                totalSet ++;
            }


            return totalSet + std::conditional_t<index == 0, TDoNothing<Ts...>, TTotalsFiller<index-1, Ts...>>::fill(std::move(tupleOfTuples), val);
        }
    };

    template<typename... Ts> size_t FillTotals(std::tuple<Ts...>&& t, const NJson::TJsonValue & val) {
        const auto size = std::tuple_size<std::tuple<Ts...>>::value;
        return TTotalsFiller<size - 1, Ts...>::fill(std::move(t), val);
    }

    void TDBSender::FromJson(const NJson::TJsonValue & val) {
        empty = 0 == FillTotals(std::make_tuple(
                std::make_tuple("ct", std::ref(firsttime)),
                std::make_tuple("ut", std::ref(lasttime)),
                std::make_tuple("ad", std::ref(activedays)),
                std::make_tuple("hc", std::ref(ham)),
                std::make_tuple("sc", std::ref(spam)),
                std::make_tuple("ry", std::ref(rcvd_fromyandex_count)),
                std::make_tuple("ch", std::ref(complaint_ham)),
                std::make_tuple("cs", std::ref(complaint_spam)),
                std::make_tuple("fl", std::ref(flags)),
                std::make_tuple("ph", std::ref(personalham)),
                std::make_tuple("ps", std::ref(personalspam)),
                std::make_tuple("ab", std::ref(abook_count)),
                std::make_tuple("ba", std::ref(kooba_count)),
                std::make_tuple("read", std::ref(read)),
                std::make_tuple("deleted", std::ref(del_wo_read)),
                std::make_tuple("dkim_ham", std::ref(dkim_ham)),
                std::make_tuple("dkim_spam", std::ref(dkim_spam)),
                std::make_tuple("pop3_ham", std::ref(pop3_ham)),
                std::make_tuple("pop3_spam", std::ref(pop3_spam)),
                std::make_tuple("dkim_complaint_spam", std::ref(compl_dkim_spam)),
                std::make_tuple("dkim_del", std::ref(dkim_del_wo_read)),
                std::make_tuple("ps_ham", std::ref(ps_ham)),
                std::make_tuple("ps_spam", std::ref(ps_spam))
        ), val);
    }

#define FILL_SENDER_2WEEKS(name, member) \
    member = cur[name].GetUIntegerRobust() + daysOnPrevWeek * prev[name].GetUIntegerRobust() / 14;

    void TDBPeriodicSender::Clear() {
        *this = {};
    }

    template<int index, typename... Ts> void FillWeeksInner(std::tuple<Ts...> & tupleOfTuples, const NJson::TJsonValue &cur, const NJson::TJsonValue &prev, int daysOnPrevWeek) {
        auto & tuple = std::get<index>(tupleOfTuples);
        const auto & name = std::get<0>(tuple);
        auto & value = std::get<1>(tuple);

        value = cur[name].GetUIntegerRobust() + daysOnPrevWeek * prev[name].GetUIntegerRobust() / 14;

        if constexpr(index > 0)
            FillWeeksInner<index-1, Ts...>(tupleOfTuples, cur, prev, daysOnPrevWeek);
    }

    template<typename... Ts> void FillWeeks(std::tuple<Ts...>& t, const NJson::TJsonValue &cur, const NJson::TJsonValue &prev, int daysOnPrevWeek) {
        const auto size = std::tuple_size<std::tuple<Ts...>>::value;
        FillWeeksInner<size - 1, Ts...>(t, cur, prev, daysOnPrevWeek);
    }

    void TDBPeriodicSender::FromJson(const NJson::TJsonValue & val) {
        FillTotals(std::make_tuple(
            std::make_tuple("ham", std::ref(ham_count)),
            std::make_tuple("spam", std::ref(spam_count)),
            std::make_tuple("compl_ham", std::ref(compl_ham_count)),
            std::make_tuple("compl_spam", std::ref(compl_spam_count)),
            std::make_tuple("pers_ham", std::ref(pers_ham_count)),
            std::make_tuple("pers_spam", std::ref(pers_spam_count)),
            std::make_tuple("kooba", std::ref(kooba_count)),
            std::make_tuple("read", std::ref(read)),
            std::make_tuple("deleted", std::ref(del_wo_read)),
            std::make_tuple("ps_ham", std::ref(ps_ham)),
            std::make_tuple("ps_spam", std::ref(ps_spam)),
            std::make_tuple("dkim_ham", std::ref(dkim_ham_count)),
            std::make_tuple("dkim_spam", std::ref(dkim_spam_count)),
            std::make_tuple("dkim_del", std::ref(dkim_del_wo_read))
        ), val);
    }

    TDBPeriodicSender TDBPeriodicSender::Interpolate(const TDBPeriodicSender & cur, const TDBPeriodicSender & prev, ui16 day) {
        auto week2 = day / 14;

        const auto daysOnPrevWeek = (week2+1)*14 - day;

        return cur + prev * daysOnPrevWeek / 14;
    }

    IOutputStream &operator<<(IOutputStream &os, const TDBPeriodicSender &sender) {
        os << "ham_count: " << sender.ham_count << " spam_count: " << sender.spam_count << " compl_ham_count: "
           << sender.compl_ham_count << " compl_spam_count: " << sender.compl_spam_count << " pers_ham_count: "
           << sender.pers_ham_count << " pers_spam_count: " << sender.pers_spam_count << " kooba_count: "
           << sender.kooba_count << " read: " << sender.read << " del_wo_read: " << sender.del_wo_read << " ps_ham: "
           << sender.ps_ham << " ps_spam: " << sender.ps_spam << " dkim_ham_count: " << sender.dkim_ham_count
           << " dkim_spam_count: " << sender.dkim_spam_count << " dkim_del_wo_read: " << sender.dkim_del_wo_read;
        return os;
    }

    TDBPeriodicSender TDBPeriodicSender::operator + (const TDBPeriodicSender & s) const {
        TDBPeriodicSender result;

        result.ham_count = ham_count + s.ham_count;
        result.spam_count = spam_count + s.spam_count;
        result.compl_ham_count = compl_ham_count + s.compl_ham_count;
        result.compl_spam_count = compl_spam_count + s.compl_spam_count;
        result.pers_ham_count = pers_ham_count + s.pers_ham_count;
        result.pers_spam_count = pers_spam_count + s.pers_spam_count;
        result.kooba_count = kooba_count + s.kooba_count;
        result.read = read + s.read;
        result.del_wo_read = del_wo_read + s.del_wo_read;
        result.ps_ham = ps_ham + s.ps_ham;
        result.ps_spam = ps_spam + s.ps_spam;
        result.dkim_ham_count = dkim_ham_count + s.dkim_ham_count;
        result.dkim_spam_count = dkim_spam_count + s.dkim_spam_count;
        result.dkim_del_wo_read = dkim_del_wo_read + s.dkim_del_wo_read;

        return result;
    }

    TDBPeriodicSender TDBPeriodicSender::operator * (int v) const {
        TDBPeriodicSender result;

        result.ham_count = ham_count * v;
        result.spam_count = spam_count * v;
        result.compl_ham_count = compl_ham_count * v;
        result.compl_spam_count = compl_spam_count * v;
        result.pers_ham_count = pers_ham_count * v;
        result.pers_spam_count = pers_spam_count * v;
        result.kooba_count = kooba_count * v;
        result.read = read * v;
        result.del_wo_read = del_wo_read * v;
        result.ps_ham = ps_ham * v;
        result.ps_spam = ps_spam * v;
        result.dkim_ham_count = dkim_ham_count * v;
        result.dkim_spam_count = dkim_spam_count * v;
        result.dkim_del_wo_read = dkim_del_wo_read * v;

        return result;
    }

    TDBPeriodicSender TDBPeriodicSender::operator / (int v) const {
        TDBPeriodicSender result;

        result.ham_count = ham_count / v;
        result.spam_count = spam_count / v;
        result.compl_ham_count = compl_ham_count / v;
        result.compl_spam_count = compl_spam_count / v;
        result.pers_ham_count = pers_ham_count / v;
        result.pers_spam_count = pers_spam_count / v;
        result.kooba_count = kooba_count / v;
        result.read = read / v;
        result.del_wo_read = del_wo_read / v;
        result.ps_ham = ps_ham / v;
        result.ps_spam = ps_spam / v;
        result.dkim_ham_count = dkim_ham_count / v;
        result.dkim_spam_count = dkim_spam_count / v;
        result.dkim_del_wo_read = dkim_del_wo_read / v;

        return result;
    }

#undef FILL_SENDER_2WEEKS
    IOutputStream& operator<<(IOutputStream& stream, const TGetData& data) {

        const auto now =(ui32)(kday_t(time(nullptr)));
        //%d %" PRId64 "_ %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "_ %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "_ %" PRId64 " %" PRId64 "_ %" PRId64 " %" PRId64 "_%s
        stream  << (now - data.sender.totals.firsttime) << ' '
                << data.sender.totals.activedays << "_ "
                << data.sender.periodic.ham_count << ' ' << data.sender.periodic.spam_count << ' '
                << data.sender.periodic.compl_ham_count << ' ' << data.sender.periodic.compl_spam_count << ' '
                << data.sender.periodic.pers_ham_count << ' ' << data.sender.periodic.pers_spam_count << ' '
                << data.sender.periodic.kooba_count << "_ "

                << data.sender.totals.ham << ' ' << data.sender.totals.spam << ' '
                << data.sender.totals.complaint_ham << ' ' << data.sender.totals.complaint_spam << "_ "
                << data.sender.totals.personalham << ' ' << data.sender.totals.personalspam << "_ "

                << data.sender.totals.abook_count << ' '
                << data.sender.totals.rcvd_fromyandex_count << "_ "
                << data.sender.periodic.del_wo_read << ' ' << data.sender.totals.del_wo_read << "_ ";


        if (data.ResIsSenderDL_FBR())
            stream << 'D';
        if (data.ResIsSenderFIXED_SOURCE())
            stream << 'S';
        if (data.ResIsSenderSOURCE_LEVEL1())
            stream << 'L';
        if (data.ResIsSenderWEBMASTER())
            stream << 'W';
        if (data.ResIsSenderMALIC())
            stream << 'M';

        return stream;
    }

    ui32 TGetData::ResSenderFirsttimeElapsed() const {
        if(sender.totals.firsttime == 0) {
            return 0;
        }
        const ui32 currenttime = (ui32)(kday_t(time(nullptr)));
        if (currenttime >= sender.totals.firsttime)
            return currenttime - sender.totals.firsttime;

        return 0;
    }

    ui32 TGetData::ResDomenFirsttimeElapsed() const {
        if(domain.totals.firsttime == 0) {
            return 0;
        }

        const auto currenttime = (ui32)(kday_t(time(nullptr)));
        if (currenttime >= domain.totals.firsttime)
            return currenttime - domain.totals.firsttime;
        return 0;
    }


    ui32 TGetData::ResPSndrFirsttimeElapsed() const {
        ui32 res = 0;
        ui32 currenttime = 0;

        currenttime = (ui32)(kday_t(time(nullptr)));
        if (currenttime >= psndr.totals.firsttime)
            res = currenttime - psndr.totals.firsttime;

        return res;
    }
} //namepace NSenderReputation
