#pragma once

#include <mail/so/spamstop/tools/so-clients/kshingle.h>
#include <mail/so/spamstop/tools/so-common/kfunc.h>

#include <util/generic/list.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <util/stream/format.h>
#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <mail/so/spamstop/tools/so-clients/tsenderrepenv.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/enumbitset/enumbitset.h>


namespace NSenderReputation {
    enum TPersonalAction { PA_NONE,
                           PA_PERSSPAM,
                           PA_PERSHAM };

    //****************************************************************************************************
    //                                     TSenderShingle
    //****************************************************************************************************

    class TEmail {
    public:
        friend IOutputStream& operator<<(IOutputStream& stream, const TEmail& email) {
            return stream << "email:" << email.email << ", domain:" << email.domain;
        }
        const TString& Email() const { return email; }
        const TMaybe<TString>& Domain() const { return domain; }
        TEmail() = default;

        explicit TEmail(TString email) : email(std::move(email)) {
            domain = extractDomen(this->email);
        }

    private:
        static TMaybe<TString> extractDomen(const TString& email) {
            size_t dogPos = email.find_last_of('@');
            if (TString::npos == dogPos || dogPos == email.Size() - 1)
                return Nothing();

            return email.substr(dogPos + 1);
        }

    private:
        TString email;
        TMaybe<TString> domain;
    };

    enum class TSenderType {
        Sender,
        Domain,
        Paysender,
    };

    using TSenderBitset = TEnumBitSet<TSenderType, static_cast<int>(TSenderType::Sender), static_cast<int>(TSenderType::Paysender)+1>;

    class TShingle {
    public:
        bool operator==(const TShingle& anotherShingle) const;

        friend IOutputStream& operator<<(IOutputStream& stream, const TShingle& shingle);

        TShingle() = default;

        explicit TShingle(TString email, TSenderBitset use_shingles = {TSenderType::Sender, TSenderType::Domain});

        TShingle(TString email, const TString& psndr, TSenderBitset use_shingles = {TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender});

        const TMaybe<ui64> & ShingleEmail() const { return m_shingle_email; }

        const TMaybe<ui64> & ShingleDomain() const { return m_shingle_domain; }

        const TMaybe<ui64> & ShinglePSndr() const { return m_shingle_psndr; }
    private:
        static TString NormalizeEmail(TString email);
    private:
        TMaybe<ui64> m_shingle_email{}, m_shingle_domain{}, m_shingle_psndr{};

    };

    //****************************************************************************************************
    //                                     TSenderMailTypeParse
    //****************************************************************************************************

    class TPutDataInMail {
    public:
        bool GetSpam() const {
            return m_spam;
        }
        ui32 GetMask() const {
            return m_mask;
        }
        TPersonalAction GetPers() const {
            return m_pers;
        }
        bool GetAbook() const {
            return m_abook;
        }
        bool GetKooba() const {
            return m_kooba;
        }
        const TMaybe<bool>& GetPop3Spam() const {
            return m_pop3_spam;
        }
        const TMaybe<bool>& GetDkimSpam() const {
            return m_dkim_spam;
        }
        const TMaybe<bool>& GetPsSpam() const {
            return m_ps_spam;
        }

        void SetSpam(bool val) {
            m_spam = val;
        }
        void SetMask(ui32 val) {
            m_mask = val;
        }
        void SetPers(TPersonalAction val) {
            m_pers = val;
        }
        void SetAbook(bool val) {
            m_abook = val;
        }
        void SetKooba(bool val) {
            m_kooba = val;
        }
        void SetPop3Spam(bool val) {
            m_pop3_spam = val;
        }
        void SetDkimSpam(bool val) {
            m_dkim_spam = val;
        }
        void SetPsSpam(bool val) {
            m_ps_spam = val;
        }

    protected:
        bool m_spam{};
        ui32 m_mask{};
        TPersonalAction m_pers = PA_NONE;
        bool m_abook{};
        bool m_kooba{};
        TMaybe<bool> m_pop3_spam;
        TMaybe<bool> m_dkim_spam;
        TMaybe<bool> m_ps_spam;
    };

    using TPutDataOutMail = TVector<TShingle>;

    class TComplaintData {
    public:
        void SetSpam(bool val) {
            spam = val;
        }
        void SetDKimSpam(bool val) {
            dkimSpam = val;
        }

        bool GetSpam() const {
            return spam;
        }
        const TMaybe<bool>& GetDKimSpam() const {
            return dkimSpam;
        }

        TComplaintData() = default;
        TComplaintData(bool spam, bool dkimPass)
            : spam(spam)
        {
            if (dkimPass)
                dkimSpam = spam;
        }

    protected:
        bool spam{};
        TMaybe<bool> dkimSpam;
    };

    struct TDBSender{
        void Clear();
        void FromJson(const NJson::TJsonValue & val);

        bool empty = true;
        ui16 firsttime = 0;
        ui16 lasttime = 0;
        ui16 activedays = 0;
        ui64 ham = 0;
        ui64 spam = 0;
        ui64 rcvd_fromyandex_count = 0;
        ui64 complaint_ham = 0;
        ui64 complaint_spam = 0;
        ui64 flags = 0;
        ui64 personalham = 0;
        ui64 personalspam = 0;
        ui64 abook_count = 0;
        ui64 kooba_count = 0;
        ui64 read = 0;
        ui64 del_wo_read = 0;
        ui64 dkim_del_wo_read = 0;
        ui64 dkim_ham = 0;
        ui64 dkim_spam = 0;
        ui64 pop3_ham = 0;
        ui64 pop3_spam = 0;
        ui64 compl_dkim_spam = 0;
        ui64 ps_ham = 0;
        ui64 ps_spam = 0;
    };

    struct TDBPeriodicSender {
        friend IOutputStream &operator<<(IOutputStream &os, const TDBPeriodicSender &sender);
//cur[name].GetUIntegerRobust() + daysOnPrevWeek * prev[name].GetUIntegerRobust() / 14;

        void Clear();
        void FromJson(const NJson::TJsonValue & val);
        static TDBPeriodicSender Interpolate(const TDBPeriodicSender & cur, const TDBPeriodicSender & prev, ui16 day);

    private:
        TDBPeriodicSender operator + (const TDBPeriodicSender & s) const;
        TDBPeriodicSender operator * (int v) const;
        TDBPeriodicSender operator / (int v) const;
    public:

        ui64 ham_count = 0;
        ui64 spam_count = 0;
        ui64 compl_ham_count = 0;
        ui64 compl_spam_count = 0;
        ui64 pers_ham_count = 0;
        ui64 pers_spam_count = 0;
        ui64 kooba_count = 0;
        ui64 read = 0;
        ui64 del_wo_read = 0;
        ui64 ps_ham = 0;     ///    число хама с pay_sender за две недели - из входящей почты, только для домена
        ui64 ps_spam = 0;    ///    число спама с pay_sender за две недели - из входящей почты, только для домена
        ui64 dkim_ham_count = 0;
        ui64 dkim_spam_count = 0;
        ui64 dkim_del_wo_read = 0;
    };


    struct TSender {
        TDBSender totals;
        TDBPeriodicSender periodic;
    };

    class TGetData {
    public:

        friend IOutputStream& operator<<(IOutputStream& stream, const TGetData& data);

        //sender
        bool ResSenderEmpty() const {
            return sender.totals.empty;
        } //true - данных в базе нет
        ui32 ResSenderFirsttime() const {
            return sender.totals.firsttime;
        } //время регистрации в базе (число прошедших дней с 1 января 2011г)
        ui32 ResSenderLasttime() const {
            return sender.totals.lasttime;
        }                                       //время последнего обновления данных (число прошедших дней с 1 января 2011г)
        ui32 ResSenderFirsttimeElapsed() const; //сколько дней прошло относительно текущего момента, когда статистика появилась в базе
        ui64 ResSenderHam() const {
            return sender.totals.ham;
        } //число неспамовых писем принято с этого адреса
        ui64 ResSenderSpam() const {
            return sender.totals.spam;
        } //число спамовых писем принято с этого адреса
        ui64 ResSenderRcvdFromYandexCount() const {
            return sender.totals.rcvd_fromyandex_count;
        } //количество полученных писем от наших пользователей
        ui64 ResSenderComplaintHam() const {
            return sender.totals.complaint_ham;
        } //счетчик жалоб на хам
        ui64 ResSenderComplaintSpam() const {
            return sender.totals.complaint_spam;
        } //счетчик жалоб на спам
        ui64 ResSenderFlags() const {
            return sender.totals.flags;
        } //битовые флаги
        ui64 ResSenderPersonalHam() const {
            return sender.totals.personalham;
        } //количество срабатываний персонального фильтра хам
        ui64 ResSenderPersonalSpam() const {
            return sender.totals.personalspam;
        } //количество срабатываний персонального фильтра спам
        ui64 ResSenderActiveDays() const {
            return sender.totals.activedays;
        } //кол-во активных дней
        bool ResIsSenderDL_FBR() const {
            return IS_SNDR_DL_FBR(sender.totals.flags);
        }
        bool ResIsSenderFIXED_SOURCE() const {
            return IS_SNDR_FIXED_SOURCE(sender.totals.flags);
        }
        bool ResIsSenderSOURCE_LEVEL1() const {
            return IS_SNDR_SOURCE_LEVEL1(sender.totals.flags);
        }
        bool ResIsSenderWEBMASTER() const {
            return IS_SNDR_WEBMASTER(sender.totals.flags);
        }
        bool ResIsSenderMALIC() const {
            return IS_SNDR_MALIC(sender.totals.flags);
        }
        ui64 ResSenderAbookCount() const {
            return sender.totals.abook_count;
        }
        ui64 ResSender2WeeksHamCount() const {
            return sender.periodic.ham_count;
        }
        ui64 ResSender2WeeksSpamCount() const {
            return sender.periodic.spam_count;
        }
        ui64 ResSender2WeeksComplHamCount() const {
            return sender.periodic.compl_ham_count;
        }
        ui64 ResSender2WeeksComplSpamCount() const {
            return sender.periodic.compl_spam_count;
        }
        ui64 ResSender2WeeksPersHamCount() const {
            return sender.periodic.pers_ham_count;
        }
        ui64 ResSender2WeeksPersSpamCount() const {
            return sender.periodic.pers_spam_count;
        }
        ui64 ResSender2WeeksKoobaCount() const {
            return sender.periodic.kooba_count;
        }

        ui64 SenderPop3Ham() const {
            return sender.totals.pop3_ham;
        }
        ui64 SenderPop3Spam() const {
            return sender.totals.pop3_spam;
        }

        ui64 SenderRead() const {
            return sender.totals.read;
        }
        ui64 SenderDeletedWithoutRead() const {
            return sender.totals.del_wo_read;
        }
        ui64 SenderDeletedWithoutRead2Weeks() const {
            return sender.periodic.del_wo_read;
        }

        //domen
        bool ResDomenEmpty() const {
            return domain.totals.empty;
        } //true - данных в базе нет
        ui32 ResDomenFirsttime() const {
            return domain.totals.firsttime;
        } //время регистрации в базе (число прошедших дней с 1 января 2011г)
        ui32 ResDomenLasttime() const {
            return domain.totals.lasttime;
        }                                      //время последнего обновления данных (число прошедших дней с 1 января 2011г)
        ui32 ResDomenFirsttimeElapsed() const; //сколько дней прошло относительно текущего момента, когда статистика появилась в базе
        ui64 ResDomenHam() const {
            return domain.totals.ham;
        } //число неспамовых писем принято с этого адреса
        ui64 ResDomenSpam() const {
            return domain.totals.spam;
        } //число спамовых писем принято с этого адреса
        ui64 ResDomenRcvdFromYandexCount() const {
            return domain.totals.rcvd_fromyandex_count;
        } //количество полученных писем от наших пользователей
        ui64 ResDomenComplaintHam() const {
            return domain.totals.complaint_ham;
        } //счетчик жалоб на хам
        ui64 ResDomenComplaintSpam() const {
            return domain.totals.complaint_spam;
        } //счетчик жалоб на спам
        ui64 ResDomenFlags() const {
            return domain.totals.flags;
        } //битовые флаги
        ui64 ResDomenPersonalHam() const {
            return domain.totals.personalham;
        } //количество срабатываний персонального фильтра хам
        ui64 ResDomenPersonalSpam() const {
            return domain.totals.personalspam;
        } //количество срабатываний персонального фильтра спам
        ui64 ResDomenActiveDays() const {
            return domain.totals.activedays;
        } //кол-во активных дней
        bool ResIsDomenDL_FBR() const {
            return IS_SNDR_DL_FBR(domain.totals.flags);
        }
        bool ResIsDomenFIXED_SOURCE() const {
            return IS_SNDR_FIXED_SOURCE(domain.totals.flags);
        }
        bool ResIsDomenSOURCE_LEVEL1() const {
            return IS_SNDR_SOURCE_LEVEL1(domain.totals.flags);
        }
        bool ResIsDomenWEBMASTER() const {
            return IS_SNDR_WEBMASTER(domain.totals.flags);
        }
        bool ResIsDomenMALIC() const {
            return IS_SNDR_MALIC(domain.totals.flags);
        }
        ui64 ResDomenAbookCount() const {
            return domain.totals.abook_count;
        }
        ui64 ResDomen2WeeksHamCount() const {
            return domain.periodic.ham_count;
        }
        ui64 ResDomen2WeeksSpamCount() const {
            return domain.periodic.spam_count;
        }
        ui64 ResDomen2WeeksComplHamCount() const {
            return domain.periodic.compl_ham_count;
        }
        ui64 ResDomen2WeeksComplSpamCount() const {
            return domain.periodic.compl_spam_count;
        }
        ui64 ResDomen2WeeksPersHamCount() const {
            return domain.periodic.pers_ham_count;
        }
        ui64 ResDomen2WeeksPersSpamCount() const {
            return domain.periodic.pers_spam_count;
        }
        ui64 ResDomen2WeeksKoobaCount() const {
            return domain.periodic.kooba_count;
        }

        ui64 DomenDKimHam() const {
            return domain.totals.dkim_ham;
        }
        ui64 DomenDKimSpam() const {
            return domain.totals.dkim_spam;
        }
        ui64 DomenDKimHam2Weeks() const {
            return domain.periodic.dkim_ham_count;
        }
        ui64 DomenDKimSpam2Weeks() const {
            return domain.periodic.dkim_spam_count;
        }

        ui64 DomenPsHam() const {
            return domain.totals.ps_ham;
        }
        ui64 DomenPsSpam() const {
            return domain.totals.ps_spam;
        }
        ui64 DomenPsHam2Weeks() const {
            return domain.periodic.ps_ham;
        }
        ui64 DomenPsSpam2Weeks() const {
            return domain.periodic.ps_spam;
        }

        ui64 DomenRead() const {
            return domain.totals.read;
        }
        ui64 DomenDeletedWithoutRead() const {
            return domain.totals.del_wo_read;
        }
        ui64 DomenDkimDeletedWithoutRead() const {
            return domain.totals.dkim_del_wo_read;
        }
        ui64 DomenDeletedWithoutRead2Weeks() const {
            return domain.periodic.del_wo_read;
        }
        ui64 DomenDkimDeletedWithoutRead2Weeks() const {
            return domain.periodic.dkim_del_wo_read;
        }

        ui32 ResPSndrFirsttimeElapsed() const; //сколько дней прошло относительно текущего момента, когда статистика появилась в базе
        ui64 ResPSndrHam() const {
            return psndr.totals.ham;
        } //число неспамовых писем принято с этого адреса
        ui64 ResPSndrSpam() const {
            return psndr.totals.spam;
        } //число спамовых писем принято с этого адреса
        ui64 ResPSndrComplaintHam() const {
            return psndr.totals.complaint_ham;
        } //счетчик жалоб на хам
        ui64 ResPSndrComplaintSpam() const {
            return psndr.totals.complaint_spam;
        } //счетчик жалоб на спам
        ui64 ResPSndrFlags() const {
            return psndr.totals.flags;
        } //битовые флаги
        ui64 ResPSndrPersonalHam() const {
            return psndr.totals.personalham;
        } //количество срабатываний персонального фильтра хам
        ui64 ResPSndrPersonalSpam() const {
            return psndr.totals.personalspam;
        } //количество срабатываний персонального фильтра спам
        ui64 ResPSndrActiveDays() const {
            return psndr.totals.activedays;
        } //кол-во активных дней
        ui64 ResPSndrAbookCount() const {
            return psndr.totals.abook_count;
        }
        ui64 ResPSndr2WeeksHamCount() const {
            return psndr.periodic.ham_count;
        }
        ui64 ResPSndr2WeeksSpamCount() const {
            return psndr.periodic.spam_count;
        }
        ui64 ResPSndr2WeeksComplHamCount() const {
            return psndr.periodic.compl_ham_count;
        }
        ui64 ResPSndr2WeeksComplSpamCount() const {
            return psndr.periodic.compl_spam_count;
        }
        ui64 ResPSndr2WeeksPersHamCount() const {
            return psndr.periodic.pers_ham_count;
        }
        ui64 ResPSndr2WeeksPersSpamCount() const {
            return psndr.periodic.pers_spam_count;
        }
        ui64 ResPSndr2WeeksKoobaCount() const {
            return psndr.periodic.kooba_count;
        }

        TGetData() = default;

    public:
        static const TString senderTag, domenTag, psndrTag;

        TSender sender, domain, psndr;
    };



} //namespace NSenderReputation
